<?php

namespace App\Services;

use App\Models\Asset;
use App\Models\DowntimeLog;
use App\Models\AssetDocument;
use App\Models\ComplianceRecord;
use App\Models\InventoryItem;
use App\Models\Maintenance;
use App\Models\WorkOrder;
use App\Models\WorkOrderCost;
use App\Models\WorkOrderPart;
use Illuminate\Support\Str;

class TelegramReportService
{
    public function report(int $tenantId, string $text): string
    {
        $lower = Str::lower($text);
        $days = $this->extractDays($lower) ?? (Str::contains($lower, 'today') ? 1 : 7);

        if (Str::contains($lower, 'pm')) {
            return $this->pmDueReport($tenantId, $days);
        }
        if (Str::contains($lower, 'downtime')) {
            return $this->downtimeReport($tenantId, $days);
        }
        if (Str::contains($lower, ['stock', 'inventory'])) {
            return $this->stockReport($tenantId);
        }
        if (Str::contains($lower, ['compliance', 'license', 'licence'])) {
            if (Str::contains($lower, ['overdue', 'expired'])) {
                return $this->complianceOverdueReport($tenantId);
            }
            $dueDays = $this->extractDays($lower) ?? 30;
            return $this->complianceDueReport($tenantId, $dueDays);
        }
        if (Str::contains($lower, ['spend', 'cost'])) {
            return $this->spendReport($tenantId, $days);
        }
        if (Str::contains($lower, ['work order', 'workorder', 'wo'])) {
            return $this->workOrderReport($tenantId, $days);
        }

        return $this->summaryReport($tenantId, $days);
    }

    public function details(int $tenantId, string $topic): string
    {
        $topic = Str::lower(trim($topic));
        return match ($topic) {
            'pm' => $this->pmDueReport($tenantId, 7),
            'stock', 'inventory' => $this->stockReport($tenantId),
            'downtime' => $this->downtimeReport($tenantId, 7),
            default => 'Try: DETAILS PM, DETAILS STOCK, or DETAILS DOWNTIME.',
        };
    }

    private function summaryReport(int $tenantId, int $days): string
    {
        $since = now()->subDays($days);
        $opened = WorkOrder::where('tenant_id', $tenantId)->where('created_at', '>=', $since)->count();
        $closed = WorkOrder::where('tenant_id', $tenantId)->where('completed_at', '>=', $since)->count();
        $assets = Asset::where('tenant_id', $tenantId)->count();

        return "REPORT SUMMARY ({$days}d)\nAssets: {$assets}\nWOs opened: {$opened} | closed: {$closed}\nReply: /report pm due 7d, /report stock critical";
    }

    private function pmDueReport(int $tenantId, int $days): string
    {
        $cutoff = now()->addDays($days);
        $items = Maintenance::where('tenant_id', $tenantId)
            ->whereNull('completed_date')
            ->where('status', 'pending')
            ->whereNotNull('scheduled_date')
            ->where('scheduled_date', '<=', $cutoff)
            ->with('asset')
            ->limit(10)
            ->get();

        if ($items->isEmpty()) {
            return "PM due report ({$days}d): none.";
        }

        $lines = $items->map(function ($item) {
            $asset = $this->assetLabel($item->asset);
            $date = $item->scheduled_date?->format('Y-m-d') ?? '-';
            return "{$asset} - {$date}";
        })->all();

        return "PM due in {$days}d:\n" . implode("\n", $lines) . "\nReply: CREATE WO <event_code>";
    }

    private function downtimeReport(int $tenantId, int $days): string
    {
        $since = now()->subDays($days);
        $logs = DowntimeLog::where('tenant_id', $tenantId)
            ->where('ended_at', '>=', $since)
            ->with('asset')
            ->limit(10)
            ->get();

        if ($logs->isEmpty()) {
            return "Downtime report ({$days}d): none.";
        }

        $lines = $logs->map(function ($log) {
            $asset = $this->assetLabel($log->asset);
            $hours = round(((float) $log->duration_minutes) / 60, 1);
            return "{$asset} - {$hours}h";
        })->all();

        return "Downtime last {$days}d:\n" . implode("\n", $lines);
    }

    private function complianceDueReport(int $tenantId, int $days): string
    {
        $cutoff = now()->addDays($days);

        $items = ComplianceRecord::where('tenant_id', $tenantId)
            ->whereNotNull('expiry_date')
            ->where('expiry_date', '<=', $cutoff)
            ->where('expiry_date', '>=', now()->startOfDay())
            ->with('asset')
            ->limit(10)
            ->get();

        if ($items->isEmpty()) {
            $fallback = AssetDocument::where('tenant_id', $tenantId)
                ->whereNotNull('expiry_date')
                ->where('expiry_date', '<=', $cutoff)
                ->where('expiry_date', '>=', now()->startOfDay())
                ->with('asset')
                ->limit(10)
                ->get()
                ->map(function ($doc) {
                    $doc->compliance_type = $doc->type ?? 'Compliance';
                    return $doc;
                });
            if ($fallback->isEmpty()) {
                return "Compliance due in {$days}d: none.";
            }
            $items = $fallback;
        }

        $lines = $items->map(function ($item) {
            $asset = $this->assetLabel($item->asset);
            $date = $item->expiry_date?->format('Y-m-d') ?? '-';
            $type = $item->compliance_type ?? 'Compliance';
            return "{$asset} - {$type} ({$date})";
        })->all();

        return "Compliance due in {$days}d:\n" . implode("\n", $lines);
    }

    private function complianceOverdueReport(int $tenantId): string
    {
        $items = ComplianceRecord::where('tenant_id', $tenantId)
            ->whereNotNull('expiry_date')
            ->where('expiry_date', '<', now()->startOfDay())
            ->with('asset')
            ->limit(10)
            ->get();

        if ($items->isEmpty()) {
            $fallback = AssetDocument::where('tenant_id', $tenantId)
                ->whereNotNull('expiry_date')
                ->where('expiry_date', '<', now()->startOfDay())
                ->with('asset')
                ->limit(10)
                ->get()
                ->map(function ($doc) {
                    $doc->compliance_type = $doc->type ?? 'Compliance';
                    return $doc;
                });
            if ($fallback->isEmpty()) {
                return 'Compliance overdue: none.';
            }
            $items = $fallback;
        }

        $lines = $items->map(function ($item) {
            $asset = $this->assetLabel($item->asset);
            $date = $item->expiry_date?->format('Y-m-d') ?? '-';
            $type = $item->compliance_type ?? 'Compliance';
            return "{$asset} - {$type} (expired {$date})";
        })->all();

        return "Compliance overdue:\n" . implode("\n", $lines);
    }

    private function stockReport(int $tenantId): string
    {
        $items = InventoryItem::where('tenant_id', $tenantId)
            ->with('part')
            ->get()
            ->filter(function ($item) {
                $reorder = $item->reorder_point ?? $item->min_quantity;
                if ($reorder === null) {
                    return false;
                }
                return (float) $item->quantity <= (float) $reorder;
            })
            ->take(10);

        if ($items->isEmpty()) {
            return 'Stock report: no low stock items.';
        }

        $lines = $items->map(function ($item) {
            $sku = $item->part?->sku ?? '-';
            $qty = $item->quantity ?? 0;
            return "{$sku} ({$qty})";
        })->all();

        return "Low stock:\n" . implode("\n", $lines) . "\nReply: CREATE PR <event_code>";
    }

    private function spendReport(int $tenantId, int $days): string
    {
        $since = now()->subDays($days);
        $parts = WorkOrderPart::where('tenant_id', $tenantId)
            ->where('created_at', '>=', $since)
            ->sum('total_cost');

        $labor = WorkOrderCost::where('tenant_id', $tenantId)
            ->where('created_at', '>=', $since)
            ->sum('total_cost');

        $total = (float) $parts + (float) $labor;

        return "Spend last {$days}d:\nParts $" . number_format((float) $parts, 2) .
            "\nLabor $" . number_format((float) $labor, 2) .
            "\nTotal $" . number_format($total, 2);
    }

    private function workOrderReport(int $tenantId, int $days): string
    {
        $since = now()->subDays($days);
        $orders = WorkOrder::where('tenant_id', $tenantId)
            ->where('created_at', '>=', $since)
            ->with('asset')
            ->limit(10)
            ->get();

        if ($orders->isEmpty()) {
            return "Work orders report ({$days}d): none.";
        }

        $lines = $orders->map(function ($order) {
            $ref = $order->reference_code ?? ('WO-' . $order->id);
            $asset = $this->assetLabel($order->asset);
            return "{$ref} - {$asset} ({$order->status})";
        })->all();

        return "Work orders last {$days}d:\n" . implode("\n", $lines);
    }

    private function extractDays(string $text): ?int
    {
        if (preg_match('/\b(\d{1,3})\s*d\b/', $text, $matches)) {
            return (int) $matches[1];
        }
        if (preg_match('/\b(\d{1,3})\s*days?\b/', $text, $matches)) {
            return (int) $matches[1];
        }

        return null;
    }

    private function assetLabel(?Asset $asset, ?string $fallbackTag = null): string
    {
        $tag = $asset?->asset_tag ?? $fallbackTag ?? '-';
        $name = $asset?->name ?? null;
        if ($name) {
            return "{$tag} - {$name}";
        }

        return (string) $tag;
    }
}
