<?php

namespace App\Services;

use App\Domain\Telegram\TelegramNotifier;
use App\Models\User;
use App\Models\WorkOrder;

class TelegramTechnicianDigestService
{
    public function __construct(private TelegramNotifier $notifier)
    {
    }

    public function sendTechnicianDailyJobs(): void
    {
        $users = User::whereHas('roles', function ($query) {
            $query->where('name', 'like', '%technician%')
                ->orWhere('name', 'like', '%artisan%');
        })->get();

        foreach ($users as $user) {
            $orders = WorkOrder::where('tenant_id', $user->tenant_id)
                ->where('assigned_to', $user->id)
                ->whereIn('status', ['open', 'scheduled', 'in_progress'])
                ->with('asset')
                ->orderBy('due_at')
                ->limit(5)
                ->get();

            if ($orders->isEmpty()) {
                continue;
            }

            $lines = $orders->map(function ($order) {
                $ref = $order->reference_code ?? ('WO-' . $order->id);
                $asset = $this->assetLabel($order->asset);
                $priority = $order->priority ?? 'medium';
                return "{$ref} {$asset} ({$priority})";
            })->all();

            $message = "Your jobs today:\n" . implode("\n", $lines) .
                "\nReply: START <WO> or COMPLETE <WO>.";

            $this->notifier->notifyUser($user->id, $message);
        }
    }

    private function assetLabel(?\App\Models\Asset $asset, ?string $fallbackTag = null): string
    {
        $tag = $asset?->asset_tag ?? $fallbackTag ?? 'Asset';
        $name = $asset?->name ?? null;
        if ($name) {
            return "{$tag} - {$name}";
        }

        return (string) $tag;
    }
}
