<?php

namespace App\Services;

use App\Models\Telemetry;
use Illuminate\Support\Facades\Log;

class TelemetryIngestionService
{
    /**
     * Process a batch of telemetry readings for a vehicle.
     *
     * @param int $vehicleId
     * @param array $readings
     * @return array
     */
    public function processTelemetryData(int $vehicleId, array $readings, ?int $tenantId = null): array
    {
        $ingested = [];

        foreach ($readings as $reading) {
            $payload = array_merge($reading, [
                'vehicle_id' => $vehicleId,
            ]);

            if ($tenantId) {
                $payload['tenant_id'] = $tenantId;
            }

            $ingested[] = $this->ingest($payload);
        }

        return $ingested;
    }

    /**
     * Ingest telemetry data from sensors.
     *
     * @param array $data
     * @return Telemetry
     */
    public function ingest(array $data): Telemetry
    {
        // Validate and normalize the incoming telemetry data
        $normalizedData = $this->normalizeData($data);

        // Create a new Telemetry record in the database
        $telemetry = Telemetry::create($normalizedData);

        // Log the ingestion for monitoring purposes
        Log::info('Telemetry data ingested', ['telemetry' => $telemetry]);

        return $telemetry;
    }

    /**
     * Normalize the telemetry data.
     *
     * @param array $data
     * @return array
     */
    protected function normalizeData(array $data): array
    {
        return [
            'tenant_id' => $data['tenant_id'] ?? null,
            'vehicle_id' => $data['vehicle_id'],
            'asset_id' => $data['asset_id'] ?? null,
            'timestamp' => $data['timestamp'] ?? now(),
            'usage_hours' => $data['usage_hours'] ?? null,
            'data' => $data,
        ];
    }
}
