<?php

namespace App\Services;

use App\Domain\Telegram\TelegramNotifier;
use App\Models\AgentNotification;
use App\Models\DowntimeLog;
use App\Models\DocumentIntakeRow;
use App\Models\NotificationOutbox;
use App\Models\TelegramGroup;
use App\Models\Tenant;
use App\Models\WeeklyReport;
use App\Models\WeeklyReportSection;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Support\Carbon;
use Illuminate\Support\Collection;
use Illuminate\Support\Str;
use RuntimeException;

class WeeklyAiAgentsReportService
{
    private const AGENT_KEYS = [
        'maintenance_compliance',
        'failure_prediction',
        'kpi_analyst',
        'cost_control',
        'ops_notifier',
        'finance_oversight',
        'hr_artisan_performance',
    ];

    public function generateForTenant(Tenant $tenant, ?string $week = null, bool $force = false, ?string $windowMode = null): WeeklyReport
    {
        $timezone = $this->resolveTimezone($tenant);
        [$start, $end, $reportWeek, $mode] = $this->resolveWindow($timezone, $week, $tenant, $windowMode);

        $existing = WeeklyReport::where('tenant_id', $tenant->id)
            ->where('report_week', $reportWeek)
            ->first();

        if ($existing && $existing->status === 'ready' && !$force) {
            return $existing;
        }

        $report = WeeklyReport::updateOrCreate(
            ['tenant_id' => $tenant->id, 'report_week' => $reportWeek],
            [
                'period_start' => $start,
                'period_end' => $end,
                'status' => 'generating',
            ]
        );

        try {
            $agentNotifications = $this->loadAgentNotifications($tenant->id, $start, $end);
            $sections = $this->buildAgentSections($agentNotifications);

            $maintenance = $this->buildMaintenanceSection($agentNotifications, $tenant->id, $start, $end);
            $finance = $this->buildFinanceSection($agentNotifications, $tenant->id);
            $hr = $this->buildHrSection($agentNotifications);
            $ops = $this->buildOpsSection($agentNotifications);
            $notifications = $this->buildNotificationSummary($tenant->id, $start, $end);
            $executive = $this->buildExecutiveSummary($agentNotifications, $finance, $maintenance, $hr, $ops);

            $summary = [
                'period' => [
                    'start' => $start->toDateTimeString(),
                    'end' => $end->toDateTimeString(),
                    'timezone' => $timezone,
                    'window_mode' => $mode,
                ],
                'executive_summary' => $executive,
                'sections' => $sections,
                'maintenance' => $maintenance,
                'finance' => $finance,
                'hr' => $hr,
                'ops_notifier' => $ops,
                'notifications' => $notifications,
            ];

            $path = $this->renderPdf($tenant, $reportWeek, $summary, $start, $end, $timezone);

            $report->fill([
                'generated_at' => now(),
                'pdf_path' => $path,
                'summary_json' => $summary,
                'status' => 'ready',
            ]);
            $report->save();

            $this->storeSectionRows($report, $sections);
            $this->sendReportToTelegram($tenant, $path, $reportWeek, $start, $end, $timezone);
        } catch (\Throwable $e) {
            $report->update(['status' => 'failed']);
            throw $e;
        }

        return $report;
    }

    private function resolveWindow(string $timezone, ?string $week, Tenant $tenant, ?string $windowMode): array
    {
        $mode = $windowMode ?: config('weekly_reports.tenant_window_overrides.' . $tenant->id)
            ?: config('weekly_reports.default_window', 'last7');

        $now = Carbon::now($timezone);
        if ($week) {
            [$year, $weekNum] = $this->parseWeek($week);
            $start = Carbon::now($timezone)->setISODate($year, $weekNum)->startOfWeek(Carbon::MONDAY);
            $end = $start->copy()->endOfWeek(Carbon::SUNDAY);
            $reportWeek = sprintf('%s-W%02d', $start->format('o'), (int) $start->format('W'));
            return [$start, $end, $reportWeek, 'week'];
        }

        if ($mode === 'last_full_week') {
            $start = $now->copy()->startOfWeek(Carbon::MONDAY)->subWeek();
            $end = $start->copy()->endOfWeek(Carbon::SUNDAY);
            $reportWeek = sprintf('%s-W%02d', $start->format('o'), (int) $start->format('W'));
            return [$start, $end, $reportWeek, $mode];
        }

        $end = $now->copy();
        $start = $end->copy()->subDays(7)->startOfDay();
        $reportWeek = sprintf('%s-W%02d', $end->format('o'), (int) $end->format('W'));

        return [$start, $end, $reportWeek, 'last7'];
    }

    private function parseWeek(string $value): array
    {
        $value = strtoupper(trim($value));
        if (!preg_match('/^(\d{4})-W?(\d{1,2})$/', $value, $matches)) {
            throw new RuntimeException('Invalid week format. Use YYYY-WW.');
        }

        return [(int) $matches[1], (int) $matches[2]];
    }

    private function resolveTimezone(Tenant $tenant): string
    {
        return config('weekly_reports.tenant_timezones.' . $tenant->id)
            ?: config('weekly_reports.default_timezone', 'Africa/Harare');
    }

    private function loadAgentNotifications(int $tenantId, Carbon $start, Carbon $end): Collection
    {
        return AgentNotification::where('tenant_id', $tenantId)
            ->whereIn('agent_key', self::AGENT_KEYS)
            ->whereBetween('created_at', [$start, $end])
            ->orderBy('created_at')
            ->get()
            ->groupBy('agent_key');
    }

    private function buildAgentSections(Collection $grouped): array
    {
        $sections = [];
        foreach (self::AGENT_KEYS as $key) {
            $items = $grouped->get($key, collect());
            $latest = $items->sortByDesc('created_at')->first();
            $purpose = config('weekly_reports.agent_purposes.' . $key);
            $section = [
                'key' => $key,
                'title' => $latest?->title ?? Str::headline(str_replace('_', ' ', $key)),
                'purpose' => $purpose ?: 'Automated weekly analysis for this domain.',
                'severity' => $latest?->severity ?? 'info',
                'metrics' => $this->agentMetrics($key, $latest),
                'findings' => $this->agentFindings($key, $latest),
                'recommended_actions' => $this->agentActions($key, $latest),
                'summary' => $latest?->summary,
                'no_data' => $latest === null,
                'data_confidence' => $latest ? [] : ['No run data in this window. Ensure the agent is scheduled.'],
            ];

            $sections[$key] = $section;
        }

        return $sections;
    }

    private function agentMetrics(string $key, ?AgentNotification $notification): array
    {
        if (!$notification) {
            return [];
        }

        $payload = $notification->payload ?? [];
        if ($key === 'maintenance_compliance') {
            return ['overdue_count' => (int) ($payload['count'] ?? 0)];
        }
        if ($key === 'kpi_analyst') {
            return [
                'availability' => $payload['availability'] ?? null,
                'utilization' => $payload['utilization'] ?? null,
                'mtbf' => $payload['mtbf'] ?? null,
                'fuel_efficiency' => $payload['fuel_efficiency'] ?? null,
                'downtime_hours' => $payload['downtime_hours'] ?? null,
            ];
        }

        return $payload['metrics'] ?? [];
    }

    private function agentFindings(string $key, ?AgentNotification $notification): array
    {
        if (!$notification) {
            return [];
        }

        $payload = $notification->payload ?? [];
        return match ($key) {
            'maintenance_compliance' => $payload['overdue'] ?? [],
            'failure_prediction' => $payload['signals'] ?? [],
            'cost_control' => $payload['findings'] ?? [],
            'ops_notifier' => $payload['items'] ?? $payload['findings'] ?? [],
            default => $payload['findings'] ?? [],
        };
    }

    private function agentActions(string $key, ?AgentNotification $notification): array
    {
        if (!$notification) {
            return [];
        }

        $payload = $notification->payload ?? [];
        if ($key === 'failure_prediction') {
            return $payload['recommendations'] ?? [];
        }

        if ($key === 'finance_oversight') {
            $action = $payload['suggested_next_action'] ?? null;
            return $action ? [$action] : [];
        }

        if ($key === 'hr_artisan_performance') {
            $action = $payload['suggested_next_action'] ?? null;
            return $action ? [$action] : [];
        }

        return [];
    }

    private function buildMaintenanceSection(Collection $grouped, int $tenantId, Carbon $start, Carbon $end): array
    {
        $compliance = $grouped->get('maintenance_compliance', collect())->sortByDesc('created_at')->first();
        $overdue = $compliance?->payload['overdue'] ?? [];
        $overdue = array_slice($overdue, 0, 15);

        $downtimeMinutes = (float) DowntimeLog::where('tenant_id', $tenantId)
            ->whereBetween('ended_at', [$start, $end])
            ->sum('duration_minutes');
        $downtimeHours = round($downtimeMinutes / 60, 1);

        $downtimeReasons = DowntimeLog::select('downtime_reason_codes.label')
            ->join('downtime_reason_codes', 'downtime_reason_codes.id', '=', 'downtime_logs.reason_code_id')
            ->where('downtime_logs.tenant_id', $tenantId)
            ->whereBetween('downtime_logs.ended_at', [$start, $end])
            ->groupBy('downtime_reason_codes.label')
            ->orderByRaw('count(*) desc')
            ->limit(6)
            ->pluck('downtime_reason_codes.label')
            ->all();

        $failure = $grouped->get('failure_prediction', collect())->sortByDesc('created_at')->first();
        $failureRisks = $failure?->payload['signals'] ?? [];

        $confidence = [];
        if (!$compliance) {
            $confidence[] = 'Maintenance compliance agent did not run in this window.';
        }
        if ($downtimeMinutes === 0.0) {
            $confidence[] = 'No downtime logs found in this window.';
        }
        if (!$failure) {
            $confidence[] = 'Failure prediction agent did not run in this window.';
        }

        return [
            'overdue' => $overdue,
            'downtime_hours' => $downtimeHours,
            'downtime_reasons' => $downtimeReasons,
            'failure_risks' => array_slice($failureRisks, 0, 10),
            'data_confidence' => $confidence,
        ];
    }

    private function buildFinanceSection(Collection $grouped, int $tenantId): array
    {
        $finance = $grouped->get('finance_oversight', collect())->sortByDesc('created_at')->first();
        $payload = $finance?->payload ?? [];
        $metrics = $payload['metrics'] ?? [];
        $findings = $payload['findings'] ?? [];
        $invoiceRows = $this->invoiceRows($tenantId);
        $vendorExposure = $this->supplierExposure($invoiceRows);
        $agingBuckets = $vendorExposure['aging_buckets'] ?? [];
        $topVendors = $vendorExposure['top_vendors'] ?? [];
        $confidence = [];
        if (!$finance) {
            $confidence[] = 'Finance oversight agent did not run in this window.';
        }
        if (!$invoiceRows->count()) {
            $confidence[] = 'No invoice intake rows found for aging and exposure calculations.';
        }

        return [
            'metrics' => $metrics,
            'findings' => $findings,
            'aging_buckets' => $agingBuckets,
            'top_vendors' => $topVendors,
            'data_confidence' => $confidence,
        ];
    }

    private function buildHrSection(Collection $grouped): array
    {
        $hr = $grouped->get('hr_artisan_performance', collect())->sortByDesc('created_at')->first();
        $payload = $hr?->payload ?? [];
        $metrics = $payload['metrics'] ?? [];
        $findings = $payload['findings'] ?? [];
        $confidence = [];
        if (!$hr) {
            $confidence[] = 'HR artisan performance agent did not run in this window.';
        }

        return [
            'metrics' => $metrics,
            'findings' => $findings,
            'data_confidence' => $confidence,
        ];
    }

    private function buildOpsSection(Collection $grouped): array
    {
        $ops = $grouped->get('ops_notifier', collect())->sortByDesc('created_at')->first();
        $payload = $ops?->payload ?? [];
        $items = $payload['items'] ?? [];
        $findings = $payload['findings'] ?? [];
        $confidence = [];
        if (!$ops) {
            $confidence[] = 'Ops notifier did not run in this window.';
        }

        return [
            'items' => array_slice($items, 0, 12),
            'findings' => array_slice($findings, 0, 12),
            'data_confidence' => $confidence,
        ];
    }

    private function buildNotificationSummary(int $tenantId, Carbon $start, Carbon $end): array
    {
        $outbox = NotificationOutbox::where('tenant_id', $tenantId)
            ->whereBetween('created_at', [$start, $end])
            ->get();

        $byChannel = $outbox->groupBy('channel')->map(function ($items, $channel) {
            return [
                'channel' => $channel,
                'queued' => $items->where('status', 'queued')->count(),
                'sent' => $items->where('status', 'sent')->count(),
                'failed' => $items->where('status', 'failed')->count(),
            ];
        })->values()->all();

        $severityCounts = $outbox->groupBy('severity')->map(fn ($items, $severity) => [
            'severity' => $severity,
            'count' => $items->count(),
        ])->values()->all();

        $alarmEscalations = $outbox->where('severity', 'alarm')
            ->sortByDesc('created_at')
            ->take(12)
            ->map(function ($item) {
                return [
                    'title' => $item->title ?? 'Alert',
                    'message' => Str::limit((string) $item->message, 180),
                    'status' => $item->status,
                    'channel' => $item->channel,
                ];
            })
            ->values()
            ->all();

        return [
            'by_channel' => $byChannel,
            'by_severity' => $severityCounts,
            'alarm_escalations' => $alarmEscalations,
        ];
    }

    private function buildExecutiveSummary(Collection $grouped, array $finance, array $maintenance, array $hr, array $ops): array
    {
        $kpi = $grouped->get('kpi_analyst', collect())->sortByDesc('created_at')->first();
        $kpiMetrics = $kpi?->payload ?? [];

        $headline = [
            'availability' => $kpiMetrics['availability'] ?? null,
            'utilization' => $kpiMetrics['utilization'] ?? null,
            'mtbf' => $kpiMetrics['mtbf'] ?? null,
            'fuel_efficiency' => $kpiMetrics['fuel_efficiency'] ?? null,
        ];

        $topRisks = $grouped->flatten(1)
            ->filter(fn ($item) => in_array($item->severity, ['alarm', 'warn'], true))
            ->sortByDesc('created_at')
            ->take(6)
            ->map(fn ($item) => $item->title . ' - ' . Str::limit((string) $item->summary, 120))
            ->values()
            ->all();

        $wins = [];
        if (($headline['availability'] ?? 0) >= 90) {
            $wins[] = 'Availability above target threshold.';
        }
        if (($headline['utilization'] ?? 0) >= 70) {
            $wins[] = 'Utilization holding above 70%.';
        }
        if (!$wins) {
            $wins[] = 'No major wins captured; focus on closing overdue PMs and approvals.';
        }

        $financeMetrics = $finance['metrics'] ?? [];

        $summaryConfidence = array_filter(array_merge(
            $maintenance['data_confidence'] ?? [],
            $finance['data_confidence'] ?? [],
            $hr['data_confidence'] ?? [],
            $ops['data_confidence'] ?? []
        ));

        return [
            'headline_kpis' => $headline,
            'top_risks' => $topRisks,
            'top_wins' => $wins,
            'total_spend' => $financeMetrics['month_to_date_spend'] ?? null,
            'spend_vs_budget' => $financeMetrics['spend_vs_budget'] ?? null,
            'unpaid_total' => $financeMetrics['unpaid_total'] ?? null,
            'pr_aging_days' => $financeMetrics['pr_aging_days'] ?? null,
            'po_aging_days' => $financeMetrics['po_aging_days'] ?? null,
            'invoice_aging_days' => $financeMetrics['invoice_aging_days'] ?? null,
            'data_confidence' => $summaryConfidence,
        ];
    }

    private function renderPdf(Tenant $tenant, string $reportWeek, array $summary, Carbon $start, Carbon $end, string $timezone): string
    {
        $directory = storage_path('app/reports/weekly/' . $tenant->id);
        if (!is_dir($directory)) {
            mkdir($directory, 0755, true);
        }

        $logo = $this->logoData();
        $filename = 'weekly_ai_agents_report_' . $reportWeek . '_' . now()->format('Ymd_His') . '.pdf';
        $path = $directory . DIRECTORY_SEPARATOR . $filename;

        $pdf = Pdf::loadView('reports.weekly_ai_agents_report', [
            'tenant' => $tenant,
            'reportWeek' => $reportWeek,
            'periodStart' => $start,
            'periodEnd' => $end,
            'timezone' => $timezone,
            'summary' => $summary,
            'logo' => $logo,
        ])->setPaper('a4', 'portrait');

        file_put_contents($path, $pdf->output());

        return $this->relativeReportPath($path);
    }

    private function invoiceRows(?int $tenantId): Collection
    {
        if (!$tenantId) {
            return collect();
        }

        return DocumentIntakeRow::where('tenant_id', $tenantId)
            ->whereHas('batch', fn ($query) => $query->where('doc_type', 'invoice'))
            ->get();
    }

    private function supplierExposure(Collection $rows): array
    {
        $totals = [];
        $buckets = [
            '0-30' => 0.0,
            '31-60' => 0.0,
            '61-90' => 0.0,
            '90+' => 0.0,
        ];

        foreach ($rows as $row) {
            $data = $row->data ?? [];
            if (!empty($data['paid_at'])) {
                continue;
            }

            $amount = (float) ($data['amount'] ?? 0);
            if ($amount <= 0) {
                continue;
            }

            $vendor = (string) ($data['vendor'] ?? $data['supplier'] ?? $data['supplier_name'] ?? 'Unknown');
            $totals[$vendor] = ($totals[$vendor] ?? 0) + $amount;

            $issuedAt = $data['invoice_date'] ?? $data['issued_at'] ?? null;
            $date = null;
            if ($issuedAt) {
                try {
                    $date = Carbon::parse($issuedAt);
                } catch (\Throwable $e) {
                    $date = null;
                }
            }
            $age = $date ? $date->diffInDays(now()) : 0;

            if ($age <= 30) {
                $buckets['0-30'] += $amount;
            } elseif ($age <= 60) {
                $buckets['31-60'] += $amount;
            } elseif ($age <= 90) {
                $buckets['61-90'] += $amount;
            } else {
                $buckets['90+'] += $amount;
            }
        }

        arsort($totals);
        $topVendors = array_slice($totals, 0, 10, true);

        return [
            'top_vendors' => $topVendors,
            'aging_buckets' => $buckets,
        ];
    }

    private function relativeReportPath(string $absolutePath): string
    {
        $storageRoot = storage_path('app');
        if (Str::startsWith($absolutePath, $storageRoot)) {
            return ltrim(str_replace($storageRoot, '', $absolutePath), DIRECTORY_SEPARATOR);
        }

        return $absolutePath;
    }

    private function storeSectionRows(WeeklyReport $report, array $sections): void
    {
        WeeklyReportSection::where('weekly_report_id', $report->id)->delete();

        foreach ($sections as $section) {
            WeeklyReportSection::create([
                'weekly_report_id' => $report->id,
                'section_code' => $section['key'],
                'title' => $section['title'],
                'content_json' => $section,
            ]);
        }
    }

    private function sendReportToTelegram(Tenant $tenant, string $relativePath, string $reportWeek, Carbon $start, Carbon $end, string $timezone): void
    {
        if (!config('weekly_reports.telegram_send_enabled', true)) {
            return;
        }
        if (!config('services.telegram.bot_token')) {
            return;
        }

        $absolutePath = storage_path('app/' . ltrim($relativePath, '/'));
        if (!is_file($absolutePath)) {
            return;
        }

        $groups = config('weekly_reports.telegram_groups', ['management']);
        if (!$groups) {
            return;
        }

        $caption = sprintf(
            'Weekly AI Agents Report (%s)\nPeriod: %s to %s (%s)',
            $reportWeek,
            $start->toDateString(),
            $end->toDateString(),
            $timezone
        );

        $notifier = app(TelegramNotifier::class);
        $groupChats = TelegramGroup::where('tenant_id', $tenant->id)
            ->whereIn('name', $groups)
            ->where('enabled', true)
            ->get();

        foreach ($groupChats as $group) {
            $notifier->notifyChatDocument($group->chat_id, $absolutePath, basename($absolutePath), $caption);
        }
    }

    private function logoData(): ?string
    {
        $path = public_path('ui-assets/images/fourways-logo.png');
        if (!is_file($path)) {
            return null;
        }

        return 'data:image/png;base64,' . base64_encode(file_get_contents($path));
    }
}
