<?php

namespace App\Support;

use ZipArchive;
use SimpleXMLElement;

class XlsxReader
{
    public static function readSheetRows(string $path, ?string $sheetName = null, ?int $headerRow = null): array
    {
        $zip = new ZipArchive();
        if ($zip->open($path) !== true) {
            return [];
        }

        $workbookXml = $zip->getFromName('xl/workbook.xml');
        $relsXml = $zip->getFromName('xl/_rels/workbook.xml.rels');
        if (!$workbookXml || !$relsXml) {
            $zip->close();
            return [];
        }

        $sheetPath = self::resolveSheetPath($workbookXml, $relsXml, $sheetName);
        if (!$sheetPath) {
            $zip->close();
            return [];
        }

        $sharedStrings = self::loadSharedStrings($zip);
        $sheetXml = $zip->getFromName($sheetPath);
        $zip->close();

        if (!$sheetXml) {
            return [];
        }

        return self::parseSheetRows($sheetXml, $sharedStrings, $headerRow);
    }

    private static function resolveSheetPath(string $workbookXml, string $relsXml, ?string $sheetName): ?string
    {
        $workbook = new SimpleXMLElement($workbookXml);
        $workbook->registerXPathNamespace('r', 'http://schemas.openxmlformats.org/officeDocument/2006/relationships');

        $sheetId = null;
        $firstId = null;

        foreach ($workbook->sheets->sheet as $sheet) {
            $id = (string) $sheet->attributes('r', true)['id'];
            if (!$firstId) {
                $firstId = $id;
            }

            if ($sheetName && strcasecmp((string) $sheet['name'], $sheetName) === 0) {
                $sheetId = $id;
                break;
            }
        }

        if (!$sheetId) {
            $sheetId = $firstId;
        }

        if (!$sheetId) {
            return null;
        }

        $rels = new SimpleXMLElement($relsXml);
        foreach ($rels->Relationship as $rel) {
            if ((string) $rel['Id'] === $sheetId) {
                $target = ltrim((string) $rel['Target'], '/');
                if (str_starts_with($target, 'xl/')) {
                    return $target;
                }
                return 'xl/' . $target;
            }
        }

        return null;
    }

    private static function loadSharedStrings(ZipArchive $zip): array
    {
        $sharedXml = $zip->getFromName('xl/sharedStrings.xml');
        if (!$sharedXml) {
            return [];
        }

        $shared = new SimpleXMLElement($sharedXml);
        $strings = [];
        foreach ($shared->si as $si) {
            $strings[] = self::extractRichText($si);
        }

        return $strings;
    }

    private static function parseSheetRows(string $sheetXml, array $sharedStrings, ?int $headerRow = null): array
    {
        $sheet = new SimpleXMLElement($sheetXml);
        $rows = [];
        $namespaces = $sheet->getNamespaces(true);
        $mainNs = $namespaces[''] ?? null;

        if ($mainNs) {
            $sheet->registerXPathNamespace('main', $mainNs);
        }

        $rowNodes = $mainNs
            ? $sheet->xpath('//main:sheetData/main:row')
            : $sheet->sheetData->row;

        if (!$rowNodes && $mainNs) {
            $rowNodes = $sheet->children($mainNs)->sheetData->row;
        }

        if (!$rowNodes) {
            return [];
        }

        foreach ($rowNodes as $row) {
            $cells = [];
            $rowIndex = (int) ($row['r'] ?? 0);
            $cellNodes = $mainNs ? $row->children($mainNs)->c : $row->c;
            if (!$cellNodes && $mainNs) {
                $cellNodes = $row->xpath('main:c');
            }
            foreach ($cellNodes as $cell) {
                $attributes = $cell->attributes();
                $ref = $attributes ? (string) $attributes->r : '';
                $colIndex = self::columnIndex($ref);
                $cells[$colIndex] = self::cellValue($cell, $sharedStrings, $mainNs);
            }

            if ($cells) {
                ksort($cells);
                $rows[$rowIndex ?: count($rows)] = $cells;
            }
        }

        if (!$rows) {
            return [];
        }

        $headerRowIndex = $headerRow ?: self::detectHeaderRow($rows);
        $headerCells = $rows[$headerRowIndex] ?? array_values($rows)[0];
        $headers = $headerCells;

        $result = [];
        foreach ($rows as $rowIndex => $row) {
            if ($rowIndex <= $headerRowIndex) {
                continue;
            }
            $record = [];
            foreach ($headers as $index => $header) {
                $header = (string) $header;
                if ($header === '') {
                    continue;
                }
                $record[$header] = $row[$index] ?? null;
            }
            if (array_filter($record, fn ($value) => $value !== null && $value !== '')) {
                $result[] = $record;
            }
        }

        return $result;
    }

    private static function detectHeaderRow(array $rows): int
    {
        $bestRow = 0;
        $bestScore = -1;
        $bestCells = -1;
        $maxScan = 30;
        $scanned = 0;

        foreach ($rows as $rowIndex => $row) {
            $scanned++;
            if ($scanned > $maxScan) {
                break;
            }

            $values = array_values($row);
            $cellCount = count(array_filter($values, fn ($value) => $value !== null && $value !== ''));
            if ($cellCount === 0) {
                continue;
            }

            $stringCount = 0;
            foreach ($values as $value) {
                if (is_string($value) && preg_match('/[A-Za-z]/', $value)) {
                    $stringCount++;
                }
            }

            if ($stringCount > $bestScore || ($stringCount === $bestScore && $cellCount > $bestCells)) {
                $bestScore = $stringCount;
                $bestCells = $cellCount;
                $bestRow = (int) $rowIndex;
            }
        }

        return $bestRow ?: (int) array_key_first($rows);
    }

    private static function padRow(array $row, int $maxColumns): array
    {
        if (count($row) >= $maxColumns) {
            return $row;
        }

        return array_pad($row, $maxColumns, null);
    }

    private static function cellValue(SimpleXMLElement $cell, array $sharedStrings, ?string $mainNs = null): ?string
    {
        $attributes = $cell->attributes();
        $type = $attributes ? (string) $attributes->t : '';

        if ($type === 'inlineStr') {
            $inline = $mainNs ? $cell->children($mainNs)->is : $cell->is;
            return trim(self::extractRichText($inline, $mainNs));
        }

        if ($type === 's') {
            $valueNode = $mainNs ? $cell->children($mainNs)->v : $cell->v;
            $index = (int) $valueNode;
            return $sharedStrings[$index] ?? null;
        }

        $valueNode = $mainNs ? $cell->children($mainNs)->v : $cell->v;
        if ($valueNode !== null) {
            return trim((string) $valueNode);
        }

        return null;
    }

    private static function extractRichText(?SimpleXMLElement $node, ?string $mainNs = null): string
    {
        $text = '';

        if (!$node) {
            return $text;
        }

        $tNode = $mainNs ? $node->children($mainNs)->t : $node->t;
        if ($tNode) {
            $text .= (string) $tNode;
        }

        $runs = $mainNs ? $node->children($mainNs)->r : $node->r;
        if ($runs) {
            foreach ($runs as $run) {
                $runText = $mainNs ? $run->children($mainNs)->t : $run->t;
                $text .= (string) $runText;
            }
        }

        return trim($text);
    }

    private static function columnIndex(string $cellRef): int
    {
        if (!preg_match('/^[A-Z]+/', $cellRef, $matches)) {
            return 0;
        }

        $letters = $matches[0];
        $index = 0;
        foreach (str_split($letters) as $letter) {
            $index = ($index * 26) + (ord($letter) - 64);
        }

        return $index - 1;
    }
}
