<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration {
    public function up()
    {
        Schema::create('houses', function (Blueprint $table) {
            $table->id();
            $table->foreignId('tenant_id')->constrained()->cascadeOnDelete();
            $table->string('name');
            $table->unsignedInteger('capacity')->default(0);
            $table->unsignedInteger('age_days')->default(0);
            $table->string('timezone')->default('UTC');
            $table->text('notes')->nullable();
            $table->string('telegram_chat_id')->nullable();
            $table->timestamps();
        });

        Schema::create('sensors', function (Blueprint $table) {
            $table->id();
            $table->foreignId('device_id')->constrained()->cascadeOnDelete();
            $table->string('type');
            $table->string('unit')->nullable();
            $table->string('name')->nullable();
            $table->timestamps();
        });

        Schema::create('sensor_readings', function (Blueprint $table) {
            $table->id();
            $table->foreignId('sensor_id')->constrained()->cascadeOnDelete();
            $table->timestamp('ts');
            $table->decimal('value', 12, 4);
            $table->json('meta_json')->nullable();
            $table->timestamps();
            $table->unique(['sensor_id', 'ts']);
            $table->index(['sensor_id', 'ts']);
        });

        Schema::create('weight_samples', function (Blueprint $table) {
            $table->id();
            $table->foreignId('house_id')->constrained()->cascadeOnDelete();
            $table->foreignId('source_device_id')->nullable()->constrained('devices')->nullOnDelete();
            $table->timestamp('ts');
            $table->decimal('avg_weight_g', 10, 2);
            $table->unsignedInteger('sample_count')->default(0);
            $table->json('meta_json')->nullable();
            $table->timestamps();
            $table->index(['house_id', 'ts']);
        });

        Schema::create('feed_logs', function (Blueprint $table) {
            $table->id();
            $table->foreignId('house_id')->constrained()->cascadeOnDelete();
            $table->date('date');
            $table->decimal('feed_kg', 10, 2);
            $table->text('note')->nullable();
            $table->timestamps();
            $table->unique(['house_id', 'date']);
        });

        Schema::create('water_logs', function (Blueprint $table) {
            $table->id();
            $table->foreignId('house_id')->constrained()->cascadeOnDelete();
            $table->date('date');
            $table->decimal('water_l', 10, 2);
            $table->text('note')->nullable();
            $table->timestamps();
            $table->unique(['house_id', 'date']);
        });

        Schema::create('mortality_logs', function (Blueprint $table) {
            $table->id();
            $table->foreignId('house_id')->constrained()->cascadeOnDelete();
            $table->date('date');
            $table->unsignedInteger('deaths')->default(0);
            $table->text('note')->nullable();
            $table->timestamps();
            $table->unique(['house_id', 'date']);
        });

        Schema::create('medication_logs', function (Blueprint $table) {
            $table->id();
            $table->foreignId('house_id')->constrained()->cascadeOnDelete();
            $table->date('date');
            $table->string('drug');
            $table->string('dosage')->nullable();
            $table->string('reason')->nullable();
            $table->unsignedInteger('withdrawal_days')->default(0);
            $table->text('note')->nullable();
            $table->timestamps();
        });

        Schema::create('cam_frames', function (Blueprint $table) {
            $table->id();
            $table->foreignId('house_id')->constrained()->cascadeOnDelete();
            $table->foreignId('device_id')->constrained()->cascadeOnDelete();
            $table->timestamp('ts');
            $table->string('path');
            $table->json('meta_json')->nullable();
            $table->timestamps();
            $table->index(['house_id', 'ts']);
        });

        Schema::create('behavior_metrics', function (Blueprint $table) {
            $table->id();
            $table->foreignId('house_id')->constrained()->cascadeOnDelete();
            $table->foreignId('device_id')->constrained()->cascadeOnDelete();
            $table->timestamp('ts');
            $table->decimal('activity_score', 5, 3)->nullable();
            $table->decimal('clustering_score', 5, 3)->nullable();
            $table->decimal('abnormal_score', 5, 3)->nullable();
            $table->json('meta_json')->nullable();
            $table->timestamps();
            $table->unique(['house_id', 'device_id', 'ts']);
            $table->index(['house_id', 'ts']);
        });

        Schema::create('alerts', function (Blueprint $table) {
            $table->id();
            $table->foreignId('house_id')->constrained()->cascadeOnDelete();
            $table->string('severity')->default('info');
            $table->string('category')->nullable();
            $table->string('title');
            $table->text('message')->nullable();
            $table->string('status')->default('open');
            $table->json('suggested_actions_json')->nullable();
            $table->timestamps();
            $table->index(['house_id', 'status']);
        });

        Schema::create('expected_growth_curves', function (Blueprint $table) {
            $table->id();
            $table->unsignedInteger('age_day')->unique();
            $table->decimal('expected_weight_g', 10, 2);
            $table->timestamps();
        });
    }

    public function down()
    {
        Schema::dropIfExists('expected_growth_curves');
        Schema::dropIfExists('alerts');
        Schema::dropIfExists('behavior_metrics');
        Schema::dropIfExists('cam_frames');
        Schema::dropIfExists('medication_logs');
        Schema::dropIfExists('mortality_logs');
        Schema::dropIfExists('water_logs');
        Schema::dropIfExists('feed_logs');
        Schema::dropIfExists('weight_samples');
        Schema::dropIfExists('sensor_readings');
        Schema::dropIfExists('sensors');
        Schema::dropIfExists('houses');
    }
};
