<?php

namespace Database\Seeders;

use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;
use App\Models\Tenant;
use App\Models\Asset;
use App\Models\Checklist;
use App\Models\Device;
use App\Models\DowntimeReasonCode;
use App\Models\FuelLog;
use App\Models\InventoryItem;
use App\Models\InventoryLocation;
use App\Models\NotificationRoute;
use App\Models\Part;
use App\Models\PmSchedule;
use App\Models\Role;
use App\Models\Site;
use App\Models\Tyre;
use App\Models\Vendor;
use App\Models\Vehicle;
use App\Models\Maintenance;
use App\Models\Trip;
use App\Models\WorkOrder;
use App\Models\AlarmRule;
use App\Models\Telemetry;
use App\Services\AssetRegisterImportService;

class DatabaseSeeder extends Seeder
{
    /**
     * Seed the application's database.
     *
     * @return void
     */
    public function run()
    {
        $tenant = Tenant::firstOrCreate(
            ['email' => 'ops@fourways.example'],
            [
                'name' => 'Fourways Group',
                'password' => Hash::make('password'),
                'subdomain' => 'fourways',
                'subscription_plan' => 'Enterprise',
                'status' => 'active',
            ]
        );

        $this->seedRoles($tenant->id);
        $checklists = $this->seedChecklists($tenant->id);
        $downtimeReasons = $this->seedDowntimeReasons($tenant->id);

        Vendor::firstOrCreate(
            ['tenant_id' => $tenant->id, 'name' => 'Summit Industrial Supplies'],
            [
                'contact_name' => 'Pat Rivera',
                'phone' => '+1 555 010 433',
                'email' => 'sales@summit.example',
                'lead_time_days' => 5,
            ]
        );

        $importPath = base_path('Compiled_Asset_Register_with_Spares_and_Budgets_UPDATED_v2.xlsx');
        if (is_file($importPath)) {
            app(AssetRegisterImportService::class)->import($importPath, $tenant->id);
        }

        $sites = $this->seedDefaultSites($tenant->id);
        $this->seedTenantUsers($tenant, $sites);

        if (Asset::where('tenant_id', $tenant->id)->count() === 0) {
            $this->seedFallbackAssets($tenant->id, $sites);
        }

        if (Part::where('tenant_id', $tenant->id)->count() === 0) {
            $this->seedFallbackParts($tenant->id, $sites);
        }

        $assets = Asset::where('tenant_id', $tenant->id)->get();
        $vehicles = $this->seedVehiclesFromAssets($tenant->id, $assets, $sites);

        $sequence = WorkOrder::where('tenant_id', $tenant->id)->count() + 200;
        $assets->each(function ($asset) use ($tenant, $checklists, $downtimeReasons, &$sequence) {
            $meter = $asset->meter_reading ?? rand(1200, 6200);
            $interval = $asset->maintenance_interval ?: 250;
            $meterType = $asset->meter_type ?: 'hours';

            Maintenance::create([
                'tenant_id' => $tenant->id,
                'asset_id' => $asset->id,
                'scheduled_date' => now()->addDays(rand(5, 20)),
                'status' => 'pending',
                'description' => 'Scheduled service',
                'usage_hours' => rand(50, 200),
                'meter_reading' => $meter + rand(20, 120),
            ]);

            PmSchedule::create([
                'tenant_id' => $tenant->id,
                'asset_id' => $asset->id,
                'name' => $interval . ' hr Service',
                'schedule_type' => 'meter',
                'interval_value' => $interval,
                'interval_unit' => $meterType === 'km' ? 'km' : 'hours',
                'meter_type' => $meterType,
                'next_meter_reading' => $meter + $interval,
                'active' => true,
            ]);

            FuelLog::create([
                'tenant_id' => $tenant->id,
                'asset_id' => $asset->id,
                'site_id' => $asset->site_id,
                'source' => 'bowser',
                'fuel_type' => 'diesel',
                'quantity' => rand(80, 220),
                'unit_cost' => 1.2,
                'total_cost' => 144,
                'meter_reading' => $meter,
                'logged_at' => now()->subDays(rand(1, 4)),
            ]);

            $referenceCode = $this->nextWorkOrderReference($tenant->id, $sequence);
            WorkOrder::create([
                'tenant_id' => $tenant->id,
                'reference_code' => $referenceCode,
                'source' => 'inspection',
                'asset_id' => $asset->id,
                'checklist_id' => $checklists->random()->id,
                'downtime_reason_id' => $downtimeReasons->random()->id,
                'type' => 'preventive',
                'status' => 'open',
                'priority' => 'medium',
                'description' => 'Quarterly inspection and lubrication.',
                'due_at' => now()->addDays(rand(5, 12)),
                'total_cost' => rand(1200, 5800),
            ]);
        });

        $assets->take(6)->each(function ($asset) use ($tenant) {
            Tyre::create([
                'tenant_id' => $tenant->id,
                'asset_id' => $asset->id,
                'position' => 'Front Left',
                'serial_number' => strtoupper(fake()->bothify('TY-####')),
                'brand' => 'Bridgestone',
                'model' => 'MX-65',
                'tread_depth' => 22.5,
                'status' => 'active',
                'installed_at' => now()->subMonths(4),
            ]);
        });

        $vehicles->each(function ($vehicle) use ($tenant) {
            Trip::create([
                'tenant_id' => $tenant->id,
                'vehicle_id' => $vehicle->id,
                'start_location' => 'Quarry Face 3',
                'end_location' => 'Primary Crusher',
                'start_time' => now()->subHours(rand(6, 30)),
                'end_time' => now()->subHours(rand(1, 5)),
                'distance' => rand(8, 22),
                'status' => 'completed',
            ]);

            Device::create([
                'tenant_id' => $tenant->id,
                'asset_id' => $vehicle->asset_id,
                'name' => 'CAN Gateway',
                'type' => 'CAN',
                'identifier' => 'DEV-' . strtoupper(fake()->bothify('###??')),
                'meta' => ['source' => 'seed'],
                'last_seen_at' => now()->subMinutes(rand(3, 40)),
            ]);

            Telemetry::create([
                'tenant_id' => $tenant->id,
                'vehicle_id' => $vehicle->id,
                'asset_id' => $vehicle->asset_id,
                'timestamp' => now()->subMinutes(rand(5, 45)),
                'usage_hours' => rand(1, 5),
                'data' => [
                    'speed' => rand(18, 42),
                    'fuel_rate' => rand(32, 58),
                    'engine_temp' => rand(78, 92),
                ],
            ]);
        });

        AlarmRule::firstOrCreate([
            'tenant_id' => $tenant->id,
            'name' => 'High coolant temp',
        ], [
            'metric' => 'coolant_temp',
            'comparison' => '>',
            'threshold' => 95,
            'severity' => 'critical',
            'actions' => ['auto_work_order' => true],
            'enabled' => true,
        ]);

        NotificationRoute::firstOrCreate([
            'tenant_id' => $tenant->id,
            'event_type' => 'alarm.triggered',
        ], [
            'channel_order' => ['whatsapp', 'sms'],
            'recipients' => ['+1555010101', '+1555010202'],
            'quiet_hours' => ['start' => '22:00', 'end' => '06:00'],
            'escalation_minutes' => 30,
        ]);
    }

    private function seedRoles(int $tenantId)
    {
        collect([
            'Admin',
            'Maintenance',
            'Operator',
            'Stores',
            'Safety',
            'Finance',
            'Viewer',
            'Managing Director',
            'Engineering Manager',
            'Engineer',
            'Foreman',
            'Artisan',
            'Planner',
            'Accountant',
            'Finance Manager',
        ])->each(function ($name) use ($tenantId) {
            Role::firstOrCreate([
                'tenant_id' => $tenantId,
                'name' => $name,
            ]);
        });
    }

    private function seedTenantUsers(Tenant $tenant, $sites): void
    {
        $defaultPassword = Hash::make('password');
        $defaultSiteId = $sites['quarry']->id ?? null;

        $profiles = [
            ['name' => 'Tendai Moyo', 'email' => 'md@fourways.example', 'role' => 'Managing Director', 'site_id' => null],
            ['name' => 'Faith Ncube', 'email' => 'engineering.manager@fourways.example', 'role' => 'Engineering Manager', 'site_id' => $defaultSiteId],
            ['name' => 'Leon Chikowore', 'email' => 'engineer@fourways.example', 'role' => 'Engineer', 'site_id' => $defaultSiteId],
            ['name' => 'Sibusiso Dube', 'email' => 'foreman@fourways.example', 'role' => 'Foreman', 'site_id' => $defaultSiteId],
            ['name' => 'Nomsa Zulu', 'email' => 'artisan@fourways.example', 'role' => 'Artisan', 'site_id' => $defaultSiteId],
            ['name' => 'Tariro Banda', 'email' => 'planner@fourways.example', 'role' => 'Planner', 'site_id' => $defaultSiteId],
            ['name' => 'Kevin Mhlanga', 'email' => 'accountant@fourways.example', 'role' => 'Accountant', 'site_id' => null],
            ['name' => 'Joyce Mutasa', 'email' => 'operator@fourways.example', 'role' => 'Operator', 'site_id' => $defaultSiteId],
            ['name' => 'Alfred Chirwa', 'email' => 'finance.manager@fourways.example', 'role' => 'Finance Manager', 'site_id' => null],
        ];

        foreach ($profiles as $profile) {
            $user = \App\Models\User::firstOrCreate(
                ['email' => $profile['email']],
                [
                    'tenant_id' => $tenant->id,
                    'name' => $profile['name'],
                    'password' => $defaultPassword,
                    'site_id' => $profile['site_id'],
                ]
            );

            if (!$user->tenant_id) {
                $user->tenant_id = $tenant->id;
                $user->site_id = $profile['site_id'];
                $user->save();
            }

            $role = Role::where('tenant_id', $tenant->id)
                ->where('name', $profile['role'])
                ->first();

            if ($role) {
                $user->roles()->syncWithoutDetaching([$role->id]);
            }
        }
    }

    private function seedChecklists(int $tenantId)
    {
        return collect([
            ['name' => 'Pre-start Checks', 'type' => 'pre-start'],
            ['name' => 'Safety Inspection', 'type' => 'safety'],
            ['name' => 'Concrete Dispatch', 'type' => 'inspection'],
        ])->map(function ($data) use ($tenantId) {
            return Checklist::firstOrCreate(
                ['tenant_id' => $tenantId, 'name' => $data['name']],
                [
                    'type' => $data['type'],
                    'items' => ['Visual inspection', 'Fluid levels', 'Brake test', 'Hydraulic lines'],
                ]
            );
        });
    }

    private function seedDowntimeReasons(int $tenantId)
    {
        return collect([
            ['code' => 'BRK', 'label' => 'Breakdown'],
            ['code' => 'NOP', 'label' => 'No Operator'],
            ['code' => 'WPR', 'label' => 'Waiting Parts'],
            ['code' => 'FUE', 'label' => 'Fuel Delay'],
        ])->map(function ($data) use ($tenantId) {
            return DowntimeReasonCode::firstOrCreate(
                ['tenant_id' => $tenantId, 'code' => $data['code']],
                ['label' => $data['label']]
            );
        });
    }

    private function seedDefaultSites(int $tenantId)
    {
        $defaults = [
            ['key' => 'quarry', 'name' => 'Fourways Quarry', 'code' => 'QRY', 'type' => 'quarry'],
            ['key' => 'crushing', 'name' => 'Crushing Plant', 'code' => 'CRS', 'type' => 'plant'],
            ['key' => 'concrete', 'name' => 'Concrete Plant', 'code' => 'CON', 'type' => 'plant'],
            ['key' => 'workshop', 'name' => 'Central Workshop', 'code' => 'WSH', 'type' => 'workshop'],
            ['key' => 'yard', 'name' => 'Service Yard', 'code' => 'YRD', 'type' => 'yard'],
        ];

        $sites = collect();
        foreach ($defaults as $site) {
            $record = Site::firstOrCreate(
                ['tenant_id' => $tenantId, 'name' => $site['name']],
                [
                    'code' => $site['code'],
                    'type' => $site['type'],
                    'address' => 'Fourways Industrial Estate',
                    'latitude' => -26.016,
                    'longitude' => 28.013,
                    'active' => true,
                ]
            );
            $sites[$site['key']] = $record;
        }

        return $sites;
    }

    private function seedFallbackAssets(int $tenantId, $sites)
    {
        $assetCatalog = [
            ['tag' => 'EX-12', 'name' => 'Excavator EX-12', 'category' => 'Mobile Plant', 'asset_type' => 'mobile', 'site_key' => 'quarry', 'make' => 'Caterpillar', 'model' => '320D', 'year' => '2018', 'meter_type' => 'hours', 'meter' => 4820, 'value' => 1240000, 'interval' => 250],
            ['tag' => 'EX-09', 'name' => 'Excavator EX-09', 'category' => 'Mobile Plant', 'asset_type' => 'mobile', 'site_key' => 'quarry', 'make' => 'Komatsu', 'model' => 'PC210', 'year' => '2017', 'meter_type' => 'hours', 'meter' => 5160, 'value' => 980000, 'interval' => 250],
            ['tag' => 'L-07', 'name' => 'Loader L-07', 'category' => 'Mobile Plant', 'asset_type' => 'mobile', 'site_key' => 'quarry', 'make' => 'Volvo', 'model' => 'L120H', 'year' => '2019', 'meter_type' => 'hours', 'meter' => 3920, 'value' => 880000, 'interval' => 200],
            ['tag' => 'D-04', 'name' => 'Dozer D-04', 'category' => 'Mobile Plant', 'asset_type' => 'mobile', 'site_key' => 'quarry', 'make' => 'Caterpillar', 'model' => 'D6T', 'year' => '2016', 'meter_type' => 'hours', 'meter' => 6200, 'value' => 740000, 'interval' => 200],
            ['tag' => 'GR-02', 'name' => 'Grader GR-02', 'category' => 'Mobile Plant', 'asset_type' => 'mobile', 'site_key' => 'quarry', 'make' => 'John Deere', 'model' => '770G', 'year' => '2018', 'meter_type' => 'hours', 'meter' => 4100, 'value' => 690000, 'interval' => 250],
            ['tag' => 'DR-01', 'name' => 'Drill Rig DR-01', 'category' => 'Mobile Plant', 'asset_type' => 'mobile', 'site_key' => 'quarry', 'make' => 'Sandvik', 'model' => 'DP1100i', 'year' => '2020', 'meter_type' => 'hours', 'meter' => 1880, 'value' => 1320000, 'interval' => 200],
            ['tag' => 'H-44', 'name' => 'Hauler H-44', 'category' => 'Truck', 'asset_type' => 'vehicle', 'site_key' => 'quarry', 'make' => 'Caterpillar', 'model' => '777G', 'year' => '2019', 'meter_type' => 'hours', 'meter' => 3600, 'value' => 1750000, 'interval' => 250],
            ['tag' => 'H-32', 'name' => 'Hauler H-32', 'category' => 'Truck', 'asset_type' => 'vehicle', 'site_key' => 'quarry', 'make' => 'Komatsu', 'model' => 'HD325', 'year' => '2017', 'meter_type' => 'hours', 'meter' => 4480, 'value' => 1620000, 'interval' => 250],
            ['tag' => 'WB-03', 'name' => 'Water Bowser WB-03', 'category' => 'Support Truck', 'asset_type' => 'vehicle', 'site_key' => 'quarry', 'make' => 'Hino', 'model' => '500', 'year' => '2020', 'meter_type' => 'km', 'meter' => 68220, 'value' => 320000, 'interval' => 150],
            ['tag' => 'CR-01', 'name' => 'Primary Crusher CR-01', 'category' => 'Crushing Plant', 'asset_type' => 'fixed', 'site_key' => 'crushing', 'make' => 'Metso', 'model' => 'C120', 'year' => '2018', 'meter_type' => 'hours', 'meter' => 6400, 'value' => 2100000, 'interval' => 300],
            ['tag' => 'CR-02', 'name' => 'Secondary Crusher CR-02', 'category' => 'Crushing Plant', 'asset_type' => 'fixed', 'site_key' => 'crushing', 'make' => 'Sandvik', 'model' => 'CH440', 'year' => '2018', 'meter_type' => 'hours', 'meter' => 5980, 'value' => 1800000, 'interval' => 300],
            ['tag' => 'SC-01', 'name' => 'Screen SC-01', 'category' => 'Crushing Plant', 'asset_type' => 'fixed', 'site_key' => 'crushing', 'make' => 'Terex', 'model' => 'TSW2060', 'year' => '2019', 'meter_type' => 'hours', 'meter' => 4320, 'value' => 820000, 'interval' => 250],
            ['tag' => 'CV-03', 'name' => 'Conveyor CV-03', 'category' => 'Crushing Plant', 'asset_type' => 'fixed', 'site_key' => 'crushing', 'make' => 'Superior', 'model' => '36x80', 'year' => '2017', 'meter_type' => 'hours', 'meter' => 7200, 'value' => 410000, 'interval' => 300],
            ['tag' => 'FD-01', 'name' => 'Feeder FD-01', 'category' => 'Crushing Plant', 'asset_type' => 'fixed', 'site_key' => 'crushing', 'make' => 'Telsmith', 'model' => 'H3248', 'year' => '2018', 'meter_type' => 'hours', 'meter' => 5010, 'value' => 460000, 'interval' => 250],
            ['tag' => 'ST-02', 'name' => 'Stacker ST-02', 'category' => 'Crushing Plant', 'asset_type' => 'fixed', 'site_key' => 'crushing', 'make' => 'McCloskey', 'model' => 'TS4080', 'year' => '2021', 'meter_type' => 'hours', 'meter' => 1880, 'value' => 620000, 'interval' => 200],
            ['tag' => 'GEN-200', 'name' => 'Generator GEN-200', 'category' => 'Generator', 'asset_type' => 'fixed', 'site_key' => 'crushing', 'make' => 'Cummins', 'model' => 'C200', 'year' => '2019', 'meter_type' => 'hours', 'meter' => 3120, 'value' => 180000, 'interval' => 250],
            ['tag' => 'GEN-110', 'name' => 'Generator GEN-110', 'category' => 'Generator', 'asset_type' => 'fixed', 'site_key' => 'crushing', 'make' => 'Perkins', 'model' => '1104C', 'year' => '2018', 'meter_type' => 'hours', 'meter' => 2900, 'value' => 140000, 'interval' => 250],
            ['tag' => 'BP-01', 'name' => 'Batch Plant BP-01', 'category' => 'Concrete Plant', 'asset_type' => 'fixed', 'site_key' => 'concrete', 'make' => 'Liebherr', 'model' => 'Betomix', 'year' => '2020', 'meter_type' => 'hours', 'meter' => 2200, 'value' => 1600000, 'interval' => 300],
            ['tag' => 'CS-01', 'name' => 'Cement Silo CS-01', 'category' => 'Concrete Plant', 'asset_type' => 'fixed', 'site_key' => 'concrete', 'make' => 'SiloPro', 'model' => 'CS50', 'year' => '2020', 'meter_type' => 'hours', 'meter' => 1400, 'value' => 220000, 'interval' => 300],
            ['tag' => 'MT-04', 'name' => 'Mixer Truck MT-04', 'category' => 'Truck', 'asset_type' => 'vehicle', 'site_key' => 'concrete', 'make' => 'Mercedes', 'model' => 'Arocs', 'year' => '2019', 'meter_type' => 'km', 'meter' => 82000, 'value' => 520000, 'interval' => 150],
            ['tag' => 'MT-09', 'name' => 'Mixer Truck MT-09', 'category' => 'Truck', 'asset_type' => 'vehicle', 'site_key' => 'concrete', 'make' => 'Volvo', 'model' => 'FMX', 'year' => '2018', 'meter_type' => 'km', 'meter' => 91000, 'value' => 490000, 'interval' => 150],
            ['tag' => 'TP-12', 'name' => 'Tipper TP-12', 'category' => 'Truck', 'asset_type' => 'vehicle', 'site_key' => 'concrete', 'make' => 'Scania', 'model' => 'P410', 'year' => '2017', 'meter_type' => 'km', 'meter' => 120000, 'value' => 380000, 'interval' => 150],
            ['tag' => 'TP-24', 'name' => 'Tipper TP-24', 'category' => 'Truck', 'asset_type' => 'vehicle', 'site_key' => 'concrete', 'make' => 'MAN', 'model' => 'TGS', 'year' => '2019', 'meter_type' => 'km', 'meter' => 98000, 'value' => 410000, 'interval' => 150],
            ['tag' => 'CP-01', 'name' => 'Concrete Pump CP-01', 'category' => 'Concrete Plant', 'asset_type' => 'mobile', 'site_key' => 'concrete', 'make' => 'Putzmeister', 'model' => 'M42', 'year' => '2021', 'meter_type' => 'hours', 'meter' => 980, 'value' => 780000, 'interval' => 200],
            ['tag' => 'GEN-075', 'name' => 'Generator GEN-075', 'category' => 'Generator', 'asset_type' => 'fixed', 'site_key' => 'concrete', 'make' => 'Cummins', 'model' => 'C75', 'year' => '2020', 'meter_type' => 'hours', 'meter' => 1600, 'value' => 120000, 'interval' => 250],
            ['tag' => 'ST-06', 'name' => 'Service Truck ST-06', 'category' => 'Support Truck', 'asset_type' => 'vehicle', 'site_key' => 'workshop', 'make' => 'Isuzu', 'model' => 'N-Series', 'year' => '2018', 'meter_type' => 'km', 'meter' => 76000, 'value' => 210000, 'interval' => 150],
        ];

        foreach ($assetCatalog as $row) {
            $site = $sites[$row['site_key']] ?? Site::firstWhere('tenant_id', $tenantId);
            Asset::create([
                'tenant_id' => $tenantId,
                'site_id' => $site?->id,
                'name' => $row['name'],
                'asset_tag' => $row['tag'],
                'make' => $row['make'],
                'model' => $row['model'],
                'serial_number' => 'SN-' . str_replace('-', '', $row['tag']),
                'year' => $row['year'],
                'category' => $row['category'],
                'asset_type' => $row['asset_type'],
                'lifecycle_status' => 'active',
                'commissioned_at' => now()->subYears(5),
                'purchase_date' => now()->subYears(6),
                'warranty_expiry' => now()->addYear(),
                'description' => $row['category'] . ' asset operating at ' . ($site?->name ?? 'Site'),
                'status' => 'active',
                'location' => $site?->name,
                'current_value' => $row['value'],
                'maintenance_interval' => $row['interval'],
                'meter_type' => $row['meter_type'],
                'meter_reading' => $row['meter'],
            ]);
        }
    }

    private function seedFallbackParts(int $tenantId, $sites)
    {
        Part::insert([
            [
                'tenant_id' => $tenantId,
                'sku' => 'FLT-100',
                'name' => 'Hydraulic Filter',
                'category' => 'Filters',
                'unit' => 'pcs',
                'created_at' => now(),
                'updated_at' => now(),
            ],
            [
                'tenant_id' => $tenantId,
                'sku' => 'OIL-80W',
                'name' => 'Engine Oil 80W',
                'category' => 'Lubricants',
                'unit' => 'liters',
                'created_at' => now(),
                'updated_at' => now(),
            ],
            [
                'tenant_id' => $tenantId,
                'sku' => 'BRG-220',
                'name' => 'Conveyor Bearing',
                'category' => 'Crushing Plant',
                'unit' => 'pcs',
                'created_at' => now(),
                'updated_at' => now(),
            ],
        ]);

        $location = InventoryLocation::firstOrCreate(
            ['tenant_id' => $tenantId, 'name' => 'Main Store'],
            ['code' => 'MS-1', 'site_id' => $sites['workshop']->id ?? null]
        );

        $partRecords = Part::where('tenant_id', $tenantId)->get();
        foreach ($partRecords as $part) {
            InventoryItem::firstOrCreate([
                'tenant_id' => $tenantId,
                'part_id' => $part->id,
                'location_id' => $location->id,
            ], [
                'quantity' => 28,
                'min_quantity' => 6,
                'max_quantity' => 50,
                'reorder_point' => 10,
            ]);
        }
    }

    private function seedVehiclesFromAssets(int $tenantId, $assets, $sites)
    {
        $vehicles = collect();
        foreach ($assets as $asset) {
            $haystack = strtolower($asset->name . ' ' . $asset->category . ' ' . $asset->asset_type);
            $isVehicle = Str::contains($haystack, ['truck', 'vehicle', 'hauler', 'tipper', 'mixer', 'bowser']);
            if (!$isVehicle) {
                continue;
            }

            $basePlate = strtoupper(Str::slug($asset->asset_tag ?: $asset->name, ''));
            $licensePlate = $this->nextLicensePlate($basePlate ?: strtoupper(fake()->bothify('###??')));

            $vehicles->push(Vehicle::updateOrCreate(
                ['tenant_id' => $tenantId, 'asset_id' => $asset->id],
                [
                    'site_id' => $asset->site_id ?? ($sites['yard']->id ?? null),
                    'name' => $asset->name,
                    'type' => $asset->category ?: 'vehicle',
                    'license_plate' => $licensePlate,
                    'status' => 'active',
                    'telemetry_data' => ['fuel_rate' => rand(32, 58), 'idle_hours' => rand(2, 6)],
                    'maintenance_schedule' => ['next_service_hours' => rand(60, 180)],
                    'odometer' => rand(48000, 125000),
                    'engine_hours' => rand(1200, 5200),
                ]
            ));
        }

        if ($vehicles->isEmpty()) {
            $assets->take(5)->each(function ($asset) use ($tenantId, $sites, $vehicles) {
                $basePlate = strtoupper(Str::slug($asset->asset_tag ?: $asset->name, ''));
                $licensePlate = $this->nextLicensePlate($basePlate ?: strtoupper(fake()->bothify('###??')));

                $vehicles->push(Vehicle::updateOrCreate(
                    ['tenant_id' => $tenantId, 'asset_id' => $asset->id],
                    [
                        'site_id' => $asset->site_id ?? ($sites['yard']->id ?? null),
                        'name' => $asset->name,
                        'type' => $asset->category ?: 'vehicle',
                        'license_plate' => $licensePlate,
                        'status' => 'active',
                        'telemetry_data' => ['fuel_rate' => rand(32, 58), 'idle_hours' => rand(2, 6)],
                        'maintenance_schedule' => ['next_service_hours' => rand(60, 180)],
                        'odometer' => rand(48000, 125000),
                        'engine_hours' => rand(1200, 5200),
                    ]
                ));
            });
        }

        return $vehicles;
    }

    private function nextWorkOrderReference(int $tenantId, int &$sequence): string
    {
        do {
            $reference = 'WO-' . str_pad((string) $sequence, 4, '0', STR_PAD_LEFT);
            $sequence++;
        } while (WorkOrder::where('tenant_id', $tenantId)->where('reference_code', $reference)->exists());

        return $reference;
    }

    private function nextLicensePlate(string $basePlate): string
    {
        $plate = $basePlate !== '' ? substr($basePlate, 0, 10) : strtoupper(Str::random(6));
        $suffix = 1;

        while (Vehicle::where('license_plate', $plate)->exists()) {
            $plate = substr($basePlate, 0, 8) . $suffix;
            $suffix++;
        }

        return $plate;
    }
}
