<?php

namespace Database\Seeders;

use App\Models\Alert;
use App\Models\BehaviorMetric;
use App\Models\Device;
use App\Models\ExpectedGrowthCurve;
use App\Models\FeedLog;
use App\Models\House;
use App\Models\MortalityLog;
use App\Models\Role;
use App\Models\Sensor;
use App\Models\SensorReading;
use App\Models\Tenant;
use App\Models\User;
use App\Models\WaterLog;
use App\Models\WeightSample;
use Carbon\Carbon;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;

class PoultryDemoSeeder extends Seeder
{
    public function run(): void
    {
        $tenant = Tenant::firstOrCreate([
            'email' => 'poultryops@example.com',
        ], [
            'name' => 'PoultryOps Farm',
            'password' => Hash::make('password'),
            'subdomain' => 'poultryops',
            'database_name' => null,
            'subscription_plan' => 'mvp',
            'status' => 'active',
        ]);

        $user = User::firstOrCreate([
            'email' => 'owner@poultryops.local',
        ], [
            'name' => 'Farm Owner',
            'password' => Hash::make('password'),
            'tenant_id' => $tenant->id,
        ]);

        foreach (['owner', 'supervisor', 'worker'] as $roleName) {
            $role = Role::firstOrCreate([
                'tenant_id' => $tenant->id,
                'name' => $roleName,
            ], [
                'description' => ucfirst($roleName) . ' role',
            ]);
            if (!$user->roles()->where('roles.id', $role->id)->exists()) {
                $user->roles()->attach($role->id);
            }
        }

        $houses = [
            House::firstOrCreate([
                'tenant_id' => $tenant->id,
                'name' => 'House A',
            ], [
                'capacity' => 2500,
                'age_days' => 21,
                'timezone' => 'Africa/Harare',
                'notes' => 'Main broiler house',
            ]),
            House::firstOrCreate([
                'tenant_id' => $tenant->id,
                'name' => 'House B',
            ], [
                'capacity' => 2500,
                'age_days' => 18,
                'timezone' => 'Africa/Harare',
                'notes' => 'Secondary broiler house',
            ]),
        ];

        if (ExpectedGrowthCurve::count() === 0) {
            for ($day = 0; $day <= 42; $day++) {
                ExpectedGrowthCurve::create([
                    'age_day' => $day,
                    'expected_weight_g' => max(40, 40 + ($day * 65)),
                ]);
            }
        }

        foreach ($houses as $house) {
            $devices = [
                'esp32_cam' => Device::firstOrCreate([
                    'tenant_id' => $tenant->id,
                    'house_id' => $house->id,
                    'type' => 'esp32_cam',
                    'name' => $house->name . ' Cam',
                ], [
                    'api_key' => Str::random(40),
                    'identifier' => 'cam-' . Str::random(6),
                    'firmware_version' => '1.0.0',
                ]),
                'esp32_sensor' => Device::firstOrCreate([
                    'tenant_id' => $tenant->id,
                    'house_id' => $house->id,
                    'type' => 'esp32_sensor',
                    'name' => $house->name . ' Sensor',
                ], [
                    'api_key' => Str::random(40),
                    'identifier' => 'sensor-' . Str::random(6),
                    'firmware_version' => '1.0.0',
                ]),
                'esp32_scale' => Device::firstOrCreate([
                    'tenant_id' => $tenant->id,
                    'house_id' => $house->id,
                    'type' => 'esp32_scale',
                    'name' => $house->name . ' Scale',
                ], [
                    'api_key' => Str::random(40),
                    'identifier' => 'scale-' . Str::random(6),
                    'firmware_version' => '1.0.0',
                ]),
            ];

            $sensorDevice = $devices['esp32_sensor'];
            $sensorTypes = [
                ['type' => 'temp_c', 'unit' => 'C'],
                ['type' => 'humidity_pct', 'unit' => '%'],
                ['type' => 'ammonia_ppm', 'unit' => 'ppm'],
                ['type' => 'water_flow_lpm', 'unit' => 'LPM'],
                ['type' => 'feed_level_pct', 'unit' => '%'],
                ['type' => 'power_state', 'unit' => 'bool'],
            ];

            $sensors = [];
            foreach ($sensorTypes as $sensorType) {
                $sensor = Sensor::firstOrCreate([
                    'device_id' => $sensorDevice->id,
                    'type' => $sensorType['type'],
                ], [
                    'unit' => $sensorType['unit'],
                    'name' => strtoupper($sensorType['type']),
                ]);
                $sensors[$sensorType['type']] = $sensor;
            }

            $start = Carbon::now('UTC')->subDays(7)->startOfDay();
            for ($day = 0; $day < 7; $day++) {
                $date = $start->copy()->addDays($day);
                FeedLog::updateOrCreate([
                    'house_id' => $house->id,
                    'date' => $date->toDateString(),
                ], [
                    'feed_kg' => 120 + ($day * 8) + rand(-5, 5),
                ]);

                WaterLog::updateOrCreate([
                    'house_id' => $house->id,
                    'date' => $date->toDateString(),
                ], [
                    'water_l' => 260 + ($day * 12) + rand(-10, 10),
                ]);

                MortalityLog::updateOrCreate([
                    'house_id' => $house->id,
                    'date' => $date->toDateString(),
                ], [
                    'deaths' => rand(0, 4),
                ]);

                WeightSample::updateOrCreate([
                    'house_id' => $house->id,
                    'source_device_id' => $devices['esp32_scale']->id,
                    'ts' => $date->copy()->addHours(7)->toDateTimeString(),
                ], [
                    'avg_weight_g' => 900 + ($day * 60) + rand(-20, 20),
                    'sample_count' => 50,
                ]);

                BehaviorMetric::updateOrCreate([
                    'house_id' => $house->id,
                    'device_id' => $devices['esp32_cam']->id,
                    'ts' => $date->copy()->addHours(9)->toDateTimeString(),
                ], [
                    'activity_score' => rand(60, 90) / 100,
                    'clustering_score' => rand(30, 70) / 100,
                    'abnormal_score' => rand(10, 40) / 100,
                ]);

                for ($hour = 0; $hour < 24; $hour += 6) {
                    $ts = $date->copy()->addHours($hour);
                    SensorReading::updateOrCreate([
                        'sensor_id' => $sensors['temp_c']->id,
                        'ts' => $ts->toDateTimeString(),
                    ], [
                        'value' => 28 + rand(-2, 2),
                    ]);
                    SensorReading::updateOrCreate([
                        'sensor_id' => $sensors['humidity_pct']->id,
                        'ts' => $ts->toDateTimeString(),
                    ], [
                        'value' => 60 + rand(-5, 5),
                    ]);
                    SensorReading::updateOrCreate([
                        'sensor_id' => $sensors['ammonia_ppm']->id,
                        'ts' => $ts->toDateTimeString(),
                    ], [
                        'value' => 8 + rand(0, 4),
                    ]);
                    SensorReading::updateOrCreate([
                        'sensor_id' => $sensors['water_flow_lpm']->id,
                        'ts' => $ts->toDateTimeString(),
                    ], [
                        'value' => 12 + rand(-2, 2),
                    ]);
                    SensorReading::updateOrCreate([
                        'sensor_id' => $sensors['feed_level_pct']->id,
                        'ts' => $ts->toDateTimeString(),
                    ], [
                        'value' => 70 + rand(-10, 5),
                    ]);
                    SensorReading::updateOrCreate([
                        'sensor_id' => $sensors['power_state']->id,
                        'ts' => $ts->toDateTimeString(),
                    ], [
                        'value' => 1,
                    ]);
                }
            }

            Alert::firstOrCreate([
                'house_id' => $house->id,
                'category' => 'behavior',
                'title' => 'Behavior anomaly',
            ], [
                'severity' => 'warn',
                'message' => 'Elevated clustering detected near feeding time.',
                'status' => 'open',
                'suggested_actions_json' => ['Check feeder access and stocking density.'],
            ]);
        }
    }
}
