"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.addTestCommands = addTestCommands;
exports.envWithoutExperimentalLoaderOptions = envWithoutExperimentalLoaderOptions;
exports.experimentalLoaderOption = experimentalLoaderOption;

var _fs = _interopRequireDefault(require("fs"));

var _url = _interopRequireDefault(require("url"));

var _path = _interopRequireDefault(require("path"));

var _runner = require("./runner");

var _profiler = require("./profiler");

var _html = require("./reporters/html");

var _loader = require("./loader");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Copyright (c) Microsoft Corporation.
 *
 * Licensed under the Apache License, Version 2.0 (the 'License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/* eslint-disable no-console */
function addTestCommands(program) {
  addTestCommand(program);
  addShowReportCommand(program);
  addListFilesCommand(program);
}

function addTestCommand(program) {
  const command = program.command('test [test-filter...]');
  command.description('Run tests with Playwright Test');
  command.option('--browser <browser>', `Browser to use for tests, one of "all", "chromium", "firefox" or "webkit" (default: "chromium")`);
  command.option('--headed', `Run tests in headed browsers (default: headless)`);
  command.option('--debug', `Run tests with Playwright Inspector. Shortcut for "PWDEBUG=1" environment variable and "--timeout=0 --maxFailures=1 --headed --workers=1" options`);
  command.option('-c, --config <file>', `Configuration file, or a test directory with optional ${_runner.kDefaultConfigFiles.map(file => `"${file}"`).join('/')}`);
  command.option('--forbid-only', `Fail if test.only is called (default: false)`);
  command.option('--fully-parallel', `Run all tests in parallel (default: false)`);
  command.option('-g, --grep <grep>', `Only run tests matching this regular expression (default: ".*")`);
  command.option('-gv, --grep-invert <grep>', `Only run tests that do not match this regular expression`);
  command.option('--global-timeout <timeout>', `Maximum time this test suite can run in milliseconds (default: unlimited)`);
  command.option('-j, --workers <workers>', `Number of concurrent workers, use 1 to run in a single worker (default: number of CPU cores / 2)`);
  command.option('--list', `Collect all the tests and report them, but do not run`);
  command.option('--max-failures <N>', `Stop after the first N failures`);
  command.option('--output <dir>', `Folder for output artifacts (default: "test-results")`);
  command.option('--quiet', `Suppress stdio`);
  command.option('--repeat-each <N>', `Run each test N times (default: 1)`);
  command.option('--reporter <reporter>', `Reporter to use, comma-separated, can be ${_runner.builtInReporters.map(name => `"${name}"`).join(', ')} (default: "${_loader.baseFullConfig.reporter[0]}")`);
  command.option('--retries <retries>', `Maximum retry count for flaky tests, zero for no retries (default: no retries)`);
  command.option('--shard <shard>', `Shard tests and execute only the selected shard, specify in the form "current/all", 1-based, for example "3/5"`);
  command.option('--project <project-name...>', `Only run tests from the specified list of projects (default: run all projects)`);
  command.option('--timeout <timeout>', `Specify test timeout threshold in milliseconds, zero for unlimited (default: ${_loader.defaultTimeout})`);
  command.option('--trace <mode>', `Force tracing mode, can be ${kTraceModes.map(mode => `"${mode}"`).join(', ')}`);
  command.option('-u, --update-snapshots', `Update snapshots with actual results (default: only create missing snapshots)`);
  command.option('-x', `Stop after the first failure`);
  command.action(async (args, opts) => {
    try {
      await runTests(args, opts);
    } catch (e) {
      console.error(e);
      process.exit(1);
    }
  });
  command.addHelpText('afterAll', `
Arguments [test-filter...]:
  Pass arguments to filter test files. Each argument is treated as a regular expression.

Examples:
  $ npx playwright test my.spec.ts
  $ npx playwright test some.spec.ts:42
  $ npx playwright test --headed
  $ npx playwright test --browser=webkit`);
}

function addListFilesCommand(program) {
  const command = program.command('list-files [file-filter...]', {
    hidden: true
  });
  command.description('List files with Playwright Test tests');
  command.option('-c, --config <file>', `Configuration file, or a test directory with optional ${_runner.kDefaultConfigFiles.map(file => `"${file}"`).join('/')}`);
  command.option('--project <project-name...>', `Only run tests from the specified list of projects (default: list all projects)`);
  command.action(async (args, opts) => {
    try {
      await listTestFiles(opts);
    } catch (e) {
      console.error(e);
      process.exit(1);
    }
  });
}

function addShowReportCommand(program) {
  const command = program.command('show-report [report]');
  command.description('show HTML report');
  command.action(report => (0, _html.showHTMLReport)(report));
  command.addHelpText('afterAll', `
Arguments [report]:
  When specified, opens given report, otherwise opens last generated report.

Examples:
  $ npx playwright show-report
  $ npx playwright show-report playwright-report`);
}

async function runTests(args, opts) {
  await (0, _profiler.startProfiling)();
  const overrides = overridesFromOptions(opts);

  if (opts.browser) {
    const browserOpt = opts.browser.toLowerCase();
    if (!['all', 'chromium', 'firefox', 'webkit'].includes(browserOpt)) throw new Error(`Unsupported browser "${opts.browser}", must be one of "all", "chromium", "firefox" or "webkit"`);
    const browserNames = browserOpt === 'all' ? ['chromium', 'firefox', 'webkit'] : [browserOpt];
    overrides.projects = browserNames.map(browserName => {
      return {
        name: browserName,
        use: {
          browserName
        }
      };
    });
  }

  if (opts.headed || opts.debug) overrides.use = {
    headless: false
  };

  if (opts.debug) {
    overrides.maxFailures = 1;
    overrides.timeout = 0;
    overrides.workers = 1;
    process.env.PWDEBUG = '1';
  }

  if (opts.trace) {
    if (!kTraceModes.includes(opts.trace)) throw new Error(`Unsupported trace mode "${opts.trace}", must be one of ${kTraceModes.map(mode => `"${mode}"`).join(', ')}`);
    overrides.use = overrides.use || {};
    overrides.use.trace = opts.trace;
  } // When no --config option is passed, let's look for the config file in the current directory.


  const configFileOrDirectory = opts.config ? _path.default.resolve(process.cwd(), opts.config) : process.cwd();

  const resolvedConfigFile = _runner.Runner.resolveConfigFile(configFileOrDirectory);

  if (restartWithExperimentalTsEsm(resolvedConfigFile)) return;
  const runner = new _runner.Runner(overrides);
  if (resolvedConfigFile) await runner.loadConfigFromResolvedFile(resolvedConfigFile);else await runner.loadEmptyConfig(configFileOrDirectory);
  const testFileFilters = args.map(arg => {
    const match = /^(.*?):(\d+):?(\d+)?$/.exec(arg);
    return {
      re: forceRegExp(match ? match[1] : arg),
      line: match ? parseInt(match[2], 10) : null,
      column: match !== null && match !== void 0 && match[3] ? parseInt(match[3], 10) : null
    };
  });
  const result = await runner.runAllTests({
    listOnly: !!opts.list,
    testFileFilters,
    projectFilter: opts.project || undefined,
    watchMode: !!process.env.PW_TEST_WATCH
  });
  await (0, _profiler.stopProfiling)(undefined);
  if (result.status === 'interrupted') process.exit(130);
  process.exit(result.status === 'passed' ? 0 : 1);
}

async function listTestFiles(opts) {
  // Redefine process.stdout.write in case config decides to pollute stdio.
  const write = process.stdout.write.bind(process.stdout);

  process.stdout.write = () => {};

  const configFileOrDirectory = opts.config ? _path.default.resolve(process.cwd(), opts.config) : process.cwd();

  const resolvedConfigFile = _runner.Runner.resolveConfigFile(configFileOrDirectory);

  if (restartWithExperimentalTsEsm(resolvedConfigFile)) return;
  const runner = new _runner.Runner();
  await runner.loadConfigFromResolvedFile(resolvedConfigFile);
  const report = await runner.listTestFiles(resolvedConfigFile, opts.project);
  write(JSON.stringify(report), () => {
    process.exit(0);
  });
}

function forceRegExp(pattern) {
  const match = pattern.match(/^\/(.*)\/([gi]*)$/);
  if (match) return new RegExp(match[1], match[2]);
  return new RegExp(pattern, 'gi');
}

function overridesFromOptions(options) {
  const shardPair = options.shard ? options.shard.split('/').map(t => parseInt(t, 10)) : undefined;
  return {
    forbidOnly: options.forbidOnly ? true : undefined,
    fullyParallel: options.fullyParallel ? true : undefined,
    globalTimeout: options.globalTimeout ? parseInt(options.globalTimeout, 10) : undefined,
    grep: options.grep ? forceRegExp(options.grep) : undefined,
    grepInvert: options.grepInvert ? forceRegExp(options.grepInvert) : undefined,
    maxFailures: options.x ? 1 : options.maxFailures ? parseInt(options.maxFailures, 10) : undefined,
    outputDir: options.output ? _path.default.resolve(process.cwd(), options.output) : undefined,
    quiet: options.quiet ? options.quiet : undefined,
    repeatEach: options.repeatEach ? parseInt(options.repeatEach, 10) : undefined,
    retries: options.retries ? parseInt(options.retries, 10) : undefined,
    reporter: options.reporter && options.reporter.length ? options.reporter.split(',').map(r => [resolveReporter(r)]) : undefined,
    shard: shardPair ? {
      current: shardPair[0],
      total: shardPair[1]
    } : undefined,
    timeout: options.timeout ? parseInt(options.timeout, 10) : undefined,
    updateSnapshots: options.updateSnapshots ? 'all' : undefined,
    workers: options.workers ? parseInt(options.workers, 10) : undefined
  };
}

function resolveReporter(id) {
  if (_runner.builtInReporters.includes(id)) return id;

  const localPath = _path.default.resolve(process.cwd(), id);

  if (_fs.default.existsSync(localPath)) return localPath;
  return require.resolve(id, {
    paths: [process.cwd()]
  });
}

function restartWithExperimentalTsEsm(configFile) {
  const nodeVersion = +process.versions.node.split('.')[0]; // New experimental loader is only supported on Node 16+.

  if (nodeVersion < 16) return false;
  if (!configFile) return false;
  if (process.env.PW_DISABLE_TS_ESM) return false;
  if (process.env.PW_TS_ESM_ON) return false;
  if (!configFile.endsWith('.ts')) return false;
  if (!(0, _loader.fileIsModule)(configFile)) return false;
  const NODE_OPTIONS = (process.env.NODE_OPTIONS || '') + experimentalLoaderOption();

  const innerProcess = require('child_process').fork(require.resolve('playwright-core/cli'), process.argv.slice(2), {
    env: { ...process.env,
      NODE_OPTIONS,
      PW_TS_ESM_ON: '1'
    }
  });

  innerProcess.on('close', code => {
    if (code !== 0 && code !== null) process.exit(code);
  });
  return true;
}

function experimentalLoaderOption() {
  return ` --experimental-loader=${_url.default.pathToFileURL(require.resolve('@playwright/test/lib/experimentalLoader')).toString()}`;
}

function envWithoutExperimentalLoaderOptions() {
  const substring = experimentalLoaderOption();
  const result = { ...process.env
  };
  if (result.NODE_OPTIONS) result.NODE_OPTIONS = result.NODE_OPTIONS.replace(substring, '').trim() || undefined;
  return result;
}

const kTraceModes = ['on', 'off', 'on-first-retry', 'retain-on-failure'];