"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.FixtureRunner = exports.FixturePool = void 0;
exports.isFixtureOption = isFixtureOption;

var _util = require("./util");

var crypto = _interopRequireWildcard(require("crypto"));

var _manualPromise = require("playwright-core/lib/utils/manualPromise");

function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }

function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }

/**
 * Copyright Microsoft Corporation. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
const kScopeOrder = ['test', 'worker'];

class Fixture {
  constructor(runner, registration) {
    this.runner = void 0;
    this.registration = void 0;
    this.value = void 0;
    this.failed = false;
    this._useFuncFinished = void 0;
    this._selfTeardownComplete = void 0;
    this._teardownWithDepsComplete = void 0;
    this._runnableDescription = void 0;
    this._deps = new Set();
    this._usages = new Set();
    this.runner = runner;
    this.registration = registration;
    this.value = null;
    const title = this.registration.customTitle || this.registration.name;
    this._runnableDescription = {
      title: this.registration.timeout !== undefined ? `Fixture "${title}"` : `setting up "${title}"`,
      location: registration.location,
      slot: this.registration.timeout === undefined ? undefined : {
        timeout: this.registration.timeout,
        elapsed: 0
      }
    };
  }

  async setup(testInfo) {
    if (typeof this.registration.fn !== 'function') {
      this.value = this.registration.fn;
      return;
    }

    const params = {};

    for (const name of this.registration.deps) {
      const registration = this.runner.pool.resolveDependency(this.registration, name);
      const dep = await this.runner.setupFixtureForRegistration(registration, testInfo); // Fixture teardown is root => leafs, when we need to teardown a fixture,
      // it recursively tears down its usages first.

      dep._usages.add(this); // Don't forget to decrement all usages when fixture goes.
      // Otherwise worker-scope fixtures will retain test-scope fixtures forever.


      this._deps.add(dep);

      params[name] = dep.value;

      if (dep.failed) {
        this.failed = true;
        return;
      }
    }

    let called = false;
    const useFuncStarted = new _manualPromise.ManualPromise();
    (0, _util.debugTest)(`setup ${this.registration.name}`);

    const useFunc = async value => {
      if (called) throw new Error(`Cannot provide fixture value for the second time`);
      called = true;
      this.value = value;
      this._useFuncFinished = new _manualPromise.ManualPromise();
      useFuncStarted.resolve();
      await this._useFuncFinished;
    };

    const workerInfo = {
      config: testInfo.config,
      parallelIndex: testInfo.parallelIndex,
      workerIndex: testInfo.workerIndex,
      project: testInfo.project
    };
    const info = this.registration.scope === 'worker' ? workerInfo : testInfo;

    testInfo._timeoutManager.setCurrentFixture(this._runnableDescription);

    this._selfTeardownComplete = Promise.resolve().then(() => this.registration.fn(params, useFunc, info)).catch(e => {
      this.failed = true;
      if (!useFuncStarted.isDone()) useFuncStarted.reject(e);else throw e;
    });
    await useFuncStarted;

    testInfo._timeoutManager.setCurrentFixture(undefined);
  }

  async teardown(timeoutManager) {
    if (this._teardownWithDepsComplete) {
      // When we are waiting for the teardown for the second time,
      // most likely after the first time did timeout, annotate current fixture
      // for better error messages.
      this._setTeardownDescription(timeoutManager);

      await this._teardownWithDepsComplete;
      timeoutManager.setCurrentFixture(undefined);
      return;
    }

    this._teardownWithDepsComplete = this._teardownInternal(timeoutManager);
    await this._teardownWithDepsComplete;
  }

  _setTeardownDescription(timeoutManager) {
    const title = this.registration.customTitle || this.registration.name;
    this._runnableDescription.title = this.registration.timeout !== undefined ? `Fixture "${title}"` : `tearing down "${title}"`;
    timeoutManager.setCurrentFixture(this._runnableDescription);
  }

  async _teardownInternal(timeoutManager) {
    if (typeof this.registration.fn !== 'function') return;

    try {
      for (const fixture of this._usages) await fixture.teardown(timeoutManager);

      if (this._usages.size !== 0) {
        // TODO: replace with assert.
        console.error('Internal error: fixture integrity at', this._runnableDescription.title); // eslint-disable-line no-console

        this._usages.clear();
      }

      if (this._useFuncFinished) {
        (0, _util.debugTest)(`teardown ${this.registration.name}`);

        this._setTeardownDescription(timeoutManager);

        this._useFuncFinished.resolve();

        await this._selfTeardownComplete;
        timeoutManager.setCurrentFixture(undefined);
      }
    } finally {
      for (const dep of this._deps) dep._usages.delete(this);

      this.runner.instanceForId.delete(this.registration.id);
    }
  }

}

function isFixtureTuple(value) {
  return Array.isArray(value) && typeof value[1] === 'object' && ('scope' in value[1] || 'auto' in value[1] || 'option' in value[1] || 'timeout' in value[1]);
}

function isFixtureOption(value) {
  return isFixtureTuple(value) && !!value[1].option;
}

class FixturePool {
  constructor(fixturesList, parentPool, disallowWorkerFixtures) {
    this.digest = void 0;
    this.registrations = void 0;
    this.registrations = new Map(parentPool ? parentPool.registrations : []);

    for (const {
      fixtures,
      location
    } of fixturesList) {
      for (const entry of Object.entries(fixtures)) {
        const name = entry[0];
        let value = entry[1];
        let options;

        if (isFixtureTuple(value)) {
          var _value$1$auto;

          options = {
            auto: (_value$1$auto = value[1].auto) !== null && _value$1$auto !== void 0 ? _value$1$auto : false,
            scope: value[1].scope || 'test',
            option: !!value[1].option,
            timeout: value[1].timeout,
            customTitle: value[1]._title
          };
          value = value[0];
        }

        let fn = value;
        const previous = this.registrations.get(name);

        if (previous && options) {
          if (previous.scope !== options.scope) throw errorWithLocations(`Fixture "${name}" has already been registered as a { scope: '${previous.scope}' } fixture.`, {
            location,
            name
          }, previous);
          if (previous.auto !== options.auto) throw errorWithLocations(`Fixture "${name}" has already been registered as a { auto: '${previous.scope}' } fixture.`, {
            location,
            name
          }, previous);
        } else if (previous) {
          options = {
            auto: previous.auto,
            scope: previous.scope,
            option: previous.option,
            timeout: previous.timeout,
            customTitle: previous.customTitle
          };
        } else if (!options) {
          options = {
            auto: false,
            scope: 'test',
            option: false,
            timeout: undefined,
            customTitle: undefined
          };
        }

        if (!kScopeOrder.includes(options.scope)) throw errorWithLocations(`Fixture "${name}" has unknown { scope: '${options.scope}' }.`, {
          location,
          name
        });
        if (options.scope === 'worker' && disallowWorkerFixtures) throw errorWithLocations(`Cannot use({ ${name} }) in a describe group, because it forces a new worker.\nMake it top-level in the test file or put in the configuration file.`, {
          location,
          name
        }); // Overriding option with "undefined" value means setting it to the default value
        // from the original declaration of the option.

        if (fn === undefined && options.option && previous) {
          let original = previous;

          while (original.super) original = original.super;

          fn = original.fn;
        }

        const deps = fixtureParameterNames(fn, location);
        const registration = {
          id: '',
          name,
          location,
          scope: options.scope,
          fn,
          auto: options.auto,
          option: options.option,
          timeout: options.timeout,
          customTitle: options.customTitle,
          deps,
          super: previous
        };
        registrationId(registration);
        this.registrations.set(name, registration);
      }
    }

    this.digest = this.validate();
  }

  validate() {
    const markers = new Map();
    const stack = [];

    const visit = registration => {
      markers.set(registration, 'visiting');
      stack.push(registration);

      for (const name of registration.deps) {
        const dep = this.resolveDependency(registration, name);

        if (!dep) {
          if (name === registration.name) throw errorWithLocations(`Fixture "${registration.name}" references itself, but does not have a base implementation.`, registration);else throw errorWithLocations(`Fixture "${registration.name}" has unknown parameter "${name}".`, registration);
        }

        if (kScopeOrder.indexOf(registration.scope) > kScopeOrder.indexOf(dep.scope)) throw errorWithLocations(`${registration.scope} fixture "${registration.name}" cannot depend on a ${dep.scope} fixture "${name}".`, registration, dep);

        if (!markers.has(dep)) {
          visit(dep);
        } else if (markers.get(dep) === 'visiting') {
          const index = stack.indexOf(dep);
          const regs = stack.slice(index, stack.length);
          const names = regs.map(r => `"${r.name}"`);
          throw errorWithLocations(`Fixtures ${names.join(' -> ')} -> "${dep.name}" form a dependency cycle.`, ...regs);
        }
      }

      markers.set(registration, 'visited');
      stack.pop();
    };

    const hash = crypto.createHash('sha1');
    const names = Array.from(this.registrations.keys()).sort();

    for (const name of names) {
      const registration = this.registrations.get(name);
      visit(registration);
      if (registration.scope === 'worker') hash.update(registration.id + ';');
    }

    return hash.digest('hex');
  }

  validateFunction(fn, prefix, location) {
    const visit = registration => {
      for (const name of registration.deps) visit(this.resolveDependency(registration, name));
    };

    for (const name of fixtureParameterNames(fn, location)) {
      const registration = this.registrations.get(name);
      if (!registration) throw errorWithLocations(`${prefix} has unknown parameter "${name}".`, {
        location,
        name: prefix,
        quoted: false
      });
      visit(registration);
    }
  }

  resolveDependency(registration, name) {
    if (name === registration.name) return registration.super;
    return this.registrations.get(name);
  }

}

exports.FixturePool = FixturePool;

class FixtureRunner {
  constructor() {
    this.testScopeClean = true;
    this.pool = void 0;
    this.instanceForId = new Map();
  }

  setPool(pool) {
    if (!this.testScopeClean) throw new Error('Did not teardown test scope');

    if (this.pool && pool.digest !== this.pool.digest) {
      throw new Error([`Playwright detected inconsistent test.use() options.`, `Most common mistakes that lead to this issue:`, `  - Calling test.use() outside of the test file, for example in a common helper.`, `  - One test file imports from another test file.`].join('\n'));
    }

    this.pool = pool;
  }

  async teardownScope(scope, timeoutManager) {
    let error; // Teardown fixtures in the reverse order.

    const fixtures = Array.from(this.instanceForId.values()).reverse();

    for (const fixture of fixtures) {
      if (fixture.registration.scope === scope) {
        try {
          await fixture.teardown(timeoutManager);
        } catch (e) {
          if (error === undefined) error = e;
        }
      }
    }

    if (scope === 'test') this.testScopeClean = true;
    if (error !== undefined) throw error;
  }

  async resolveParametersForFunction(fn, testInfo, autoFixtures) {
    // Install automatic fixtures.
    for (const registration of this.pool.registrations.values()) {
      if (registration.auto === false) continue;
      let shouldRun = true;
      if (autoFixtures === 'all-hooks-only') shouldRun = registration.scope === 'worker' || registration.auto === 'all-hooks-included';else if (autoFixtures === 'worker') shouldRun = registration.scope === 'worker';

      if (shouldRun) {
        const fixture = await this.setupFixtureForRegistration(registration, testInfo);
        if (fixture.failed) return null;
      }
    } // Install used fixtures.


    const names = fixtureParameterNames(fn, {
      file: '<unused>',
      line: 1,
      column: 1
    });
    const params = {};

    for (const name of names) {
      const registration = this.pool.registrations.get(name);
      const fixture = await this.setupFixtureForRegistration(registration, testInfo);
      if (fixture.failed) return null;
      params[name] = fixture.value;
    }

    return params;
  }

  async resolveParametersAndRunFunction(fn, testInfo, autoFixtures) {
    const params = await this.resolveParametersForFunction(fn, testInfo, autoFixtures);

    if (params === null) {
      // Do not run the function when fixture setup has already failed.
      return null;
    }

    return fn(params, testInfo);
  }

  async setupFixtureForRegistration(registration, testInfo) {
    if (registration.scope === 'test') this.testScopeClean = false;
    let fixture = this.instanceForId.get(registration.id);
    if (fixture) return fixture;
    fixture = new Fixture(this, registration);
    this.instanceForId.set(registration.id, fixture);
    await fixture.setup(testInfo);
    return fixture;
  }

  dependsOnWorkerFixturesOnly(fn, location) {
    const names = fixtureParameterNames(fn, location);

    for (const name of names) {
      const registration = this.pool.registrations.get(name);
      if (registration.scope !== 'worker') return false;
    }

    return true;
  }

}

exports.FixtureRunner = FixtureRunner;
const signatureSymbol = Symbol('signature');

function fixtureParameterNames(fn, location) {
  if (typeof fn !== 'function') return [];
  if (!fn[signatureSymbol]) fn[signatureSymbol] = innerFixtureParameterNames(fn, location);
  return fn[signatureSymbol];
}

function innerFixtureParameterNames(fn, location) {
  const text = fn.toString();
  const match = text.match(/(?:async)?(?:\s+function)?[^(]*\(([^)]*)/);
  if (!match) return [];
  const trimmedParams = match[1].trim();
  if (!trimmedParams) return [];
  const [firstParam] = splitByComma(trimmedParams);
  if (firstParam[0] !== '{' || firstParam[firstParam.length - 1] !== '}') throw errorWithLocations('First argument must use the object destructuring pattern: ' + firstParam, {
    location
  });
  const props = splitByComma(firstParam.substring(1, firstParam.length - 1)).map(prop => {
    const colon = prop.indexOf(':');
    return colon === -1 ? prop : prop.substring(0, colon).trim();
  });
  return props;
}

function splitByComma(s) {
  const result = [];
  const stack = [];
  let start = 0;

  for (let i = 0; i < s.length; i++) {
    if (s[i] === '{' || s[i] === '[') {
      stack.push(s[i] === '{' ? '}' : ']');
    } else if (s[i] === stack[stack.length - 1]) {
      stack.pop();
    } else if (!stack.length && s[i] === ',') {
      const token = s.substring(start, i).trim();
      if (token) result.push(token);
      start = i + 1;
    }
  }

  const lastToken = s.substring(start).trim();
  if (lastToken) result.push(lastToken);
  return result;
} // name + superId, fn -> id


const registrationIdMap = new Map();
let lastId = 0;

function registrationId(registration) {
  if (registration.id) return registration.id;
  const key = registration.name + '@@@' + (registration.super ? registrationId(registration.super) : '');
  let map = registrationIdMap.get(key);

  if (!map) {
    map = new Map();
    registrationIdMap.set(key, map);
  }

  if (!map.has(registration.fn)) map.set(registration.fn, String(lastId++));
  registration.id = map.get(registration.fn);
  return registration.id;
}

function errorWithLocations(message, ...defined) {
  for (const {
    name,
    location,
    quoted
  } of defined) {
    let prefix = '';
    if (name && quoted === false) prefix = name + ' ';else if (name) prefix = `"${name}" `;
    message += `\n  ${prefix}defined at ${(0, _util.formatLocation)(location)}`;
  }

  return new Error(message);
}