"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;

var _fs = _interopRequireDefault(require("fs"));

var _path = _interopRequireDefault(require("path"));

var _utils = require("playwright-core/lib/utils");

var _util = require("../util");

var _base = require("./base");

var _json = require("./json");

var _multimap = require("playwright-core/lib/utils/multimap");

var _babelBundle = require("../babelBundle");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Copyright (c) Microsoft Corporation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
class RawReporter {
  constructor() {
    this.config = void 0;
    this.suite = void 0;
    this.stepsInFile = new _multimap.MultiMap();
  }

  onBegin(config, suite) {
    this.config = config;
    this.suite = suite;
  }

  async onEnd() {
    const projectSuites = this.suite.suites;

    for (const suite of projectSuites) {
      const project = suite.project();
      (0, _utils.assert)(project, 'Internal Error: Invalid project structure');

      const reportFolder = _path.default.join(project.outputDir, 'report');

      _fs.default.mkdirSync(reportFolder, {
        recursive: true
      });

      let reportFile;

      for (let i = 0; i < 10; ++i) {
        reportFile = _path.default.join(reportFolder, (0, _util.sanitizeForFilePath)(project.name || 'project') + (i ? '-' + i : '') + '.report');

        try {
          if (_fs.default.existsSync(reportFile)) continue;
        } catch (e) {}

        break;
      }

      if (!reportFile) throw new Error('Internal error, could not create report file');
      const report = this.generateProjectReport(this.config, suite);

      _fs.default.writeFileSync(reportFile, JSON.stringify(report, undefined, 2));
    }
  }

  generateAttachments(attachments, ioStreams) {
    const out = [];

    for (const attachment of attachments) {
      if (attachment.body) {
        out.push({
          name: attachment.name,
          contentType: attachment.contentType,
          body: attachment.body
        });
      } else if (attachment.path) {
        out.push({
          name: attachment.name,
          contentType: attachment.contentType,
          path: attachment.path
        });
      }
    }

    if (ioStreams) {
      for (const chunk of ioStreams.stdout) out.push(this._stdioAttachment(chunk, 'stdout'));

      for (const chunk of ioStreams.stderr) out.push(this._stdioAttachment(chunk, 'stderr'));
    }

    return out;
  }

  generateProjectReport(config, suite) {
    this.config = config;
    const project = suite.project();
    (0, _utils.assert)(project, 'Internal Error: Invalid project structure');
    const report = {
      config: filterOutPrivateFields(config),
      project: {
        metadata: project.metadata,
        name: project.name,
        outputDir: project.outputDir,
        repeatEach: project.repeatEach,
        retries: project.retries,
        testDir: project.testDir,
        testIgnore: (0, _json.serializePatterns)(project.testIgnore),
        testMatch: (0, _json.serializePatterns)(project.testMatch),
        timeout: project.timeout
      },
      suites: suite.suites.map(fileSuite => {
        return this._serializeSuite(fileSuite);
      })
    };

    for (const file of this.stepsInFile.keys()) {
      let source;

      try {
        source = _fs.default.readFileSync(file, 'utf-8') + '\n//';
      } catch (e) {
        continue;
      }

      const lines = source.split('\n').length;
      const highlighted = (0, _babelBundle.codeFrameColumns)(source, {
        start: {
          line: lines,
          column: 1
        }
      }, {
        highlightCode: true,
        linesAbove: lines,
        linesBelow: 0
      });
      const highlightedLines = highlighted.split('\n');
      const lineWithArrow = highlightedLines[highlightedLines.length - 1];

      for (const step of this.stepsInFile.get(file)) {
        // Don't bother with snippets that have less than 3 lines.
        if (step.location.line < 2 || step.location.line >= lines) continue; // Cut out snippet.

        const snippetLines = highlightedLines.slice(step.location.line - 2, step.location.line + 1); // Relocate arrow.

        const index = lineWithArrow.indexOf('^');
        const shiftedArrow = lineWithArrow.slice(0, index) + ' '.repeat(step.location.column - 1) + lineWithArrow.slice(index); // Insert arrow line.

        snippetLines.splice(2, 0, shiftedArrow);
        step.snippet = snippetLines.join('\n');
      }
    }

    return report;
  }

  _serializeSuite(suite) {
    const location = this._relativeLocation(suite.location);

    const result = {
      title: suite.title,
      fileId: suite._fileId,
      location,
      suites: suite.suites.map(s => this._serializeSuite(s)),
      tests: suite.tests.map(t => this._serializeTest(t))
    };
    return result;
  }

  _serializeTest(test) {
    return {
      testId: test.id,
      title: test.title,
      location: this._relativeLocation(test.location),
      expectedStatus: test.expectedStatus,
      timeout: test.timeout,
      annotations: test.annotations,
      retries: test.retries,
      ok: test.ok(),
      outcome: test.outcome(),
      results: test.results.map(r => this._serializeResult(test, r))
    };
  }

  _serializeResult(test, result) {
    return {
      retry: result.retry,
      workerIndex: result.workerIndex,
      startTime: result.startTime.toISOString(),
      duration: result.duration,
      status: result.status,
      errors: (0, _base.formatResultFailure)(this.config, test, result, '', true).map(error => error.message),
      attachments: this.generateAttachments(result.attachments, result),
      steps: dedupeSteps(result.steps.map(step => this._serializeStep(test, step)))
    };
  }

  _serializeStep(test, step) {
    var _step$error;

    const result = {
      title: step.title,
      category: step.category,
      startTime: step.startTime.toISOString(),
      duration: step.duration,
      error: (_step$error = step.error) === null || _step$error === void 0 ? void 0 : _step$error.message,
      location: this._relativeLocation(step.location),
      steps: dedupeSteps(step.steps.map(step => this._serializeStep(test, step))),
      count: 1
    };
    if (step.location) this.stepsInFile.set(step.location.file, result);
    return result;
  }

  _stdioAttachment(chunk, type) {
    if (typeof chunk === 'string') {
      return {
        name: type,
        contentType: 'text/plain',
        body: chunk
      };
    }

    return {
      name: type,
      contentType: 'application/octet-stream',
      body: chunk
    };
  }

  _relativeLocation(location) {
    if (!location) return undefined;
    const file = (0, _json.toPosixPath)(_path.default.relative(this.config.rootDir, location.file));
    return {
      file,
      line: location.line,
      column: location.column
    };
  }

}

function dedupeSteps(steps) {
  const result = [];
  let lastStep;

  for (const step of steps) {
    var _step$location, _step$location2, _lastStep$location, _step$location3, _lastStep$location2, _step$location4, _lastStep$location3;

    const canDedupe = !step.error && step.duration >= 0 && ((_step$location = step.location) === null || _step$location === void 0 ? void 0 : _step$location.file) && !step.steps.length;

    if (canDedupe && lastStep && step.category === lastStep.category && step.title === lastStep.title && ((_step$location2 = step.location) === null || _step$location2 === void 0 ? void 0 : _step$location2.file) === ((_lastStep$location = lastStep.location) === null || _lastStep$location === void 0 ? void 0 : _lastStep$location.file) && ((_step$location3 = step.location) === null || _step$location3 === void 0 ? void 0 : _step$location3.line) === ((_lastStep$location2 = lastStep.location) === null || _lastStep$location2 === void 0 ? void 0 : _lastStep$location2.line) && ((_step$location4 = step.location) === null || _step$location4 === void 0 ? void 0 : _step$location4.column) === ((_lastStep$location3 = lastStep.location) === null || _lastStep$location3 === void 0 ? void 0 : _lastStep$location3.column)) {
      ++lastStep.count;
      lastStep.duration += step.duration;
      continue;
    }

    result.push(step);
    lastStep = canDedupe ? step : undefined;
  }

  return result;
}

function filterOutPrivateFields(object) {
  if (!object || typeof object !== 'object') return object;
  if (Array.isArray(object)) return object.map(filterOutPrivateFields);
  return Object.fromEntries(Object.entries(object).filter(entry => !entry[0].startsWith('_')).map(entry => [entry[0], filterOutPrivateFields(entry[1])]));
}

var _default = RawReporter;
exports.default = _default;