"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.belongsToNodeModules = belongsToNodeModules;
exports.installTransform = installTransform;
exports.resolveHook = resolveHook;
exports.transformHook = transformHook;
exports.wrapFunctionWithLocation = wrapFunctionWithLocation;

var _crypto = _interopRequireDefault(require("crypto"));

var _os = _interopRequireDefault(require("os"));

var _path = _interopRequireDefault(require("path"));

var _fs = _interopRequireDefault(require("fs"));

var _utilsBundle = require("./utilsBundle");

var _url = _interopRequireDefault(require("url"));

var _tsconfigLoader = require("./third_party/tsconfig-loader");

var _module = _interopRequireDefault(require("module"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/**
 * Copyright (c) Microsoft Corporation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
const version = 13;

const cacheDir = process.env.PWTEST_CACHE_DIR || _path.default.join(_os.default.tmpdir(), 'playwright-transform-cache');

const sourceMaps = new Map();
const cachedTSConfigs = new Map();
const kStackTraceLimit = 15;
Error.stackTraceLimit = kStackTraceLimit;

_utilsBundle.sourceMapSupport.install({
  environment: 'node',
  handleUncaughtExceptions: false,

  retrieveSourceMap(source) {
    if (!sourceMaps.has(source)) return null;
    const sourceMapPath = sourceMaps.get(source);
    if (!_fs.default.existsSync(sourceMapPath)) return null;
    return {
      map: JSON.parse(_fs.default.readFileSync(sourceMapPath, 'utf-8')),
      url: source
    };
  }

});

function calculateCachePath(content, filePath, isModule) {
  const hash = _crypto.default.createHash('sha1').update(process.env.PW_TEST_SOURCE_TRANSFORM || '').update(isModule ? 'esm' : 'no_esm').update(content).update(filePath).update(String(version)).digest('hex');

  const fileName = _path.default.basename(filePath, _path.default.extname(filePath)).replace(/\W/g, '') + '_' + hash;
  return _path.default.join(cacheDir, hash[0] + hash[1], fileName);
}

function validateTsConfig(tsconfig) {
  if (!tsconfig.tsConfigPath || !tsconfig.baseUrl) return; // Make 'baseUrl' absolute, because it is relative to the tsconfig.json, not to cwd.

  const absoluteBaseUrl = _path.default.resolve(_path.default.dirname(tsconfig.tsConfigPath), tsconfig.baseUrl);

  const paths = tsconfig.paths || {
    '*': ['*']
  };
  return {
    absoluteBaseUrl,
    paths: Object.entries(paths).map(([key, values]) => ({
      key,
      values
    }))
  };
}

function loadAndValidateTsconfigForFile(file) {
  const cwd = _path.default.dirname(file);

  if (!cachedTSConfigs.has(cwd)) {
    const loaded = (0, _tsconfigLoader.tsConfigLoader)({
      getEnv: name => process.env[name],
      cwd
    });
    cachedTSConfigs.set(cwd, validateTsConfig(loaded));
  }

  return cachedTSConfigs.get(cwd);
}

const pathSeparator = process.platform === 'win32' ? ';' : ':';
const scriptPreprocessor = process.env.PW_TEST_SOURCE_TRANSFORM ? require(process.env.PW_TEST_SOURCE_TRANSFORM) : undefined;
const builtins = new Set(_module.default.builtinModules);

function resolveHook(filename, specifier) {
  if (builtins.has(specifier)) return;
  const isTypeScript = filename.endsWith('.ts') || filename.endsWith('.tsx');
  if (!isTypeScript) return;
  const tsconfig = loadAndValidateTsconfigForFile(filename);

  if (tsconfig && !isRelativeSpecifier(specifier)) {
    let longestPrefixLength = -1;
    let pathMatchedByLongestPrefix;

    for (const {
      key,
      values
    } of tsconfig.paths) {
      let matchedPartOfSpecifier = specifier;
      const [keyPrefix, keySuffix] = key.split('*');

      if (key.includes('*')) {
        // * If pattern contains '*' then to match pattern "<prefix>*<suffix>" module name must start with the <prefix> and end with <suffix>.
        // * <MatchedStar> denotes part of the module name between <prefix> and <suffix>.
        // * If module name can be matches with multiple patterns then pattern with the longest prefix will be picked.
        // https://github.com/microsoft/TypeScript/blob/f82d0cb3299c04093e3835bc7e29f5b40475f586/src/compiler/moduleNameResolver.ts#L1049
        if (keyPrefix) {
          if (!specifier.startsWith(keyPrefix)) continue;
          matchedPartOfSpecifier = matchedPartOfSpecifier.substring(keyPrefix.length, matchedPartOfSpecifier.length);
        }

        if (keySuffix) {
          if (!specifier.endsWith(keySuffix)) continue;
          matchedPartOfSpecifier = matchedPartOfSpecifier.substring(0, matchedPartOfSpecifier.length - keySuffix.length);
        }
      } else {
        if (specifier !== key) continue;
        matchedPartOfSpecifier = specifier;
      }

      for (const value of values) {
        let candidate = value;
        if (value.includes('*')) candidate = candidate.replace('*', matchedPartOfSpecifier);
        candidate = _path.default.resolve(tsconfig.absoluteBaseUrl, candidate.replace(/\//g, _path.default.sep));

        for (const ext of ['', '.js', '.ts', '.mjs', '.cjs', '.jsx', '.tsx', '.cjs', '.mts', '.cts']) {
          if (_fs.default.existsSync(candidate + ext)) {
            if (keyPrefix.length > longestPrefixLength) {
              longestPrefixLength = keyPrefix.length;
              pathMatchedByLongestPrefix = candidate;
            }
          }
        }
      }
    }

    if (pathMatchedByLongestPrefix) return pathMatchedByLongestPrefix;
  }

  if (specifier.endsWith('.js')) {
    const resolved = _path.default.resolve(_path.default.dirname(filename), specifier);

    if (resolved.endsWith('.js')) {
      const tsResolved = resolved.substring(0, resolved.length - 3) + '.ts';
      if (!_fs.default.existsSync(resolved) && _fs.default.existsSync(tsResolved)) return tsResolved;
    }
  }
}

function transformHook(code, filename, moduleUrl) {
  // If we are not TypeScript and there is no applicable preprocessor - bail out.
  const isModule = !!moduleUrl;
  const isTypeScript = filename.endsWith('.ts') || filename.endsWith('.tsx');
  const hasPreprocessor = process.env.PW_TEST_SOURCE_TRANSFORM && process.env.PW_TEST_SOURCE_TRANSFORM_SCOPE && process.env.PW_TEST_SOURCE_TRANSFORM_SCOPE.split(pathSeparator).some(f => filename.startsWith(f));
  const cachePath = calculateCachePath(code, filename, isModule);
  const codePath = cachePath + '.js';
  const sourceMapPath = cachePath + '.map';
  sourceMaps.set(moduleUrl || filename, sourceMapPath);
  if (!process.env.PW_IGNORE_COMPILE_CACHE && _fs.default.existsSync(codePath)) return _fs.default.readFileSync(codePath, 'utf8'); // We don't use any browserslist data, but babel checks it anyway.
  // Silence the annoying warning.

  process.env.BROWSERSLIST_IGNORE_OLD_DATA = 'true';

  try {
    const {
      babelTransform
    } = require('./babelBundle');

    const result = babelTransform(filename, isTypeScript, isModule, hasPreprocessor ? scriptPreprocessor : undefined, [require.resolve('./tsxTransform')]);

    if (result.code) {
      _fs.default.mkdirSync(_path.default.dirname(cachePath), {
        recursive: true
      });

      if (result.map) _fs.default.writeFileSync(sourceMapPath, JSON.stringify(result.map), 'utf8');

      _fs.default.writeFileSync(codePath, result.code, 'utf8');
    }

    return result.code || '';
  } catch (e) {
    // Re-throw error with a playwright-test stack
    // that could be filtered out.
    throw new Error(e.message);
  }
}

function installTransform() {
  let reverted = false;
  const originalResolveFilename = _module.default._resolveFilename;

  function resolveFilename(specifier, parent, ...rest) {
    if (!reverted && parent) {
      const resolved = resolveHook(parent.filename, specifier);
      if (resolved !== undefined) specifier = resolved;
    }

    return originalResolveFilename.call(this, specifier, parent, ...rest);
  }

  _module.default._resolveFilename = resolveFilename;

  const revertPirates = _utilsBundle.pirates.addHook((code, filename) => {
    if (belongsToNodeModules(filename)) return code;
    return transformHook(code, filename);
  }, {
    exts: ['.ts', '.tsx', '.js', '.jsx', '.mjs']
  });

  return () => {
    reverted = true;
    _module.default._resolveFilename = originalResolveFilename;
    revertPirates();
  };
}

function wrapFunctionWithLocation(func) {
  return (...args) => {
    const oldPrepareStackTrace = Error.prepareStackTrace;

    Error.prepareStackTrace = (error, stackFrames) => {
      const frame = _utilsBundle.sourceMapSupport.wrapCallSite(stackFrames[1]);

      const fileName = frame.getFileName(); // Node error stacks for modules use file:// urls instead of paths.

      const file = fileName && fileName.startsWith('file://') ? _url.default.fileURLToPath(fileName) : fileName;
      return {
        file,
        line: frame.getLineNumber(),
        column: frame.getColumnNumber()
      };
    };

    Error.stackTraceLimit = 2;
    const obj = {};
    Error.captureStackTrace(obj);
    const location = obj.stack;
    Error.stackTraceLimit = kStackTraceLimit;
    Error.prepareStackTrace = oldPrepareStackTrace;
    return func(location, ...args);
  };
} // This will catch the playwright-test package as well


const kPlaywrightInternalPrefix = _path.default.resolve(__dirname, '../../playwright');

const kPlaywrightCoveragePrefix = _path.default.resolve(__dirname, '../../../tests/config/coverage.js');

function belongsToNodeModules(file) {
  if (file.includes(`${_path.default.sep}node_modules${_path.default.sep}`)) return true;
  if (file.startsWith(kPlaywrightInternalPrefix)) return true;
  if (file.startsWith(kPlaywrightCoveragePrefix)) return true;
  return false;
}

function isRelativeSpecifier(specifier) {
  return specifier === '.' || specifier === '..' || specifier.startsWith('./') || specifier.startsWith('../');
}