// This file is generated by /utils/generate_types/index.js
/**
 * Copyright (c) Microsoft Corporation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import type { APIRequestContext, Browser, BrowserContext, BrowserContextOptions, Page, LaunchOptions, ViewportSize, Geolocation, HTTPCredentials, Locator, APIResponse } from 'playwright-core';
export * from 'playwright-core';

export type ReporterDescription =
  ['dot'] |
  ['line'] |
  ['list'] |
  ['github'] |
  ['junit'] | ['junit', { outputFile?: string, stripANSIControlSequences?: boolean }] |
  ['json'] | ['json', { outputFile?: string }] |
  ['html'] | ['html', { outputFolder?: string, open?: 'always' | 'never' | 'on-failure' }] |
  ['null'] |
  [string] | [string, any];

type UseOptions<TestArgs, WorkerArgs> = { [K in keyof WorkerArgs]?: WorkerArgs[K] } & { [K in keyof TestArgs]?: TestArgs[K] };

/**
 * Playwright Test supports running multiple test projects at the same time. This is useful for running tests in multiple
 * configurations. For example, consider running tests against multiple browsers.
 *
 * `TestProject` encapsulates configuration specific to a single project. Projects are configured in
 * [testConfig.projects](https://playwright.dev/docs/api/class-testconfig#test-config-projects) specified in the
 * [configuration file](https://playwright.dev/docs/test-configuration). Note that all properties of [TestProject] are available in the top-level
 * [TestConfig], in which case they are shared between all projects.
 *
 * Here is an example configuration that runs every test in Chromium, Firefox and WebKit, both Desktop and Mobile versions.
 *
 * ```js
 * // playwright.config.ts
 * import { type PlaywrightTestConfig, devices } from '@playwright/test';
 *
 * const config: PlaywrightTestConfig = {
 *   // Options shared for all projects.
 *   timeout: 30000,
 *   use: {
 *     ignoreHTTPSErrors: true,
 *   },
 *
 *   // Options specific to each project.
 *   projects: [
 *     {
 *       name: 'Desktop Chromium',
 *       use: {
 *         browserName: 'chromium',
 *         viewport: { width: 1280, height: 720 },
 *       },
 *     },
 *     {
 *       name: 'Desktop Safari',
 *       use: {
 *         browserName: 'webkit',
 *         viewport: { width: 1280, height: 720 },
 *       }
 *     },
 *     {
 *       name: 'Desktop Firefox',
 *       use: {
 *         browserName: 'firefox',
 *         viewport: { width: 1280, height: 720 },
 *       }
 *     },
 *     {
 *       name: 'Mobile Chrome',
 *       use: devices['Pixel 5'],
 *     },
 *     {
 *       name: 'Mobile Safari',
 *       use: devices['iPhone 12'],
 *     },
 *   ],
 * };
 * export default config;
 * ```
 *
 */
export interface Project<TestArgs = {}, WorkerArgs = {}> extends TestProject {
  /**
   * Options for all tests in this project, for example
   * [testOptions.browserName](https://playwright.dev/docs/api/class-testoptions#test-options-browser-name). Learn more about
   * [configuration](https://playwright.dev/docs/test-configuration) and see [available options][TestOptions].
   *
   * ```js
   * // playwright.config.ts
   * import type { PlaywrightTestConfig } from '@playwright/test';
   *
   * const config: PlaywrightTestConfig = {
   *   projects: [
   *     {
   *       name: 'Chromium',
   *       use: {
   *         browserName: 'chromium',
   *       },
   *     },
   *   ],
   * };
   * export default config;
   * ```
   *
   * Use [testConfig.use](https://playwright.dev/docs/api/class-testconfig#test-config-use) to change this option for all
   * projects.
   */
  use?: UseOptions<TestArgs, WorkerArgs>;
}

/**
 * Playwright Test supports running multiple test projects at the same time. This is useful for running tests in multiple
 * configurations. For example, consider running tests against multiple browsers.
 *
 * `TestProject` encapsulates configuration specific to a single project. Projects are configured in
 * [testConfig.projects](https://playwright.dev/docs/api/class-testconfig#test-config-projects) specified in the
 * [configuration file](https://playwright.dev/docs/test-configuration). Note that all properties of [TestProject] are available in the top-level
 * [TestConfig], in which case they are shared between all projects.
 *
 * Here is an example configuration that runs every test in Chromium, Firefox and WebKit, both Desktop and Mobile versions.
 *
 * ```js
 * // playwright.config.ts
 * import { type PlaywrightTestConfig, devices } from '@playwright/test';
 *
 * const config: PlaywrightTestConfig = {
 *   // Options shared for all projects.
 *   timeout: 30000,
 *   use: {
 *     ignoreHTTPSErrors: true,
 *   },
 *
 *   // Options specific to each project.
 *   projects: [
 *     {
 *       name: 'Desktop Chromium',
 *       use: {
 *         browserName: 'chromium',
 *         viewport: { width: 1280, height: 720 },
 *       },
 *     },
 *     {
 *       name: 'Desktop Safari',
 *       use: {
 *         browserName: 'webkit',
 *         viewport: { width: 1280, height: 720 },
 *       }
 *     },
 *     {
 *       name: 'Desktop Firefox',
 *       use: {
 *         browserName: 'firefox',
 *         viewport: { width: 1280, height: 720 },
 *       }
 *     },
 *     {
 *       name: 'Mobile Chrome',
 *       use: devices['Pixel 5'],
 *     },
 *     {
 *       name: 'Mobile Safari',
 *       use: devices['iPhone 12'],
 *     },
 *   ],
 * };
 * export default config;
 * ```
 *
 */
export interface FullProject<TestArgs = {}, WorkerArgs = {}> {
  /**
   * Filter to only run tests with a title matching one of the patterns. For example, passing `grep: /cart/` should only run
   * tests with "cart" in the title. Also available globally and in the [command line](https://playwright.dev/docs/test-cli) with the `-g` option.
   *
   * `grep` option is also useful for [tagging tests](https://playwright.dev/docs/test-annotations#tag-tests).
   */
  grep: RegExp | RegExp[];
  /**
   * Filter to only run tests with a title **not** matching one of the patterns. This is the opposite of
   * [testProject.grep](https://playwright.dev/docs/api/class-testproject#test-project-grep). Also available globally and in
   * the [command line](https://playwright.dev/docs/test-cli) with the `--grep-invert` option.
   *
   * `grepInvert` option is also useful for [tagging tests](https://playwright.dev/docs/test-annotations#tag-tests).
   */
  grepInvert: RegExp | RegExp[] | null;
  /**
   * Metadata that will be put directly to the test report serialized as JSON.
   */
  metadata: Metadata;
  /**
   * Project name is visible in the report and during test execution.
   */
  name: string;
  /**
   * The base directory, relative to the config file, for snapshot files created with `toMatchSnapshot`. Defaults to
   * [testProject.testDir](https://playwright.dev/docs/api/class-testproject#test-project-test-dir).
   *
   * The directory for each test can be accessed by
   * [testInfo.snapshotDir](https://playwright.dev/docs/api/class-testinfo#test-info-snapshot-dir) and
   * [testInfo.snapshotPath(...pathSegments)](https://playwright.dev/docs/api/class-testinfo#test-info-snapshot-path).
   *
   * This path will serve as the base directory for each test file snapshot directory. Setting `snapshotDir` to
   * `'snapshots'`, the [testInfo.snapshotDir](https://playwright.dev/docs/api/class-testinfo#test-info-snapshot-dir) would
   * resolve to `snapshots/a.spec.js-snapshots`.
   */
  snapshotDir: string;
  /**
   * The output directory for files created during test execution. Defaults to `<package.json-directory>/test-results`.
   *
   * This directory is cleaned at the start. When running a test, a unique subdirectory inside the
   * [testProject.outputDir](https://playwright.dev/docs/api/class-testproject#test-project-output-dir) is created,
   * guaranteeing that test running in parallel do not conflict. This directory can be accessed by
   * [testInfo.outputDir](https://playwright.dev/docs/api/class-testinfo#test-info-output-dir) and
   * [testInfo.outputPath(...pathSegments)](https://playwright.dev/docs/api/class-testinfo#test-info-output-path).
   *
   * Here is an example that uses
   * [testInfo.outputPath(...pathSegments)](https://playwright.dev/docs/api/class-testinfo#test-info-output-path) to create a
   * temporary file.
   *
   * ```js
   * import { test, expect } from '@playwright/test';
   * import fs from 'fs';
   *
   * test('example test', async ({}, testInfo) => {
   *   const file = testInfo.outputPath('temporary-file.txt');
   *   await fs.promises.writeFile(file, 'Put some data to the file', 'utf8');
   * });
   * ```
   *
   * Use [testConfig.outputDir](https://playwright.dev/docs/api/class-testconfig#test-config-output-dir) to change this
   * option for all projects.
   */
  outputDir: string;
  /**
   * The number of times to repeat each test, useful for debugging flaky tests.
   *
   * Use [testConfig.repeatEach](https://playwright.dev/docs/api/class-testconfig#test-config-repeat-each) to change this
   * option for all projects.
   */
  repeatEach: number;
  /**
   * The maximum number of retry attempts given to failed tests. Learn more about [test retries](https://playwright.dev/docs/test-retries#retries).
   *
   * Use [testConfig.retries](https://playwright.dev/docs/api/class-testconfig#test-config-retries) to change this option for
   * all projects.
   */
  retries: number;
  /**
   * Directory that will be recursively scanned for test files. Defaults to the directory of the configuration file.
   *
   * Each project can use a different directory. Here is an example that runs smoke tests in three browsers and all other
   * tests in stable Chrome browser.
   *
   * ```js
   * // playwright.config.ts
   * import type { PlaywrightTestConfig } from '@playwright/test';
   *
   * const config: PlaywrightTestConfig = {
   *   projects: [
   *     {
   *       name: 'Smoke Chromium',
   *       testDir: './smoke-tests',
   *       use: {
   *         browserName: 'chromium',
   *       }
   *     },
   *     {
   *       name: 'Smoke WebKit',
   *       testDir: './smoke-tests',
   *       use: {
   *         browserName: 'webkit',
   *       }
   *     },
   *     {
   *       name: 'Smoke Firefox',
   *       testDir: './smoke-tests',
   *       use: {
   *         browserName: 'firefox',
   *       }
   *     },
   *     {
   *       name: 'Chrome Stable',
   *       testDir: './',
   *       use: {
   *         browserName: 'chromium',
   *         channel: 'chrome',
   *       }
   *     },
   *   ],
   * };
   * export default config;
   * ```
   *
   * Use [testConfig.testDir](https://playwright.dev/docs/api/class-testconfig#test-config-test-dir) to change this option
   * for all projects.
   */
  testDir: string;
  /**
   * Files matching one of these patterns are not executed as test files. Matching is performed against the absolute file
   * path. Strings are treated as glob patterns.
   *
   * For example, `'**\/test-assets/**'` will ignore any files in the `test-assets` directory.
   *
   * Use [testConfig.testIgnore](https://playwright.dev/docs/api/class-testconfig#test-config-test-ignore) to change this
   * option for all projects.
   */
  testIgnore: string | RegExp | (string | RegExp)[];
  /**
   * Only the files matching one of these patterns are executed as test files. Matching is performed against the absolute
   * file path. Strings are treated as glob patterns.
   *
   * By default, Playwright Test looks for files matching `.*(test|spec)\.(js|ts|mjs)`.
   *
   * Use [testConfig.testMatch](https://playwright.dev/docs/api/class-testconfig#test-config-test-match) to change this
   * option for all projects.
   */
  testMatch: string | RegExp | (string | RegExp)[];
  /**
   * Timeout for each test in milliseconds. Defaults to 30 seconds.
   *
   * This is a base timeout for all tests. In addition, each test can configure its own timeout with
   * [test.setTimeout(timeout)](https://playwright.dev/docs/api/class-test#test-set-timeout).
   *
   * Use [testConfig.timeout](https://playwright.dev/docs/api/class-testconfig#test-config-timeout) to change this option for
   * all projects.
   */
  timeout: number;
  /**
   * Options for all tests in this project, for example
   * [testOptions.browserName](https://playwright.dev/docs/api/class-testoptions#test-options-browser-name). Learn more about
   * [configuration](https://playwright.dev/docs/test-configuration) and see [available options][TestOptions].
   *
   * ```js
   * // playwright.config.ts
   * import type { PlaywrightTestConfig } from '@playwright/test';
   *
   * const config: PlaywrightTestConfig = {
   *   projects: [
   *     {
   *       name: 'Chromium',
   *       use: {
   *         browserName: 'chromium',
   *       },
   *     },
   *   ],
   * };
   * export default config;
   * ```
   *
   * Use [testConfig.use](https://playwright.dev/docs/api/class-testconfig#test-config-use) to change this option for all
   * projects.
   */
  use: UseOptions<PlaywrightTestOptions & TestArgs, PlaywrightWorkerOptions & WorkerArgs>;
}

type LiteralUnion<T extends U, U = string> = T | (U & { zz_IGNORE_ME?: never });

/**
 * Playwright Test provides many options to configure how your tests are collected and executed, for example `timeout` or
 * `testDir`. These options are described in the [TestConfig] object in the [configuration file](https://playwright.dev/docs/test-configuration).
 *
 * Playwright Test supports running multiple test projects at the same time. Project-specific options should be put to
 * [testConfig.projects](https://playwright.dev/docs/api/class-testconfig#test-config-projects), but top-level [TestConfig]
 * can also define base options shared between all projects.
 *
 * ```js
 * // playwright.config.ts
 * import type { PlaywrightTestConfig } from '@playwright/test';
 *
 * const config: PlaywrightTestConfig = {
 *   timeout: 30000,
 *   globalTimeout: 600000,
 *   reporter: 'list',
 *   testDir: './tests',
 * };
 * export default config;
 * ```
 *
 */
interface TestConfig {
  /**
   * The list of reporters to use. Each reporter can be:
   * - A builtin reporter name like `'list'` or `'json'`.
   * - A module name like `'my-awesome-reporter'`.
   * - A relative path to the reporter like `'./reporters/my-awesome-reporter.js'`.
   *
   * You can pass options to the reporter in a tuple like `['json', { outputFile: './report.json' }]`.
   *
   * Learn more in the [reporters guide](https://playwright.dev/docs/test-reporters).
   *
   * ```js
   * // playwright.config.ts
   * import type { PlaywrightTestConfig } from '@playwright/test';
   *
   * const config: PlaywrightTestConfig = {
   *   reporter: 'line',
   * };
   * export default config;
   * ```
   *
   */
  reporter?: LiteralUnion<'list'|'dot'|'line'|'github'|'json'|'junit'|'null'|'html', string> | ReporterDescription[];
  /**
   * Launch a development web server (or multiple) during the tests.
   *
   * If the port is specified, Playwright Test will wait for it to be available on `127.0.0.1` or `::1`, before running the
   * tests. If the url is specified, Playwright Test will wait for the URL to return a 2xx, 3xx, 400, 401, 402, or 403 status
   * code before running the tests.
   *
   * For continuous integration, you may want to use the `reuseExistingServer: !process.env.CI` option which does not use an
   * existing server on the CI. To see the stdout, you can set the `DEBUG=pw:webserver` environment variable.
   *
   * The `port` (but not the `url`) gets passed over to Playwright as a
   * [testOptions.baseURL](https://playwright.dev/docs/api/class-testoptions#test-options-base-url). For example port `8080`
   * produces `baseURL` equal `http://localhost:8080`. If `webServer` is specified as an array, you must explicitly configure
   * the `baseURL` (even if it only has one entry).
   *
   * > NOTE: It is also recommended to specify
   * [testOptions.baseURL](https://playwright.dev/docs/api/class-testoptions#test-options-base-url) in the config, so that
   * tests could use relative urls.
   *
   * ```js
   * // playwright.config.ts
   * import type { PlaywrightTestConfig } from '@playwright/test';
   * const config: PlaywrightTestConfig = {
   *   webServer: {
   *     command: 'npm run start',
   *     port: 3000,
   *     timeout: 120 * 1000,
   *     reuseExistingServer: !process.env.CI,
   *   },
   *   use: {
   *     baseURL: 'http://localhost:3000/',
   *   },
   * };
   * export default config;
   * ```
   *
   * Now you can use a relative path when navigating the page:
   *
   * ```js
   * // test.spec.ts
   * import { test } from '@playwright/test';
   *
   * test('test', async ({ page }) => {
   *   // This will result in http://localhost:3000/foo
   *   await page.goto('/foo');
   * });
   * ```
   *
   * Multiple web servers (or background processes) can be launched:
   *
   * ```js
   * // playwright.config.ts
   * import type { PlaywrightTestConfig } from '@playwright/test';
   * const config: PlaywrightTestConfig = {
   *   webServer: [
   *     {
   *       command: 'npm run start',
   *       port: 3000,
   *       timeout: 120 * 1000,
   *       reuseExistingServer: !process.env.CI,
   *     },
   *     {
   *       command: 'npm run backend',
   *       port: 3333,
   *       timeout: 120 * 1000,
   *       reuseExistingServer: !process.env.CI,
   *     }
   *   ],
   *   use: {
   *     baseURL: 'http://localhost:3000/',
   *   },
   * };
   * export default config;
   * ```
   *
   */
  webServer?: TestConfigWebServer | TestConfigWebServer[];
  /**
   * Configuration for the `expect` assertion library. Learn more about [various timeouts](https://playwright.dev/docs/test-timeouts).
   *
   * ```js
   * // playwright.config.ts
   * import type { PlaywrightTestConfig } from '@playwright/test';
   *
   * const config: PlaywrightTestConfig = {
   *   expect: {
   *     timeout: 10000,
   *     toMatchSnapshot: {
   *       maxDiffPixels: 10,
   *     },
   *   },
   * };
   * export default config;
   * ```
   *
   */
  expect?: {
    /**
     * Default timeout for async expect matchers in milliseconds, defaults to 5000ms.
     */
    timeout?: number;

    /**
     * Configuration for the
     * [pageAssertions.toHaveScreenshot(name[, options])](https://playwright.dev/docs/api/class-pageassertions#page-assertions-to-have-screenshot-1)
     * method.
     */
    toHaveScreenshot?: {
      /**
       * an acceptable perceived color difference in the [YIQ color space](https://en.wikipedia.org/wiki/YIQ) between the same
       * pixel in compared images, between zero (strict) and one (lax). Defaults to `0.2`.
       */
      threshold?: number;

      /**
       * an acceptable amount of pixels that could be different, unset by default.
       */
      maxDiffPixels?: number;

      /**
       * an acceptable ratio of pixels that are different to the total amount of pixels, between `0` and `1` , unset by default.
       */
      maxDiffPixelRatio?: number;

      /**
       * See `animations` in [page.screenshot([options])](https://playwright.dev/docs/api/class-page#page-screenshot). Defaults
       * to `"disabled"`.
       */
      animations?: "allow"|"disabled";

      /**
       * See `caret` in [page.screenshot([options])](https://playwright.dev/docs/api/class-page#page-screenshot). Defaults to
       * `"hide"`.
       */
      caret?: "hide"|"initial";

      /**
       * See `scale` in [page.screenshot([options])](https://playwright.dev/docs/api/class-page#page-screenshot). Defaults to
       * `"css"`.
       */
      scale?: "css"|"device";
    };

    /**
     * Configuration for the
     * [screenshotAssertions.toMatchSnapshot(name[, options])](https://playwright.dev/docs/api/class-screenshotassertions#screenshot-assertions-to-match-snapshot-1)
     * method.
     */
    toMatchSnapshot?: {
      /**
       * an acceptable perceived color difference in the [YIQ color space](https://en.wikipedia.org/wiki/YIQ) between the same
       * pixel in compared images, between zero (strict) and one (lax). Defaults to `0.2`.
       */
      threshold?: number;

      /**
       * an acceptable amount of pixels that could be different, unset by default.
       */
      maxDiffPixels?: number;

      /**
       * an acceptable ratio of pixels that are different to the total amount of pixels, between `0` and `1` , unset by default.
       */
      maxDiffPixelRatio?: number;
    };
  };

  /**
   * Whether to exit with an error if any tests or groups are marked as
   * [test.only(title, testFunction)](https://playwright.dev/docs/api/class-test#test-only) or
   * [test.describe.only(title, callback)](https://playwright.dev/docs/api/class-test#test-describe-only). Useful on CI.
   *
   * ```js
   * // playwright.config.ts
   * import type { PlaywrightTestConfig } from '@playwright/test';
   *
   * const config: PlaywrightTestConfig = {
   *   forbidOnly: !!process.env.CI,
   * };
   * export default config;
   * ```
   *
   */
  forbidOnly?: boolean;

  /**
   * Playwright Test runs tests in parallel. In order to achieve that, it runs several worker processes that run at the same
   * time. By default, **test files** are run in parallel. Tests in a single file are run in order, in the same worker
   * process.
   *
   * You can configure entire test run to concurrently execute all tests in all files using this option.
   */
  fullyParallel?: boolean;

  /**
   * Path to the global setup file. This file will be required and run before all the tests. It must export a single function
   * that takes a [`TestConfig`] argument.
   *
   * Learn more about [global setup and teardown](https://playwright.dev/docs/test-advanced#global-setup-and-teardown).
   *
   * ```js
   * // playwright.config.ts
   * import { type PlaywrightTestConfig } from '@playwright/test';
   *
   * const config: PlaywrightTestConfig = {
   *   globalSetup: './global-setup',
   * };
   * export default config;
   * ```
   *
   */
  globalSetup?: string;

  /**
   * Path to the global teardown file. This file will be required and run after all the tests. It must export a single
   * function. See also [testConfig.globalSetup](https://playwright.dev/docs/api/class-testconfig#test-config-global-setup).
   *
   * Learn more about [global setup and teardown](https://playwright.dev/docs/test-advanced#global-setup-and-teardown).
   *
   * ```js
   * // playwright.config.ts
   * import { type PlaywrightTestConfig } from '@playwright/test';
   *
   * const config: PlaywrightTestConfig = {
   *   globalTeardown: './global-teardown',
   * };
   * export default config;
   * ```
   *
   */
  globalTeardown?: string;

  /**
   * Maximum time in milliseconds the whole test suite can run. Zero timeout (default) disables this behavior. Useful on CI
   * to prevent broken setup from running too long and wasting resources. Learn more about
   * [various timeouts](https://playwright.dev/docs/test-timeouts).
   *
   * ```js
   * // playwright.config.ts
   * import type { PlaywrightTestConfig } from '@playwright/test';
   *
   * const config: PlaywrightTestConfig = {
   *   globalTimeout: process.env.CI ? 60 * 60 * 1000 : undefined,
   * };
   * export default config;
   * ```
   *
   */
  globalTimeout?: number;

  /**
   * Filter to only run tests with a title matching one of the patterns. For example, passing `grep: /cart/` should only run
   * tests with "cart" in the title. Also available in the [command line](https://playwright.dev/docs/test-cli) with the `-g` option.
   *
   * `grep` option is also useful for [tagging tests](https://playwright.dev/docs/test-annotations#tag-tests).
   */
  grep?: RegExp|Array<RegExp>;

  /**
   * Filter to only run tests with a title **not** matching one of the patterns. This is the opposite of
   * [testConfig.grep](https://playwright.dev/docs/api/class-testconfig#test-config-grep). Also available in the
   * [command line](https://playwright.dev/docs/test-cli) with the `--grep-invert` option.
   *
   * `grepInvert` option is also useful for [tagging tests](https://playwright.dev/docs/test-annotations#tag-tests).
   */
  grepInvert?: RegExp|Array<RegExp>;

  /**
   * The maximum number of test failures for the whole test suite run. After reaching this number, testing will stop and exit
   * with an error. Setting to zero (default) disables this behavior.
   *
   * Also available in the [command line](https://playwright.dev/docs/test-cli) with the `--max-failures` and `-x` options.
   *
   * ```js
   * // playwright.config.ts
   * import type { PlaywrightTestConfig } from '@playwright/test';
   *
   * const config: PlaywrightTestConfig = {
   *   maxFailures: process.env.CI ? 1 : 0,
   * };
   * export default config;
   * ```
   *
   */
  maxFailures?: number;

  /**
   * Metadata that will be put directly to the test report serialized as JSON.
   */
  metadata?: Metadata;

  /**
   * Config name is visible in the report and during test execution, unless overridden by
   * [testProject.name](https://playwright.dev/docs/api/class-testproject#test-project-name).
   */
  name?: string;

  /**
   * The output directory for files created during test execution. Defaults to `<package.json-directory>/test-results`.
   *
   * ```js
   * // playwright.config.ts
   * import { type PlaywrightTestConfig } from '@playwright/test';
   *
   * const config: PlaywrightTestConfig = {
   *   outputDir: './test-results',
   * };
   * export default config;
   * ```
   *
   * This directory is cleaned at the start. When running a test, a unique subdirectory inside the
   * [testConfig.outputDir](https://playwright.dev/docs/api/class-testconfig#test-config-output-dir) is created, guaranteeing
   * that test running in parallel do not conflict. This directory can be accessed by
   * [testInfo.outputDir](https://playwright.dev/docs/api/class-testinfo#test-info-output-dir) and
   * [testInfo.outputPath(...pathSegments)](https://playwright.dev/docs/api/class-testinfo#test-info-output-path).
   *
   * Here is an example that uses
   * [testInfo.outputPath(...pathSegments)](https://playwright.dev/docs/api/class-testinfo#test-info-output-path) to create a
   * temporary file.
   *
   * ```js
   * import { test, expect } from '@playwright/test';
   * import fs from 'fs';
   *
   * test('example test', async ({}, testInfo) => {
   *   const file = testInfo.outputPath('temporary-file.txt');
   *   await fs.promises.writeFile(file, 'Put some data to the file', 'utf8');
   * });
   * ```
   *
   */
  outputDir?: string;

  /**
   * The base directory, relative to the config file, for snapshot files created with `toMatchSnapshot`. Defaults to
   * [testConfig.testDir](https://playwright.dev/docs/api/class-testconfig#test-config-test-dir).
   *
   * The directory for each test can be accessed by
   * [testInfo.snapshotDir](https://playwright.dev/docs/api/class-testinfo#test-info-snapshot-dir) and
   * [testInfo.snapshotPath(...pathSegments)](https://playwright.dev/docs/api/class-testinfo#test-info-snapshot-path).
   *
   * This path will serve as the base directory for each test file snapshot directory. Setting `snapshotDir` to
   * `'snapshots'`, the [testInfo.snapshotDir](https://playwright.dev/docs/api/class-testinfo#test-info-snapshot-dir) would
   * resolve to `snapshots/a.spec.js-snapshots`.
   */
  snapshotDir?: string;

  /**
   * Whether to preserve test output in the
   * [testConfig.outputDir](https://playwright.dev/docs/api/class-testconfig#test-config-output-dir). Defaults to `'always'`.
   * - `'always'` - preserve output for all tests;
   * - `'never'` - do not preserve output for any tests;
   * - `'failures-only'` - only preserve output for failed tests.
   */
  preserveOutput?: "always"|"never"|"failures-only";

  /**
   * Playwright Test supports running multiple test projects at the same time. See [TestProject] for more information.
   */
  projects?: Array<TestProject>;

  /**
   * Whether to suppress stdio and stderr output from the tests.
   */
  quiet?: boolean;

  /**
   * The number of times to repeat each test, useful for debugging flaky tests.
   */
  repeatEach?: number;

  /**
   * Whether to report slow test files. Pass `null` to disable this feature.
   *
   * Test files that took more than `threshold` milliseconds are considered slow, and the slowest ones are reported, no more
   * than `max` number of them. Passing zero as `max` reports all test files that exceed the threshold.
   */
  reportSlowTests?: null|{
    /**
     * The maximum number of slow test files to report. Defaults to `5`.
     */
    max: number;

    /**
     * Test duration in milliseconds that is considered slow. Defaults to 15 seconds.
     */
    threshold: number;
  };

  /**
   * The maximum number of retry attempts given to failed tests. By default failing tests are not retried. Learn more about
   * [test retries](https://playwright.dev/docs/test-retries#retries).
   *
   * ```js
   * // playwright.config.ts
   * import type { PlaywrightTestConfig } from '@playwright/test';
   *
   * const config: PlaywrightTestConfig = {
   *   retries: 2,
   * };
   * export default config;
   * ```
   *
   */
  retries?: number;

  /**
   * Shard tests and execute only the selected shard. Specify in the one-based form like `{ total: 5, current: 2 }`.
   *
   * Learn more about [parallelism and sharding](https://playwright.dev/docs/test-parallel) with Playwright Test.
   */
  shard?: null|{
    /**
     * The total number of shards.
     */
    total: number;

    /**
     * The index of the shard to execute, one-based.
     */
    current: number;
  };

  /**
   * Directory that will be recursively scanned for test files. Defaults to the directory of the configuration file.
   *
   * ```js
   * // playwright.config.ts
   * import type { PlaywrightTestConfig } from '@playwright/test';
   *
   * const config: PlaywrightTestConfig = {
   *   testDir: './tests/playwright',
   * };
   * export default config;
   * ```
   *
   */
  testDir?: string;

  /**
   * Files matching one of these patterns are not executed as test files. Matching is performed against the absolute file
   * path. Strings are treated as glob patterns.
   *
   * For example, `'**\/test-assets/**'` will ignore any files in the `test-assets` directory.
   *
   * ```js
   * // playwright.config.ts
   * import { type PlaywrightTestConfig } from '@playwright/test';
   *
   * const config: PlaywrightTestConfig = {
   *   testIgnore: '**\/test-assets/**',
   * };
   * export default config;
   * ```
   *
   */
  testIgnore?: string|RegExp|Array<string|RegExp>;

  /**
   * Only the files matching one of these patterns are executed as test files. Matching is performed against the absolute
   * file path. Strings are treated as glob patterns.
   *
   * By default, Playwright Test looks for files matching `.*(test|spec)\.(js|ts|mjs)`.
   *
   * ```js
   * // playwright.config.ts
   * import { type PlaywrightTestConfig } from '@playwright/test';
   *
   * const config: PlaywrightTestConfig = {
   *   testMatch: /.*\.e2e\.js/,
   * };
   * export default config;
   * ```
   *
   */
  testMatch?: string|RegExp|Array<string|RegExp>;

  /**
   * Timeout for each test in milliseconds. Defaults to 30 seconds.
   *
   * This is a base timeout for all tests. In addition, each test can configure its own timeout with
   * [test.setTimeout(timeout)](https://playwright.dev/docs/api/class-test#test-set-timeout). Learn more about
   * [various timeouts](https://playwright.dev/docs/test-timeouts).
   *
   * ```js
   * // playwright.config.ts
   * import type { PlaywrightTestConfig } from '@playwright/test';
   *
   * const config: PlaywrightTestConfig = {
   *   timeout: 5 * 60 * 1000,
   * };
   * export default config;
   * ```
   *
   */
  timeout?: number;

  /**
   * Whether to update expected snapshots with the actual results produced by the test run. Defaults to `'missing'`.
   * - `'all'` - All tests that are executed will update snapshots that did not match. Matching snapshots will not be
   *   updated.
   * - `'none'` - No snapshots are updated.
   * - `'missing'` - Missing snapshots are created, for example when authoring a new test and running it for the first
   *   time. This is the default.
   *
   * Learn more about [snapshots](https://playwright.dev/docs/test-snapshots).
   */
  updateSnapshots?: "all"|"none"|"missing";

  /**
   * The maximum number of concurrent worker processes to use for parallelizing tests.
   *
   * Playwright Test uses worker processes to run tests. There is always at least one worker process, but more can be used to
   * speed up test execution.
   *
   * Defaults to one half of the number of CPU cores. Learn more about [parallelism and sharding](https://playwright.dev/docs/test-parallel) with
   * Playwright Test.
   *
   * ```js
   * // playwright.config.ts
   * import type { PlaywrightTestConfig } from '@playwright/test';
   *
   * const config: PlaywrightTestConfig = {
   *   workers: 3,
   * };
   * export default config;
   * ```
   *
   */
  workers?: number;}

/**
 * Playwright Test provides many options to configure how your tests are collected and executed, for example `timeout` or
 * `testDir`. These options are described in the [TestConfig] object in the [configuration file](https://playwright.dev/docs/test-configuration).
 *
 * Playwright Test supports running multiple test projects at the same time. Project-specific options should be put to
 * [testConfig.projects](https://playwright.dev/docs/api/class-testconfig#test-config-projects), but top-level [TestConfig]
 * can also define base options shared between all projects.
 *
 * ```js
 * // playwright.config.ts
 * import type { PlaywrightTestConfig } from '@playwright/test';
 *
 * const config: PlaywrightTestConfig = {
 *   timeout: 30000,
 *   globalTimeout: 600000,
 *   reporter: 'list',
 *   testDir: './tests',
 * };
 * export default config;
 * ```
 *
 */
export interface Config<TestArgs = {}, WorkerArgs = {}> extends TestConfig {
  /**
   * Playwright Test supports running multiple test projects at the same time. See [TestProject] for more information.
   */
  projects?: Project<TestArgs, WorkerArgs>[];
  /**
   * Global options for all tests, for example
   * [testOptions.browserName](https://playwright.dev/docs/api/class-testoptions#test-options-browser-name). Learn more about
   * [configuration](https://playwright.dev/docs/test-configuration) and see [available options][TestOptions].
   *
   * ```js
   * // playwright.config.ts
   * import type { PlaywrightTestConfig } from '@playwright/test';
   *
   * const config: PlaywrightTestConfig = {
   *   use: {
   *     browserName: 'chromium',
   *   },
   * };
   * export default config;
   * ```
   *
   */
  use?: UseOptions<TestArgs, WorkerArgs>;
}

export type Metadata = { [key: string]: any };

/**
 * Playwright Test provides many options to configure how your tests are collected and executed, for example `timeout` or
 * `testDir`. These options are described in the [TestConfig] object in the [configuration file](https://playwright.dev/docs/test-configuration).
 *
 * Playwright Test supports running multiple test projects at the same time. Project-specific options should be put to
 * [testConfig.projects](https://playwright.dev/docs/api/class-testconfig#test-config-projects), but top-level [TestConfig]
 * can also define base options shared between all projects.
 *
 * ```js
 * // playwright.config.ts
 * import type { PlaywrightTestConfig } from '@playwright/test';
 *
 * const config: PlaywrightTestConfig = {
 *   timeout: 30000,
 *   globalTimeout: 600000,
 *   reporter: 'list',
 *   testDir: './tests',
 * };
 * export default config;
 * ```
 *
 */
export interface FullConfig<TestArgs = {}, WorkerArgs = {}> {
  /**
   * Whether to exit with an error if any tests or groups are marked as
   * [test.only(title, testFunction)](https://playwright.dev/docs/api/class-test#test-only) or
   * [test.describe.only(title, callback)](https://playwright.dev/docs/api/class-test#test-describe-only). Useful on CI.
   *
   * ```js
   * // playwright.config.ts
   * import type { PlaywrightTestConfig } from '@playwright/test';
   *
   * const config: PlaywrightTestConfig = {
   *   forbidOnly: !!process.env.CI,
   * };
   * export default config;
   * ```
   *
   */
  forbidOnly: boolean;
  /**
   * Playwright Test runs tests in parallel. In order to achieve that, it runs several worker processes that run at the same
   * time. By default, **test files** are run in parallel. Tests in a single file are run in order, in the same worker
   * process.
   *
   * You can configure entire test run to concurrently execute all tests in all files using this option.
   */
  fullyParallel: boolean;
  /**
   * Path to the global setup file. This file will be required and run before all the tests. It must export a single function
   * that takes a [`TestConfig`] argument.
   *
   * Learn more about [global setup and teardown](https://playwright.dev/docs/test-advanced#global-setup-and-teardown).
   *
   * ```js
   * // playwright.config.ts
   * import { type PlaywrightTestConfig } from '@playwright/test';
   *
   * const config: PlaywrightTestConfig = {
   *   globalSetup: './global-setup',
   * };
   * export default config;
   * ```
   *
   */
  globalSetup: string | null;
  /**
   * Path to the global teardown file. This file will be required and run after all the tests. It must export a single
   * function. See also [testConfig.globalSetup](https://playwright.dev/docs/api/class-testconfig#test-config-global-setup).
   *
   * Learn more about [global setup and teardown](https://playwright.dev/docs/test-advanced#global-setup-and-teardown).
   *
   * ```js
   * // playwright.config.ts
   * import { type PlaywrightTestConfig } from '@playwright/test';
   *
   * const config: PlaywrightTestConfig = {
   *   globalTeardown: './global-teardown',
   * };
   * export default config;
   * ```
   *
   */
  globalTeardown: string | null;
  /**
   * Maximum time in milliseconds the whole test suite can run. Zero timeout (default) disables this behavior. Useful on CI
   * to prevent broken setup from running too long and wasting resources. Learn more about
   * [various timeouts](https://playwright.dev/docs/test-timeouts).
   *
   * ```js
   * // playwright.config.ts
   * import type { PlaywrightTestConfig } from '@playwright/test';
   *
   * const config: PlaywrightTestConfig = {
   *   globalTimeout: process.env.CI ? 60 * 60 * 1000 : undefined,
   * };
   * export default config;
   * ```
   *
   */
  globalTimeout: number;
  /**
   * Filter to only run tests with a title matching one of the patterns. For example, passing `grep: /cart/` should only run
   * tests with "cart" in the title. Also available in the [command line](https://playwright.dev/docs/test-cli) with the `-g` option.
   *
   * `grep` option is also useful for [tagging tests](https://playwright.dev/docs/test-annotations#tag-tests).
   */
  grep: RegExp | RegExp[];
  /**
   * Filter to only run tests with a title **not** matching one of the patterns. This is the opposite of
   * [testConfig.grep](https://playwright.dev/docs/api/class-testconfig#test-config-grep). Also available in the
   * [command line](https://playwright.dev/docs/test-cli) with the `--grep-invert` option.
   *
   * `grepInvert` option is also useful for [tagging tests](https://playwright.dev/docs/test-annotations#tag-tests).
   */
  grepInvert: RegExp | RegExp[] | null;
  /**
   * The maximum number of test failures for the whole test suite run. After reaching this number, testing will stop and exit
   * with an error. Setting to zero (default) disables this behavior.
   *
   * Also available in the [command line](https://playwright.dev/docs/test-cli) with the `--max-failures` and `-x` options.
   *
   * ```js
   * // playwright.config.ts
   * import type { PlaywrightTestConfig } from '@playwright/test';
   *
   * const config: PlaywrightTestConfig = {
   *   maxFailures: process.env.CI ? 1 : 0,
   * };
   * export default config;
   * ```
   *
   */
  maxFailures: number;
  /**
   * Metadata that will be put directly to the test report serialized as JSON.
   */
  metadata: Metadata;
  version: string;
  /**
   * Whether to preserve test output in the
   * [testConfig.outputDir](https://playwright.dev/docs/api/class-testconfig#test-config-output-dir). Defaults to `'always'`.
   * - `'always'` - preserve output for all tests;
   * - `'never'` - do not preserve output for any tests;
   * - `'failures-only'` - only preserve output for failed tests.
   */
  preserveOutput: 'always' | 'never' | 'failures-only';
  /**
   * Playwright Test supports running multiple test projects at the same time. See [TestProject] for more information.
   */
  projects: FullProject<TestArgs, WorkerArgs>[];
  /**
   * The list of reporters to use. Each reporter can be:
   * - A builtin reporter name like `'list'` or `'json'`.
   * - A module name like `'my-awesome-reporter'`.
   * - A relative path to the reporter like `'./reporters/my-awesome-reporter.js'`.
   *
   * You can pass options to the reporter in a tuple like `['json', { outputFile: './report.json' }]`.
   *
   * Learn more in the [reporters guide](https://playwright.dev/docs/test-reporters).
   *
   * ```js
   * // playwright.config.ts
   * import type { PlaywrightTestConfig } from '@playwright/test';
   *
   * const config: PlaywrightTestConfig = {
   *   reporter: 'line',
   * };
   * export default config;
   * ```
   *
   */
  reporter: ReporterDescription[];
  /**
   * Whether to report slow test files. Pass `null` to disable this feature.
   *
   * Test files that took more than `threshold` milliseconds are considered slow, and the slowest ones are reported, no more
   * than `max` number of them. Passing zero as `max` reports all test files that exceed the threshold.
   */
  reportSlowTests: { max: number, threshold: number } | null;
  rootDir: string;
  /**
   * Whether to suppress stdio and stderr output from the tests.
   */
  quiet: boolean;
  /**
   * Shard tests and execute only the selected shard. Specify in the one-based form like `{ total: 5, current: 2 }`.
   *
   * Learn more about [parallelism and sharding](https://playwright.dev/docs/test-parallel) with Playwright Test.
   */
  shard: { total: number, current: number } | null;
  /**
   * Whether to update expected snapshots with the actual results produced by the test run. Defaults to `'missing'`.
   * - `'all'` - All tests that are executed will update snapshots that did not match. Matching snapshots will not be
   *   updated.
   * - `'none'` - No snapshots are updated.
   * - `'missing'` - Missing snapshots are created, for example when authoring a new test and running it for the first
   *   time. This is the default.
   *
   * Learn more about [snapshots](https://playwright.dev/docs/test-snapshots).
   */
  updateSnapshots: 'all' | 'none' | 'missing';
  /**
   * The maximum number of concurrent worker processes to use for parallelizing tests.
   *
   * Playwright Test uses worker processes to run tests. There is always at least one worker process, but more can be used to
   * speed up test execution.
   *
   * Defaults to one half of the number of CPU cores. Learn more about [parallelism and sharding](https://playwright.dev/docs/test-parallel) with
   * Playwright Test.
   *
   * ```js
   * // playwright.config.ts
   * import type { PlaywrightTestConfig } from '@playwright/test';
   *
   * const config: PlaywrightTestConfig = {
   *   workers: 3,
   * };
   * export default config;
   * ```
   *
   */
  workers: number;
  /**
   * Launch a development web server (or multiple) during the tests.
   *
   * If the port is specified, Playwright Test will wait for it to be available on `127.0.0.1` or `::1`, before running the
   * tests. If the url is specified, Playwright Test will wait for the URL to return a 2xx, 3xx, 400, 401, 402, or 403 status
   * code before running the tests.
   *
   * For continuous integration, you may want to use the `reuseExistingServer: !process.env.CI` option which does not use an
   * existing server on the CI. To see the stdout, you can set the `DEBUG=pw:webserver` environment variable.
   *
   * The `port` (but not the `url`) gets passed over to Playwright as a
   * [testOptions.baseURL](https://playwright.dev/docs/api/class-testoptions#test-options-base-url). For example port `8080`
   * produces `baseURL` equal `http://localhost:8080`. If `webServer` is specified as an array, you must explicitly configure
   * the `baseURL` (even if it only has one entry).
   *
   * > NOTE: It is also recommended to specify
   * [testOptions.baseURL](https://playwright.dev/docs/api/class-testoptions#test-options-base-url) in the config, so that
   * tests could use relative urls.
   *
   * ```js
   * // playwright.config.ts
   * import type { PlaywrightTestConfig } from '@playwright/test';
   * const config: PlaywrightTestConfig = {
   *   webServer: {
   *     command: 'npm run start',
   *     port: 3000,
   *     timeout: 120 * 1000,
   *     reuseExistingServer: !process.env.CI,
   *   },
   *   use: {
   *     baseURL: 'http://localhost:3000/',
   *   },
   * };
   * export default config;
   * ```
   *
   * Now you can use a relative path when navigating the page:
   *
   * ```js
   * // test.spec.ts
   * import { test } from '@playwright/test';
   *
   * test('test', async ({ page }) => {
   *   // This will result in http://localhost:3000/foo
   *   await page.goto('/foo');
   * });
   * ```
   *
   * Multiple web servers (or background processes) can be launched:
   *
   * ```js
   * // playwright.config.ts
   * import type { PlaywrightTestConfig } from '@playwright/test';
   * const config: PlaywrightTestConfig = {
   *   webServer: [
   *     {
   *       command: 'npm run start',
   *       port: 3000,
   *       timeout: 120 * 1000,
   *       reuseExistingServer: !process.env.CI,
   *     },
   *     {
   *       command: 'npm run backend',
   *       port: 3333,
   *       timeout: 120 * 1000,
   *       reuseExistingServer: !process.env.CI,
   *     }
   *   ],
   *   use: {
   *     baseURL: 'http://localhost:3000/',
   *   },
   * };
   * export default config;
   * ```
   *
   */
  webServer: TestConfigWebServer | null;
}

export type TestStatus = 'passed' | 'failed' | 'timedOut' | 'skipped' | 'interrupted';

/**
 * `WorkerInfo` contains information about the worker that is running tests. It is available to
 * [test.beforeAll(hookFunction)](https://playwright.dev/docs/api/class-test#test-before-all) and
 * [test.afterAll(hookFunction)](https://playwright.dev/docs/api/class-test#test-after-all) hooks and worker-scoped
 * fixtures.
 *
 * ```js
 * import { test, expect } from '@playwright/test';
 *
 * test.beforeAll(async ({ browserName }, workerInfo) => {
 *   console.log(`Running ${browserName} in worker #${workerInfo.workerIndex}`);
 * });
 * ```
 *
 */
export interface WorkerInfo {
  /**
   * Processed configuration from the [configuration file](https://playwright.dev/docs/test-configuration).
   */
  config: FullConfig;
  /**
   * Processed project configuration from the [configuration file](https://playwright.dev/docs/test-configuration).
   */
  project: FullProject;
  /**
   * The index of the worker between `0` and `workers - 1`. It is guaranteed that workers running at the same time have a
   * different `parallelIndex`. When a worker is restarted, for example after a failure, the new worker process has the same
   * `parallelIndex`.
   *
   * Also available as `process.env.TEST_PARALLEL_INDEX`. Learn more about [parallelism and sharding](https://playwright.dev/docs/test-parallel)
   * with Playwright Test.
   */
  parallelIndex: number;

  /**
   * The unique index of the worker process that is running the test. When a worker is restarted, for example after a
   * failure, the new worker process gets a new unique `workerIndex`.
   *
   * Also available as `process.env.TEST_WORKER_INDEX`. Learn more about [parallelism and sharding](https://playwright.dev/docs/test-parallel) with
   * Playwright Test.
   */
  workerIndex: number;}

/**
 * `TestInfo` contains information about currently running test. It is available to any test function,
 * [test.beforeEach(hookFunction)](https://playwright.dev/docs/api/class-test#test-before-each) and
 * [test.afterEach(hookFunction)](https://playwright.dev/docs/api/class-test#test-after-each) hooks and test-scoped
 * fixtures. `TestInfo` provides utilities to control test execution: attach files, update test timeout, determine which
 * test is currently running and whether it was retried, etc.
 *
 * ```js
 * import { test, expect } from '@playwright/test';
 *
 * test('basic test', async ({ page }, testInfo) => {
 *   expect(testInfo.title).toBe('basic test');
 *   await page.screenshot(testInfo.outputPath('screenshot.png'));
 * });
 * ```
 *
 */
export interface TestInfo {
  /**
   * Processed configuration from the [configuration file](https://playwright.dev/docs/test-configuration).
   */
  config: FullConfig;
  /**
   * Processed project configuration from the [configuration file](https://playwright.dev/docs/test-configuration).
   */
  project: FullProject;
  /**
   * The list of annotations applicable to the current test. Includes annotations from the test, annotations from all
   * [test.describe(title, callback)](https://playwright.dev/docs/api/class-test#test-describe-1) groups the test belongs to
   * and file-level annotations for the test file.
   *
   * Learn more about [test annotations](https://playwright.dev/docs/test-annotations).
   */
  annotations: Array<{
    /**
     * Annotation type, for example `'skip'` or `'fail'`.
     */
    type: string;

    /**
     * Optional description.
     */
    description?: string;
  }>;

  /**
   * The list of files or buffers attached to the current test. Some reporters show test attachments.
   *
   * To add an attachment, use
   * [testInfo.attach(name[, options])](https://playwright.dev/docs/api/class-testinfo#test-info-attach) instead of directly
   * pushing onto this array.
   */
  attachments: Array<{
    /**
     * Attachment name.
     */
    name: string;

    /**
     * Content type of this attachment to properly present in the report, for example `'application/json'` or `'image/png'`.
     */
    contentType: string;

    /**
     * Optional path on the filesystem to the attached file.
     */
    path?: string;

    /**
     * Optional attachment body used instead of a file.
     */
    body?: Buffer;
  }>;

  /**
   * Attach a value or a file from disk to the current test. Some reporters show test attachments. Either `path` or `body`
   * must be specified, but not both.
   *
   * For example, you can attach a screenshot to the test:
   *
   * ```js
   * import { test, expect } from '@playwright/test';
   *
   * test('basic test', async ({ page }, testInfo) => {
   *   await page.goto('https://playwright.dev');
   *   const screenshot = await page.screenshot();
   *   await testInfo.attach('screenshot', { body: screenshot, contentType: 'image/png' });
   * });
   * ```
   *
   * Or you can attach files returned by your APIs:
   *
   * ```js
   * import { test, expect } from '@playwright/test';
   *
   * test('basic test', async ({}, testInfo) => {
   *   const { download } = require('./my-custom-helpers');
   *   const tmpPath = await download('a');
   *   await testInfo.attach('downloaded', { path: tmpPath });
   * });
   * ```
   *
   * > NOTE: [testInfo.attach(name[, options])](https://playwright.dev/docs/api/class-testinfo#test-info-attach)
   * automatically takes care of copying attached files to a location that is accessible to reporters. You can safely remove
   * the attachment after awaiting the attach call.
   * @param name Attachment name.
   * @param options
   */
  attach(name: string, options?: {
    /**
     * Attachment body. Mutually exclusive with `path`.
     */
    body?: string|Buffer;

    /**
     * Content type of this attachment to properly present in the report, for example `'application/json'` or `'image/png'`. If
     * omitted, content type is inferred based on the `path`, or defaults to `text/plain` for [string] attachments and
     * `application/octet-stream` for [Buffer] attachments.
     */
    contentType?: string;

    /**
     * Path on the filesystem to the attached file. Mutually exclusive with `body`.
     */
    path?: string;
  }): Promise<void>;

  /**
   * Column number where the currently running test is declared.
   */
  column: number;

  /**
   * The number of milliseconds the test took to finish. Always zero before the test finishes, either successfully or not.
   * Can be used in [test.afterEach(hookFunction)](https://playwright.dev/docs/api/class-test#test-after-each) hook.
   */
  duration: number;

  /**
   * First error thrown during test execution, if any. This is equal to the first element in
   * [testInfo.errors](https://playwright.dev/docs/api/class-testinfo#test-info-errors).
   */
  error?: TestError;

  /**
   * Errors thrown during test execution, if any.
   */
  errors: Array<TestError>;

  /**
   * Expected status for the currently running test. This is usually `'passed'`, except for a few cases:
   * - `'skipped'` for skipped tests, e.g. with [test.skip()](https://playwright.dev/docs/api/class-test#test-skip-2);
   * - `'failed'` for tests marked as failed with [test.fail()](https://playwright.dev/docs/api/class-test#test-fail-1).
   *
   * Expected status is usually compared with the actual
   * [testInfo.status](https://playwright.dev/docs/api/class-testinfo#test-info-status):
   *
   * ```js
   * import { test, expect } from '@playwright/test';
   *
   * test.afterEach(async ({}, testInfo) => {
   *   if (testInfo.status !== testInfo.expectedStatus)
   *     console.log(`${testInfo.title} did not run as expected!`);
   * });
   * ```
   *
   */
  expectedStatus: "passed"|"failed"|"timedOut"|"skipped"|"interrupted";

  /**
   * Marks the currently running test as "should fail". Playwright Test runs this test and ensures that it is actually
   * failing. This is useful for documentation purposes to acknowledge that some functionality is broken until it is fixed.
   * This is similar to [test.fail()](https://playwright.dev/docs/api/class-test#test-fail-1).
   */
  fail(): void;

  /**
   * Conditionally mark the currently running test as "should fail" with an optional description. This is similar to
   * [test.fail(condition[, description])](https://playwright.dev/docs/api/class-test#test-fail-2).
   * @param condition Test is marked as "should fail" when the condition is `true`.
   * @param description Optional description that will be reflected in a test report.
   */
  fail(condition: boolean, description?: string): void;

  /**
   * Absolute path to a file where the currently running test is declared.
   */
  file: string;

  /**
   * Mark a test as "fixme", with the intention to fix it. Test is immediately aborted. This is similar to
   * [test.fixme()](https://playwright.dev/docs/api/class-test#test-fixme-2).
   */
  fixme(): void;

  /**
   * Conditionally mark the currently running test as "fixme" with an optional description. This is similar to
   * [test.fixme(condition[, description])](https://playwright.dev/docs/api/class-test#test-fixme-3).
   * @param condition Test is marked as "fixme" when the condition is `true`.
   * @param description Optional description that will be reflected in a test report.
   */
  fixme(condition: boolean, description?: string): void;

  /**
   * Test function as passed to `test(title, testFunction)`.
   */
  fn: Function;

  /**
   * Line number where the currently running test is declared.
   */
  line: number;

  /**
   * Absolute path to the snapshot output directory for this specific test. Each test suite gets its own directory so they
   * cannot conflict.
   */
  snapshotDir: string;

  /**
   * Absolute path to the output directory for this specific test run. Each test run gets its own directory so they cannot
   * conflict.
   */
  outputDir: string;

  /**
   * Returns a path inside the [testInfo.outputDir](https://playwright.dev/docs/api/class-testinfo#test-info-output-dir)
   * where the test can safely put a temporary file. Guarantees that tests running in parallel will not interfere with each
   * other.
   *
   * ```js
   * import { test, expect } from '@playwright/test';
   * import fs from 'fs';
   *
   * test('example test', async ({}, testInfo) => {
   *   const file = testInfo.outputPath('dir', 'temporary-file.txt');
   *   await fs.promises.writeFile(file, 'Put some data to the dir/temporary-file.txt', 'utf8');
   * });
   * ```
   *
   * > Note that `pathSegments` accepts path segments to the test output directory such as `testInfo.outputPath('relative',
   * 'path', 'to', 'output')`.
   * > However, this path must stay within the
   * [testInfo.outputDir](https://playwright.dev/docs/api/class-testinfo#test-info-output-dir) directory for each test (i.e.
   * `test-results/a-test-title`), otherwise it will throw.
   * @param pathSegments Path segments to append at the end of the resulting path.
   */
  outputPath(...pathSegments: Array<string>): string;

  /**
   * The index of the worker between `0` and `workers - 1`. It is guaranteed that workers running at the same time have a
   * different `parallelIndex`. When a worker is restarted, for example after a failure, the new worker process has the same
   * `parallelIndex`.
   *
   * Also available as `process.env.TEST_PARALLEL_INDEX`. Learn more about [parallelism and sharding](https://playwright.dev/docs/test-parallel)
   * with Playwright Test.
   */
  parallelIndex: number;

  /**
   * Specifies a unique repeat index when running in "repeat each" mode. This mode is enabled by passing `--repeat-each` to
   * the [command line](https://playwright.dev/docs/test-cli).
   */
  repeatEachIndex: number;

  /**
   * Specifies the retry number when the test is retried after a failure. The first test run has
   * [testInfo.retry](https://playwright.dev/docs/api/class-testinfo#test-info-retry) equal to zero, the first retry has it
   * equal to one, and so on. Learn more about [retries](https://playwright.dev/docs/test-retries#retries).
   *
   * ```js
   * import { test, expect } from '@playwright/test';
   *
   * test.beforeEach(async ({}, testInfo) => {
   *   // You can access testInfo.retry in any hook or fixture.
   *   if (testInfo.retry > 0)
   *     console.log(`Retrying!`);
   * });
   *
   * test('my test', async ({ page }, testInfo) => {
   *   // Here we clear some server-side state when retrying.
   *   if (testInfo.retry)
   *     await cleanSomeCachesOnTheServer();
   *   // ...
   * });
   * ```
   *
   */
  retry: number;

  /**
   * Changes the timeout for the currently running test. Zero means no timeout. Learn more about
   * [various timeouts](https://playwright.dev/docs/test-timeouts).
   *
   * Timeout is usually specified in the [configuration file](https://playwright.dev/docs/test-configuration), but it could be useful to change the
   * timeout in certain scenarios:
   *
   * ```js
   * import { test, expect } from '@playwright/test';
   *
   * test.beforeEach(async ({ page }, testInfo) => {
   *   // Extend timeout for all tests running this hook by 30 seconds.
   *   testInfo.setTimeout(testInfo.timeout + 30000);
   * });
   * ```
   *
   * @param timeout Timeout in milliseconds.
   */
  setTimeout(timeout: number): void;

  /**
   * Unconditionally skip the currently running test. Test is immediately aborted. This is similar to
   * [test.skip()](https://playwright.dev/docs/api/class-test#test-skip-2).
   */
  skip(): void;

  /**
   * Conditionally skips the currently running test with an optional description. This is similar to
   * [test.skip(condition[, description])](https://playwright.dev/docs/api/class-test#test-skip-3).
   * @param condition A skip condition. Test is skipped when the condition is `true`.
   * @param description Optional description that will be reflected in a test report.
   */
  skip(condition: boolean, description?: string): void;

  /**
   * Marks the currently running test as "slow", giving it triple the default timeout. This is similar to
   * [test.slow()](https://playwright.dev/docs/api/class-test#test-slow-1).
   */
  slow(): void;

  /**
   * Conditionally mark the currently running test as "slow" with an optional description, giving it triple the default
   * timeout. This is similar to
   * [test.slow(condition[, description])](https://playwright.dev/docs/api/class-test#test-slow-2).
   * @param condition Test is marked as "slow" when the condition is `true`.
   * @param description Optional description that will be reflected in a test report.
   */
  slow(condition: boolean, description?: string): void;

  /**
   * Returns a path to a snapshot file with the given `pathSegments`. Learn more about [snapshots](https://playwright.dev/docs/test-snapshots).
   *
   * > Note that `pathSegments` accepts path segments to the snapshot file such as `testInfo.snapshotPath('relative', 'path',
   * 'to', 'snapshot.png')`.
   * > However, this path must stay within the snapshots directory for each test file (i.e. `a.spec.js-snapshots`), otherwise
   * it will throw.
   * @param pathSegments The name of the snapshot or the path segments to define the snapshot file path. Snapshots with the same name in the same test file are expected to be the same.
   */
  snapshotPath(...pathSegments: Array<string>): string;

  /**
   * Suffix used to differentiate snapshots between multiple test configurations. For example, if snapshots depend on the
   * platform, you can set `testInfo.snapshotSuffix` equal to `process.platform`. In this case
   * `expect(value).toMatchSnapshot(snapshotName)` will use different snapshots depending on the platform. Learn more about
   * [snapshots](https://playwright.dev/docs/test-snapshots).
   */
  snapshotSuffix: string;

  /**
   * Actual status for the currently running test. Available after the test has finished in
   * [test.afterEach(hookFunction)](https://playwright.dev/docs/api/class-test#test-after-each) hook and fixtures.
   *
   * Status is usually compared with the
   * [testInfo.expectedStatus](https://playwright.dev/docs/api/class-testinfo#test-info-expected-status):
   *
   * ```js
   * import { test, expect } from '@playwright/test';
   *
   * test.afterEach(async ({}, testInfo) => {
   *   if (testInfo.status !== testInfo.expectedStatus)
   *     console.log(`${testInfo.title} did not run as expected!`);
   * });
   * ```
   *
   */
  status?: "passed"|"failed"|"timedOut"|"skipped"|"interrupted";

  /**
   * Output written to `process.stderr` or `console.error` during the test execution.
   */
  stderr: Array<string|Buffer>;

  /**
   * Output written to `process.stdout` or `console.log` during the test execution.
   */
  stdout: Array<string|Buffer>;

  /**
   * Timeout in milliseconds for the currently running test. Zero means no timeout. Learn more about
   * [various timeouts](https://playwright.dev/docs/test-timeouts).
   *
   * Timeout is usually specified in the [configuration file](https://playwright.dev/docs/test-configuration)
   *
   * ```js
   * import { test, expect } from '@playwright/test';
   *
   * test.beforeEach(async ({ page }, testInfo) => {
   *   // Extend timeout for all tests running this hook by 30 seconds.
   *   testInfo.setTimeout(testInfo.timeout + 30000);
   * });
   * ```
   *
   */
  timeout: number;

  /**
   * The title of the currently running test as passed to `test(title, testFunction)`.
   */
  title: string;

  /**
   * The full title path starting with the project.
   */
  titlePath: Array<string>;

  /**
   * The unique index of the worker process that is running the test. When a worker is restarted, for example after a
   * failure, the new worker process gets a new unique `workerIndex`.
   *
   * Also available as `process.env.TEST_WORKER_INDEX`. Learn more about [parallelism and sharding](https://playwright.dev/docs/test-parallel) with
   * Playwright Test.
   */
  workerIndex: number;}

interface SuiteFunction {
  /**
   * Declares a group of tests.
   *
   * ```js
   * test.describe('two tests', () => {
   *   test('one', async ({ page }) => {
   *     // ...
   *   });
   *
   *   test('two', async ({ page }) => {
   *     // ...
   *   });
   * });
   * ```
   *
   * @param title Group title.
   * @param callback A callback that is run immediately when calling [test.describe(title, callback)](https://playwright.dev/docs/api/class-test#test-describe-1). Any tests added in this
   * callback will belong to the group.
   */
  (title: string, callback: () => void): void;
  /**
   * Declares an anonymous group of tests. This is convenient to give a group of tests a common option with
   * [test.use(options)](https://playwright.dev/docs/api/class-test#test-use).
   *
   * ```js
   * test.describe(() => {
   *   test.use({ colorScheme: 'dark' });
   *
   *   test('one', async ({ page }) => {
   *     // ...
   *   });
   *
   *   test('two', async ({ page }) => {
   *     // ...
   *   });
   * });
   * ```
   *
   * @param callback A callback that is run immediately when calling [test.describe(callback)](https://playwright.dev/docs/api/class-test#test-describe-2). Any tests added in this callback
   * will belong to the group.
   */
  (callback: () => void): void;
}

interface TestFunction<TestArgs> {
  (title: string, testFunction: (args: TestArgs, testInfo: TestInfo) => Promise<void> | void): void;
}

/**
 * Playwright Test provides a `test` function to declare tests and [`expect` function](https://jestjs.io/docs/expect) to
 * write assertions.
 *
 * ```js
 * import { test, expect } from '@playwright/test';
 *
 * test('basic test', async ({ page }) => {
 *   await page.goto('https://playwright.dev/');
 *   const name = await page.innerText('.navbar__title');
 *   expect(name).toBe('Playwright');
 * });
 * ```
 *
 */
export interface TestType<TestArgs extends KeyValue, WorkerArgs extends KeyValue> extends TestFunction<TestArgs & WorkerArgs> {
  /**
   * Declares a focused test. If there are some focused tests or suites, all of them will be run but nothing else.
   *
   * ```js
   * test.only('focus this test', async ({ page }) => {
   *   // Run only focused tests in the entire project.
   * });
   * ```
   *
   * @param title Test title.
   * @param testFunction Test function that takes one or two arguments: an object with fixtures and optional [TestInfo].
   */
  only: TestFunction<TestArgs & WorkerArgs>;
  /**
   * Declares a group of tests.
   *
   * ```js
   * test.describe('two tests', () => {
   *   test('one', async ({ page }) => {
   *     // ...
   *   });
   *
   *   test('two', async ({ page }) => {
   *     // ...
   *   });
   * });
   * ```
   *
   * @param title Group title.
   * @param callback A callback that is run immediately when calling [test.describe(title, callback)](https://playwright.dev/docs/api/class-test#test-describe-1). Any tests added in this
   * callback will belong to the group.
   */
  describe: SuiteFunction & {
    /**
   * Declares a focused group of tests. If there are some focused tests or suites, all of them will be run but nothing else.
   *
   * ```js
   * test.describe.only('focused group', () => {
   *   test('in the focused group', async ({ page }) => {
   *     // This test will run
   *   });
   * });
   * test('not in the focused group', async ({ page }) => {
   *   // This test will not run
   * });
   * ```
   *
   * @param title Group title.
   * @param callback A callback that is run immediately when calling [test.describe.only(title, callback)](https://playwright.dev/docs/api/class-test#test-describe-only). Any tests added in
   * this callback will belong to the group.
   */
  only: SuiteFunction;
    /**
   * Declares a skipped test group, similarly to
   * [test.describe(title, callback)](https://playwright.dev/docs/api/class-test#test-describe-1). Tests in the skipped group
   * are never run.
   *
   * ```js
   * test.describe.skip('skipped group', () => {
   *   test('example', async ({ page }) => {
   *     // This test will not run
   *   });
   * });
   * ```
   *
   * @param title Group title.
   * @param callback A callback that is run immediately when calling [test.describe.skip(title, callback)](https://playwright.dev/docs/api/class-test#test-describe-skip). Any tests added in
   * this callback will belong to the group, and will not be run.
   */
  skip: SuiteFunction;
    /**
   * Declares a test group similarly to
   * [test.describe(title, callback)](https://playwright.dev/docs/api/class-test#test-describe-1). Tests in this group are
   * marked as "fixme" and will not be executed.
   *
   * ```js
   * test.describe.fixme('broken tests', () => {
   *   test('example', async ({ page }) => {
   *     // This test will not run
   *   });
   * });
   * ```
   *
   * @param title Group title.
   * @param callback A callback that is run immediately when calling [test.describe.fixme(title, callback)](https://playwright.dev/docs/api/class-test#test-describe-fixme). Any tests added
   * in this callback will belong to the group, and will not be run.
   */
  fixme: SuiteFunction;
    /**
   * Declares a group of tests that should always be run serially. If one of the tests fails, all subsequent tests are
   * skipped. All tests in a group are retried together.
   *
   * > NOTE: See [test.describe.configure([options])](https://playwright.dev/docs/api/class-test#test-describe-configure) for
   * the preferred way of configuring the execution mode.
   * > NOTE: Using serial is not recommended. It is usually better to make your tests isolated, so they can be run
   * independently.
   *
   * ```js
   * test.describe.serial('group', () => {
   *   test('runs first', async ({ page }) => {});
   *   test('runs second', async ({ page }) => {});
   * });
   * ```
   *
   * @param title Group title.
   * @param callback A callback that is run immediately when calling [test.describe.serial(title, callback)](https://playwright.dev/docs/api/class-test#test-describe-serial). Any tests
   * added in this callback will belong to the group.
   */
  serial: SuiteFunction & {
      /**
   * Declares a focused group of tests that should always be run serially. If one of the tests fails, all subsequent tests
   * are skipped. All tests in a group are retried together. If there are some focused tests or suites, all of them will be
   * run but nothing else.
   *
   * > NOTE: Using serial is not recommended. It is usually better to make your tests isolated, so they can be run
   * independently.
   *
   * ```js
   * test.describe.serial.only('group', () => {
   *   test('runs first', async ({ page }) => {
   *   });
   *   test('runs second', async ({ page }) => {
   *   });
   * });
   * ```
   *
   * @param title Group title.
   * @param callback A callback that is run immediately when calling [test.describe.serial.only(title, callback)](https://playwright.dev/docs/api/class-test#test-describe-serial-only). Any
   * tests added in this callback will belong to the group.
   */
  only: SuiteFunction;
    };
    /**
   * Declares a group of tests that could be run in parallel. By default, tests in a single test file run one after another,
   * but using [test.describe.parallel(title, callback)](https://playwright.dev/docs/api/class-test#test-describe-parallel)
   * allows them to run in parallel.
   *
   * > NOTE: See [test.describe.configure([options])](https://playwright.dev/docs/api/class-test#test-describe-configure) for
   * the preferred way of configuring the execution mode.
   *
   * ```js
   * test.describe.parallel('group', () => {
   *   test('runs in parallel 1', async ({ page }) => {});
   *   test('runs in parallel 2', async ({ page }) => {});
   * });
   * ```
   *
   * Note that parallel tests are executed in separate processes and cannot share any state or global variables. Each of the
   * parallel tests executes all relevant hooks.
   * @param title Group title.
   * @param callback A callback that is run immediately when calling [test.describe.parallel(title, callback)](https://playwright.dev/docs/api/class-test#test-describe-parallel). Any tests
   * added in this callback will belong to the group.
   */
  parallel: SuiteFunction & {
      /**
   * Declares a focused group of tests that could be run in parallel. This is similar to
   * [test.describe.parallel(title, callback)](https://playwright.dev/docs/api/class-test#test-describe-parallel), but
   * focuses the group. If there are some focused tests or suites, all of them will be run but nothing else.
   * @param title Group title.
   * @param callback A callback that is run immediately when calling [test.describe.parallel.only(title, callback)](https://playwright.dev/docs/api/class-test#test-describe-parallel-only).
   * Any tests added in this callback will belong to the group.
   */
  only: SuiteFunction;
    };
    /**
   * Set execution mode of execution for the enclosing scope. Can be executed either on the top level or inside a describe.
   * Configuration applies to the entire scope, regardless of whether it run before or after the test declaration.
   *
   * Learn more about the execution modes [here](https://playwright.dev/docs/test-parallel).
   *
   * Running tests in parallel:
   *
   * ```js
   * // Run all the tests in the file concurrently using parallel workers.
   * test.describe.configure({ mode: 'parallel' });
   * test('runs in parallel 1', async ({ page }) => {});
   * test('runs in parallel 2', async ({ page }) => {});
   * ```
   *
   * Running tests sequentially:
   *
   * ```js
   * // Annotate tests as inter-dependent.
   * test.describe.configure({ mode: 'serial' });
   * test('runs first', async ({ page }) => {});
   * test('runs second', async ({ page }) => {});
   * ```
   *
   * @param options
   */
  configure: (options: { mode?: 'parallel' | 'serial' }) => void;
  };
  /**
   * Declares a skipped test, similarly to
   * [test.(call)(title, testFunction)](https://playwright.dev/docs/api/class-test#test-call). Skipped test is never run.
   *
   * ```js
   * import { test, expect } from '@playwright/test';
   *
   * test.skip('broken test', async ({ page }) => {
   *   // ...
   * });
   * ```
   *
   * @param title Test title.
   * @param testFunction Test function that takes one or two arguments: an object with fixtures and optional [TestInfo].
   */
  skip(title: string, testFunction: (args: TestArgs & WorkerArgs, testInfo: TestInfo) => Promise<void> | void): void;
  /**
   * Unconditionally skip a test. Test is immediately aborted when you call
   * [test.skip()](https://playwright.dev/docs/api/class-test#test-skip-2).
   *
   * ```js
   * import { test, expect } from '@playwright/test';
   *
   * test('skipped test', async ({ page }) => {
   *   test.skip();
   *   // ...
   * });
   * ```
   *
   * Unconditionally skip all tests in a file or
   * [test.describe(title, callback)](https://playwright.dev/docs/api/class-test#test-describe-1) group:
   *
   * ```js
   * import { test, expect } from '@playwright/test';
   *
   * test.skip();
   *
   * test('skipped test 1', async ({ page }) => {
   *   // ...
   * });
   * test('skipped test 2', async ({ page }) => {
   *   // ...
   * });
   * ```
   *
   */
  skip(): void;
  /**
   * Conditionally skip a test with an optional description.
   *
   * ```js
   * import { test, expect } from '@playwright/test';
   *
   * test('skip in WebKit', async ({ page, browserName }) => {
   *   test.skip(browserName === 'webkit', 'This feature is not implemented for Mac');
   *   // ...
   * });
   * ```
   *
   * Skip from [test.beforeEach(hookFunction)](https://playwright.dev/docs/api/class-test#test-before-each) hook:
   *
   * ```js
   * import { test, expect } from '@playwright/test';
   *
   * test.beforeEach(async ({ page }) => {
   *   test.skip(process.env.APP_VERSION === 'v1', 'There are no settings in v1');
   *   await page.goto('/settings');
   * });
   * ```
   *
   * @param condition A skip condition. Test is skipped when the condition is `true`.
   * @param description Optional description that will be reflected in a test report.
   */
  skip(condition: boolean, description?: string): void;
  /**
   * Conditionally skips all tests in a file or
   * [test.describe(title, callback)](https://playwright.dev/docs/api/class-test#test-describe-1) group.
   *
   * ```js
   * import { test, expect } from '@playwright/test';
   *
   * test.skip(({ browserName }) => browserName === 'webkit');
   *
   * test('skip in WebKit 1', async ({ page }) => {
   *   // ...
   * });
   * test('skip in WebKit 2', async ({ page }) => {
   *   // ...
   * });
   * ```
   *
   * @param callback A function that returns whether to skip, based on test fixtures. Test or tests are skipped when the return value is `true`.
   * @param description Optional description that will be reflected in a test report.
   */
  skip(callback: (args: TestArgs & WorkerArgs) => boolean, description?: string): void;
  /**
   * Declares a test to be fixed, similarly to
   * [test.(call)(title, testFunction)](https://playwright.dev/docs/api/class-test#test-call). This test will not be run.
   *
   * ```js
   * import { test, expect } from '@playwright/test';
   *
   * test.fixme('test to be fixed', async ({ page }) => {
   *   // ...
   * });
   * ```
   *
   * @param title Test title.
   * @param testFunction Test function that takes one or two arguments: an object with fixtures and optional [TestInfo].
   */
  fixme(title: string, testFunction: (args: TestArgs & WorkerArgs, testInfo: TestInfo) => Promise<void> | void): void;
  /**
   * Mark a test as "fixme", with the intention to fix it. Test is immediately aborted when you call
   * [test.fixme()](https://playwright.dev/docs/api/class-test#test-fixme-2).
   *
   * ```js
   * import { test, expect } from '@playwright/test';
   *
   * test('test to be fixed', async ({ page }) => {
   *   test.fixme();
   *   // ...
   * });
   * ```
   *
   * Mark all tests in a file or [test.describe(title, callback)](https://playwright.dev/docs/api/class-test#test-describe-1)
   * group as "fixme".
   *
   * ```js
   * import { test, expect } from '@playwright/test';
   *
   * test.fixme();
   *
   * test('test to be fixed 1', async ({ page }) => {
   *   // ...
   * });
   * test('test to be fixed 2', async ({ page }) => {
   *   // ...
   * });
   * ```
   *
   */
  fixme(): void;
  /**
   * Conditionally mark a test as "fixme" with an optional description.
   *
   * ```js
   * import { test, expect } from '@playwright/test';
   *
   * test('broken in WebKit', async ({ page, browserName }) => {
   *   test.fixme(browserName === 'webkit', 'This feature is not implemented on Mac yet');
   *   // ...
   * });
   * ```
   *
   * @param condition Test is marked as "fixme" when the condition is `true`.
   * @param description Optional description that will be reflected in a test report.
   */
  fixme(condition: boolean, description?: string): void;
  /**
   * Conditionally mark all tests in a file or
   * [test.describe(title, callback)](https://playwright.dev/docs/api/class-test#test-describe-1) group as "fixme".
   *
   * ```js
   * import { test, expect } from '@playwright/test';
   *
   * test.fixme(({ browserName }) => browserName === 'webkit');
   *
   * test('broken in WebKit 1', async ({ page }) => {
   *   // ...
   * });
   * test('broken in WebKit 2', async ({ page }) => {
   *   // ...
   * });
   * ```
   *
   * @param callback A function that returns whether to mark as "fixme", based on test fixtures. Test or tests are marked as "fixme" when the return value is `true`.
   * @param description Optional description that will be reflected in a test report.
   */
  fixme(callback: (args: TestArgs & WorkerArgs) => boolean, description?: string): void;
  /**
   * Unconditonally marks a test as "should fail". Playwright Test runs this test and ensures that it is actually failing.
   * This is useful for documentation purposes to acknowledge that some functionality is broken until it is fixed.
   *
   * ```js
   * import { test, expect } from '@playwright/test';
   *
   * test('not yet ready', async ({ page }) => {
   *   test.fail();
   *   // ...
   * });
   * ```
   *
   */
  fail(): void;
  /**
   * Conditionally mark a test as "should fail" with an optional description.
   *
   * ```js
   * import { test, expect } from '@playwright/test';
   *
   * test('fail in WebKit', async ({ page, browserName }) => {
   *   test.fail(browserName === 'webkit', 'This feature is not implemented for Mac yet');
   *   // ...
   * });
   * ```
   *
   * @param condition Test is marked as "should fail" when the condition is `true`.
   * @param description Optional description that will be reflected in a test report.
   */
  fail(condition: boolean, description?: string): void;
  /**
   * Conditionally mark all tests in a file or
   * [test.describe(title, callback)](https://playwright.dev/docs/api/class-test#test-describe-1) group as "should fail".
   *
   * ```js
   * import { test, expect } from '@playwright/test';
   *
   * test.fail(({ browserName }) => browserName === 'webkit');
   *
   * test('fail in WebKit 1', async ({ page }) => {
   *   // ...
   * });
   * test('fail in WebKit 2', async ({ page }) => {
   *   // ...
   * });
   * ```
   *
   * @param callback A function that returns whether to mark as "should fail", based on test fixtures. Test or tests are marked as "should fail" when the return value is `true`.
   * @param description Optional description that will be reflected in a test report.
   */
  fail(callback: (args: TestArgs & WorkerArgs) => boolean, description?: string): void;
  /**
   * Unconditionally marks a test as "slow". Slow test will be given triple the default timeout.
   *
   * ```js
   * import { test, expect } from '@playwright/test';
   *
   * test('slow test', async ({ page }) => {
   *   test.slow();
   *   // ...
   * });
   * ```
   *
   * > NOTE: [test.slow()](https://playwright.dev/docs/api/class-test#test-slow-1) cannot be used in a `beforeAll` or
   * `afterAll` hook. Use [test.setTimeout(timeout)](https://playwright.dev/docs/api/class-test#test-set-timeout) instead.
   */
  slow(): void;
  /**
   * Conditionally mark a test as "slow" with an optional description. Slow test will be given triple the default timeout.
   *
   * ```js
   * import { test, expect } from '@playwright/test';
   *
   * test('slow in WebKit', async ({ page, browserName }) => {
   *   test.slow(browserName === 'webkit', 'This feature is slow on Mac');
   *   // ...
   * });
   * ```
   *
   * @param condition Test is marked as "slow" when the condition is `true`.
   * @param description Optional description that will be reflected in a test report.
   */
  slow(condition: boolean, description?: string): void;
  /**
   * Conditionally mark all tests in a file or
   * [test.describe(title, callback)](https://playwright.dev/docs/api/class-test#test-describe-1) group as "slow". Slow tests
   * will be given triple the default timeout.
   *
   * ```js
   * import { test, expect } from '@playwright/test';
   *
   * test.slow(({ browserName }) => browserName === 'webkit');
   *
   * test('slow in WebKit 1', async ({ page }) => {
   *   // ...
   * });
   * test('fail in WebKit 2', async ({ page }) => {
   *   // ...
   * });
   * ```
   *
   * @param callback A function that returns whether to mark as "slow", based on test fixtures. Test or tests are marked as "slow" when the return value is `true`.
   * @param description Optional description that will be reflected in a test report.
   */
  slow(callback: (args: TestArgs & WorkerArgs) => boolean, description?: string): void;
  /**
   * Changes the timeout for the test. Zero means no timeout. Learn more about [various timeouts](https://playwright.dev/docs/test-timeouts).
   *
   * ```js
   * import { test, expect } from '@playwright/test';
   *
   * test('very slow test', async ({ page }) => {
   *   test.setTimeout(120000);
   *   // ...
   * });
   * ```
   *
   * Changing timeout from a slow `beforeEach` or `afterEach` hook. Note that this affects the test timeout that is shared
   * with `beforeEach`/`afterEach` hooks.
   *
   * ```js
   * import { test, expect } from '@playwright/test';
   *
   * test.beforeEach(async ({ page }, testInfo) => {
   *   // Extend timeout for all tests running this hook by 30 seconds.
   *   test.setTimeout(testInfo.timeout + 30000);
   * });
   * ```
   *
   * Changing timeout for a `beforeAll` or `afterAll` hook. Note this affects the hook's timeout, not the test timeout.
   *
   * ```js
   * import { test, expect } from '@playwright/test';
   *
   * test.beforeAll(async () => {
   *   // Set timeout for this hook.
   *   test.setTimeout(60000);
   * });
   * ```
   *
   * Changing timeout for all tests in a
   * [test.describe(title, callback)](https://playwright.dev/docs/api/class-test#test-describe-1) group.
   *
   * ```js
   * import { test, expect } from '@playwright/test';
   *
   * test.describe('group', () => {
   *   // Applies to all tests in this group.
   *   test.setTimeout(60000);
   *
   *   test('test one', async () => { /* ... *\/ });
   *   test('test two', async () => { /* ... *\/ });
   *   test('test three', async () => { /* ... *\/ });
   * });
   * ```
   *
   * Timeout for the currently running test is available through
   * [testInfo.timeout](https://playwright.dev/docs/api/class-testinfo#test-info-timeout).
   * @param timeout Timeout in milliseconds.
   */
  setTimeout(timeout: number): void;
  /**
   * Declares a `beforeEach` hook that is executed before each test. When called in the scope of a test file, runs before
   * each test in the file. When called inside a
   * [test.describe(title, callback)](https://playwright.dev/docs/api/class-test#test-describe-1) group, runs before each
   * test in the group.  If multiple `beforeEach` hooks are added, they will run in the order of their registration.
   *
   * You can access all the same [Fixtures] as the test function itself, and also the [TestInfo] object that gives a lot of
   * useful information. For example, you can navigate the page before starting the test.
   *
   * ```js
   * // example.spec.ts
   * import { test, expect } from '@playwright/test';
   *
   * test.beforeEach(async ({ page }, testInfo) => {
   *   console.log(`Running ${testInfo.title}`);
   *   await page.goto('https://my.start.url/');
   * });
   *
   * test('my test', async ({ page }) => {
   *   expect(page.url()).toBe('https://my.start.url/');
   * });
   * ```
   *
   * You can use [test.afterEach(hookFunction)](https://playwright.dev/docs/api/class-test#test-after-each) to teardown any
   * resources set up in `beforeEach`.
   * @param hookFunction Hook function that takes one or two arguments: an object with fixtures and optional [TestInfo].
   */
  beforeEach(inner: (args: TestArgs & WorkerArgs, testInfo: TestInfo) => Promise<any> | any): void;
  /**
   * Declares an `afterEach` hook that is executed after each test. When called in the scope of a test file, runs after each
   * test in the file. When called inside a
   * [test.describe(title, callback)](https://playwright.dev/docs/api/class-test#test-describe-1) group, runs after each test
   * in the group. If multiple `afterEach` hooks are added, they will run in the order of their registration.
   *
   * You can access all the same [Fixtures] as the test function itself, and also the [TestInfo] object that gives a lot of
   * useful information. For example, you can check whether the test succeeded or failed.
   *
   * ```js
   * // example.spec.ts
   * import { test, expect } from '@playwright/test';
   *
   * test.afterEach(async ({ page }, testInfo) => {
   *   console.log(`Finished ${testInfo.title} with status ${testInfo.status}`);
   *
   *   if (testInfo.status !== testInfo.expectedStatus)
   *     console.log(`Did not run as expected, ended up at ${page.url()}`);
   * });
   *
   * test('my test', async ({ page }) => {
   *   // ...
   * });
   * ```
   *
   * @param hookFunction Hook function that takes one or two arguments: an object with fixtures and optional [TestInfo].
   */
  afterEach(inner: (args: TestArgs & WorkerArgs, testInfo: TestInfo) => Promise<any> | any): void;
  /**
   * Declares a `beforeAll` hook that is executed once per worker process before all tests. When called in the scope of a
   * test file, runs before all tests in the file. When called inside a
   * [test.describe(title, callback)](https://playwright.dev/docs/api/class-test#test-describe-1) group, runs before all
   * tests in the group. If multiple `beforeAll` hooks are added, they will run in the order of their registration.
   *
   * ```js
   * // example.spec.ts
   * import { test, expect } from '@playwright/test';
   *
   * test.beforeAll(async () => {
   *   console.log('Before tests');
   * });
   *
   * test.afterAll(async () => {
   *   console.log('After tests');
   * });
   *
   * test('my test', async ({ page }) => {
   *   // ...
   * });
   * ```
   *
   * Note that worker process is restarted on test failures, and `beforeAll` hook runs again in the new worker. Learn more
   * about [workers and failures](https://playwright.dev/docs/test-retries).
   *
   * You can use [test.afterAll(hookFunction)](https://playwright.dev/docs/api/class-test#test-after-all) to teardown any
   * resources set up in `beforeAll`.
   * @param hookFunction Hook function that takes one or two arguments: an object with worker fixtures and optional [TestInfo].
   */
  beforeAll(inner: (args: TestArgs & WorkerArgs, testInfo: TestInfo) => Promise<any> | any): void;
  /**
   * Declares an `afterAll` hook that is executed once per worker after all tests. When called in the scope of a test file,
   * runs after all tests in the file. When called inside a
   * [test.describe(title, callback)](https://playwright.dev/docs/api/class-test#test-describe-1) group, runs after all tests
   * in the group. If multiple `afterAll` hooks are added, they will run in the order of their registration.
   *
   * Note that worker process is restarted on test failures, and `afterAll` hook runs again in the new worker. Learn more
   * about [workers and failures](https://playwright.dev/docs/test-retries).
   * @param hookFunction Hook function that takes one or two arguments: an object with worker fixtures and optional [TestInfo].
   */
  afterAll(inner: (args: TestArgs & WorkerArgs, testInfo: TestInfo) => Promise<any> | any): void;
  /**
   * Specifies options or fixtures to use in a single test file or a
   * [test.describe(title, callback)](https://playwright.dev/docs/api/class-test#test-describe-1) group. Most useful to set
   * an option, for example set `locale` to configure `context` fixture. `test.use` can be called either in the global scope
   * or inside `test.describe`, it's is an error to call it within `beforeEach` or `beforeAll`.
   *
   * ```js
   * import { test, expect } from '@playwright/test';
   *
   * test.use({ locale: 'en-US' });
   *
   * test('test with locale', async ({ page }) => {
   *   // Default context and page have locale as specified
   * });
   * ```
   *
   * It is also possible to override a fixture by providing a function.
   *
   * ```js
   * import { test, expect } from '@playwright/test';
   *
   * test.use({
   *   locale: async ({}, use) => {
   *     // Read locale from some configuration file.
   *     const locale = await fs.promises.readFile('test-locale', 'utf-8');
   *     await use(locale);
   *   },
   * });
   *
   * test('test with locale', async ({ page }) => {
   *   // Default context and page have locale as specified
   * });
   * ```
   *
   * @param options An object with local options.
   */
  use(fixtures: Fixtures<{}, {}, TestArgs, WorkerArgs>): void;
  /**
   * Declares a test step.
   *
   * ```js
   * import { test, expect } from '@playwright/test';
   *
   * test('test', async ({ page }) => {
   *   await test.step('Log in', async () => {
   *     // ...
   *   });
   * });
   * ```
   *
   * The method returns value retuned by the step callback.
   *
   * ```js
   * import { test, expect } from '@playwright/test';
   *
   * test('test', async ({ page }) => {
   *   const user = await test.step('Log in', async () => {
   *     // ...
   *     return 'john';
   *   });
   *   expect(user).toBe('john');
   * });
   * ```
   *
   * @param title Step name.
   * @param body Step body.
   */
  step<T>(title: string, body: () => Promise<T>): Promise<T>;
  /**
   * `expect` function can be used to create test assertions. Read
   * [expect library documentation](https://jestjs.io/docs/expect) for more details.
   */
  expect: Expect;
  /**
   * Extends the `test` object by defining fixtures and/or options that can be used in the tests.
   *
   * First define a fixture and/or an option.
   *
   * ```js
   * import { test as base } from '@playwright/test';
   * import { TodoPage } from './todo-page';
   *
   * export type Options = { defaultItem: string };
   *
   * // Extend basic test by providing a "defaultItem" option and a "todoPage" fixture.
   * export const test = base.extend<Options & { todoPage: TodoPage }>({
   *   // Define an option and provide a default value.
   *   // We can later override it in the config.
   *   defaultItem: ['Do stuff', { option: true }],
   *
   *   // Define a fixture. Note that it can use built-in fixture "page"
   *   // and a new option "defaultItem".
   *   todoPage: async ({ page, defaultItem }, use) => {
   *     const todoPage = new TodoPage(page);
   *     await todoPage.goto();
   *     await todoPage.addToDo(defaultItem);
   *     await use(todoPage);
   *     await todoPage.removeAll();
   *   },
   * });
   * ```
   *
   * Then use the fixture in the test.
   *
   * ```js
   * // example.spec.ts
   * import { test } from './my-test';
   *
   * test('test 1', async ({ todoPage }) => {
   *   await todoPage.addToDo('my todo');
   *   // ...
   * });
   * ```
   *
   * Configure the option in config file.
   *
   * ```js
   * // playwright.config.ts
   * import type { PlaywrightTestConfig } from '@playwright/test';
   * import { Options } from './my-test';
   *
   * const config: PlaywrightTestConfig<Options> = {
   *   projects: [
   *     {
   *       name: 'shopping',
   *       use: { defaultItem: 'Buy milk' },
   *     },
   *     {
   *       name: 'wellbeing',
   *       use: { defaultItem: 'Exercise!' },
   *     },
   *   ]
   * };
   * export default config;
   * ```
   *
   * Learn more about [fixtures](https://playwright.dev/docs/test-fixtures) and [parametrizing tests](https://playwright.dev/docs/test-parameterize).
   * @param fixtures An object containing fixtures and/or options. Learn more about [fixtures format](https://playwright.dev/docs/test-fixtures).
   */
  extend<T extends KeyValue, W extends KeyValue = {}>(fixtures: Fixtures<T, W, TestArgs, WorkerArgs>): TestType<TestArgs & T, WorkerArgs & W>;
  /**
   * Returns information about the currently running test. This method can only be called during the test execution,
   * otherwise it throws.
   */
  info(): TestInfo;
}

type KeyValue = { [key: string]: any };
export type TestFixture<R, Args extends KeyValue> = (args: Args, use: (r: R) => Promise<void>, testInfo: TestInfo) => any;
export type WorkerFixture<R, Args extends KeyValue> = (args: Args, use: (r: R) => Promise<void>, workerInfo: WorkerInfo) => any;
type TestFixtureValue<R, Args extends KeyValue> = Exclude<R, Function> | TestFixture<R, Args>;
type WorkerFixtureValue<R, Args extends KeyValue> = Exclude<R, Function> | WorkerFixture<R, Args>;
export type Fixtures<T extends KeyValue = {}, W extends KeyValue = {}, PT extends KeyValue = {}, PW extends KeyValue = {}> = {
  [K in keyof PW]?: WorkerFixtureValue<PW[K], W & PW> | [WorkerFixtureValue<PW[K], W & PW>, { scope: 'worker', timeout?: number | undefined }];
} & {
  [K in keyof PT]?: TestFixtureValue<PT[K], T & W & PT & PW> | [TestFixtureValue<PT[K], T & W & PT & PW>, { scope: 'test', timeout?: number | undefined }];
} & {
  [K in keyof W]?: [WorkerFixtureValue<W[K], W & PW>, { scope: 'worker', auto?: boolean, option?: boolean, timeout?: number | undefined }];
} & {
  [K in keyof T]?: TestFixtureValue<T[K], T & W & PT & PW> | [TestFixtureValue<T[K], T & W & PT & PW>, { scope?: 'test', auto?: boolean, option?: boolean, timeout?: number | undefined }];
};

type BrowserName = 'chromium' | 'firefox' | 'webkit';
type BrowserChannel = Exclude<LaunchOptions['channel'], undefined>;
type ColorScheme = Exclude<BrowserContextOptions['colorScheme'], undefined>;
type ExtraHTTPHeaders = Exclude<BrowserContextOptions['extraHTTPHeaders'], undefined>;
type Proxy = Exclude<BrowserContextOptions['proxy'], undefined>;
type StorageState = Exclude<BrowserContextOptions['storageState'], undefined>;
type ServiceWorkerPolicy = Exclude<BrowserContextOptions['serviceWorkers'], undefined>;
type ConnectOptions = {
  /**
   * A browser websocket endpoint to connect to.
   */
  wsEndpoint: string;

  /**
   * Additional HTTP headers to be sent with web socket connect request.
   */
  headers?: { [key: string]: string; };

  /**
   * Timeout in milliseconds for the connection to be established. Optional, defaults to no timeout.
   */
  timeout?: number;
};

/**
 * Playwright Test provides many options to configure test environment, [Browser], [BrowserContext] and more.
 *
 * These options are usually provided in the [configuration file](https://playwright.dev/docs/test-configuration) through
 * [testConfig.use](https://playwright.dev/docs/api/class-testconfig#test-config-use) and
 * [testProject.use](https://playwright.dev/docs/api/class-testproject#test-project-use).
 *
 * ```js
 * import type { PlaywrightTestConfig } from '@playwright/test';
 * const config: PlaywrightTestConfig = {
 *   use: {
 *     headless: false,
 *     viewport: { width: 1280, height: 720 },
 *     ignoreHTTPSErrors: true,
 *     video: 'on-first-retry',
 *   },
 * };
 * export default config;
 * ```
 *
 * Alternatively, with [test.use(options)](https://playwright.dev/docs/api/class-test#test-use) you can override some
 * options for a file.
 *
 * ```js
 * // example.spec.ts
 * import { test, expect } from '@playwright/test';
 *
 * // Run tests in this file with portrait-like viewport.
 * test.use({ viewport: { width: 600, height: 900 } });
 *
 * test('my portrait test', async ({ page }) => {
 *   // ...
 * });
 * ```
 *
 */
export interface PlaywrightWorkerOptions {
  /**
   * Name of the browser that runs tests. Defaults to `'chromium'`. Most of the time you should set `browserName` in your
   * [TestConfig]:
   *
   * ```js
   * // playwright.config.ts
   * import { type PlaywrightTestConfig, devices } from '@playwright/test';
   *
   * const config: PlaywrightTestConfig = {
   *   use: {
   *     browserName: 'firefox',
   *   },
   * };
   * export default config;
   * ```
   *
   */
  browserName: BrowserName;
  defaultBrowserType: BrowserName;
  /**
   * Whether to run browser in headless mode. More details for
   * [Chromium](https://developers.google.com/web/updates/2017/04/headless-chrome) and
   * [Firefox](https://developer.mozilla.org/en-US/docs/Mozilla/Firefox/Headless_mode). Defaults to `true` unless the
   * `devtools` option is `true`.
   */
  headless: boolean | undefined;
  /**
   * Browser distribution channel.  Supported values are "chrome", "chrome-beta", "chrome-dev", "chrome-canary", "msedge",
   * "msedge-beta", "msedge-dev", "msedge-canary". Read more about using
   * [Google Chrome and Microsoft Edge](https://playwright.dev/docs/browsers#google-chrome--microsoft-edge).
   */
  channel: BrowserChannel | undefined;
  /**
   * Options used to launch the browser, as passed to
   * [browserType.launch([options])](https://playwright.dev/docs/api/class-browsertype#browser-type-launch). Specific options
   * [testOptions.headless](https://playwright.dev/docs/api/class-testoptions#test-options-headless) and
   * [testOptions.channel](https://playwright.dev/docs/api/class-testoptions#test-options-channel) take priority over this.
   */
  launchOptions: LaunchOptions;
  /**
   * When connect options are specified, default
   * [fixtures.browser](https://playwright.dev/docs/api/class-fixtures#fixtures-browser),
   * [fixtures.context](https://playwright.dev/docs/api/class-fixtures#fixtures-context) and
   * [fixtures.page](https://playwright.dev/docs/api/class-fixtures#fixtures-page) use the remote browser instead of
   * launching a browser locally, and any launch options like
   * [testOptions.headless](https://playwright.dev/docs/api/class-testoptions#test-options-headless) or
   * [testOptions.channel](https://playwright.dev/docs/api/class-testoptions#test-options-channel) are ignored.
   */
  connectOptions: ConnectOptions | undefined;
  /**
   * Whether to automatically capture a screenshot after each test. Defaults to `'off'`.
   * - `'off'`: Do not capture screenshots.
   * - `'on'`: Capture screenshot after each test.
   * - `'only-on-failure'`: Capture screenshot after each test failure.
   *
   * Learn more about [automatic screenshots](https://playwright.dev/docs/test-configuration#automatic-screenshots).
   */
  screenshot: 'off' | 'on' | 'only-on-failure';
  /**
   * Whether to record trace for each test. Defaults to `'off'`.
   * - `'off'`: Do not record trace.
   * - `'on'`: Record trace for each test.
   * - `'retain-on-failure'`: Record trace for each test, but remove all traces from successful test runs.
   * - `'on-first-retry'`: Record trace only when retrying a test for the first time.
   *
   * For more control, pass an object that specifies `mode` and trace features to enable.
   *
   * Learn more about [recording trace](https://playwright.dev/docs/test-configuration#record-test-trace).
   */
  trace: TraceMode | /** deprecated */ 'retry-with-trace' | { mode: TraceMode, snapshots?: boolean, screenshots?: boolean, sources?: boolean };
  /**
   * Whether to record video for each test. Defaults to `'off'`.
   * - `'off'`: Do not record video.
   * - `'on'`: Record video for each test.
   * - `'retain-on-failure'`: Record video for each test, but remove all videos from successful test runs.
   * - `'on-first-retry'`: Record video only when retrying a test for the first time.
   *
   * To control video size, pass an object with `mode` and `size` properties. If video size is not specified, it will be
   * equal to [testOptions.viewport](https://playwright.dev/docs/api/class-testoptions#test-options-viewport) scaled down to
   * fit into 800x800. If `viewport` is not configured explicitly the video size defaults to 800x450. Actual picture of each
   * page will be scaled down if necessary to fit the specified size.
   *
   * Learn more about [recording video](https://playwright.dev/docs/test-configuration#record-video).
   */
  video: VideoMode | /** deprecated */ 'retry-with-video' | { mode: VideoMode, size?: ViewportSize };
}

export type TraceMode = 'off' | 'on' | 'retain-on-failure' | 'on-first-retry';
export type VideoMode = 'off' | 'on' | 'retain-on-failure' | 'on-first-retry';

/**
 * Playwright Test provides many options to configure test environment, [Browser], [BrowserContext] and more.
 *
 * These options are usually provided in the [configuration file](https://playwright.dev/docs/test-configuration) through
 * [testConfig.use](https://playwright.dev/docs/api/class-testconfig#test-config-use) and
 * [testProject.use](https://playwright.dev/docs/api/class-testproject#test-project-use).
 *
 * ```js
 * import type { PlaywrightTestConfig } from '@playwright/test';
 * const config: PlaywrightTestConfig = {
 *   use: {
 *     headless: false,
 *     viewport: { width: 1280, height: 720 },
 *     ignoreHTTPSErrors: true,
 *     video: 'on-first-retry',
 *   },
 * };
 * export default config;
 * ```
 *
 * Alternatively, with [test.use(options)](https://playwright.dev/docs/api/class-test#test-use) you can override some
 * options for a file.
 *
 * ```js
 * // example.spec.ts
 * import { test, expect } from '@playwright/test';
 *
 * // Run tests in this file with portrait-like viewport.
 * test.use({ viewport: { width: 600, height: 900 } });
 *
 * test('my portrait test', async ({ page }) => {
 *   // ...
 * });
 * ```
 *
 */
export interface PlaywrightTestOptions {
  /**
   * Whether to automatically download all the attachments. Defaults to `true` where all the downloads are accepted.
   */
  acceptDownloads: boolean | undefined;
  /**
   * Toggles bypassing page's Content-Security-Policy.
   */
  bypassCSP: boolean | undefined;
  /**
   * Emulates `'prefers-colors-scheme'` media feature, supported values are `'light'`, `'dark'`, `'no-preference'`. See
   * [page.emulateMedia([options])](https://playwright.dev/docs/api/class-page#page-emulate-media) for more details. Defaults
   * to `'light'`.
   */
  colorScheme: ColorScheme | undefined;
  /**
   * Specify device scale factor (can be thought of as dpr). Defaults to `1`.
   */
  deviceScaleFactor: number | undefined;
  /**
   * An object containing additional HTTP headers to be sent with every request.
   */
  extraHTTPHeaders: ExtraHTTPHeaders | undefined;
  geolocation: Geolocation | undefined;
  /**
   * Specifies if viewport supports touch events. Defaults to false.
   */
  hasTouch: boolean | undefined;
  /**
   * Credentials for [HTTP authentication](https://developer.mozilla.org/en-US/docs/Web/HTTP/Authentication).
   */
  httpCredentials: HTTPCredentials | undefined;
  /**
   * Whether to ignore HTTPS errors when sending network requests. Defaults to `false`.
   */
  ignoreHTTPSErrors: boolean | undefined;
  /**
   * Whether the `meta viewport` tag is taken into account and touch events are enabled. Defaults to `false`. Not supported
   * in Firefox.
   */
  isMobile: boolean | undefined;
  /**
   * Whether or not to enable JavaScript in the context. Defaults to `true`.
   */
  javaScriptEnabled: boolean | undefined;
  /**
   * Specify user locale, for example `en-GB`, `de-DE`, etc. Locale will affect `navigator.language` value, `Accept-Language`
   * request header value as well as number and date formatting rules.
   */
  locale: string | undefined;
  /**
   * Whether to emulate network being offline. Defaults to `false`.
   */
  offline: boolean | undefined;
  /**
   * A list of permissions to grant to all pages in this context. See
   * [browserContext.grantPermissions(permissions[, options])](https://playwright.dev/docs/api/class-browsercontext#browser-context-grant-permissions)
   * for more details.
   */
  permissions: string[] | undefined;
  /**
   * Network proxy settings.
   */
  proxy: Proxy | undefined;
  /**
   * Populates context with given storage state. This option can be used to initialize context with logged-in information
   * obtained via
   * [browserContext.storageState([options])](https://playwright.dev/docs/api/class-browsercontext#browser-context-storage-state).
   * Either a path to the file with saved storage, or an object with the following fields:
   */
  storageState: StorageState | undefined;
  /**
   * Changes the timezone of the context. See
   * [ICU's metaZones.txt](https://cs.chromium.org/chromium/src/third_party/icu/source/data/misc/metaZones.txt?rcl=faee8bc70570192d82d2978a71e2a615788597d1)
   * for a list of supported timezone IDs.
   */
  timezoneId: string | undefined;
  /**
   * Specific user agent to use in this context.
   */
  userAgent: string | undefined;
  /**
   * Emulates consistent viewport for each page. Defaults to an 1280x720 viewport. `null` disables the default viewport.
   */
  viewport: ViewportSize | null | undefined;
  /**
   * When using [page.goto(url[, options])](https://playwright.dev/docs/api/class-page#page-goto),
   * [page.route(url, handler[, options])](https://playwright.dev/docs/api/class-page#page-route),
   * [page.waitForURL(url[, options])](https://playwright.dev/docs/api/class-page#page-wait-for-url),
   * [page.waitForRequest(urlOrPredicate[, options])](https://playwright.dev/docs/api/class-page#page-wait-for-request), or
   * [page.waitForResponse(urlOrPredicate[, options])](https://playwright.dev/docs/api/class-page#page-wait-for-response) it
   * takes the base URL in consideration by using the [`URL()`](https://developer.mozilla.org/en-US/docs/Web/API/URL/URL)
   * constructor for building the corresponding URL. Examples:
   * - baseURL: `http://localhost:3000` and navigating to `/bar.html` results in `http://localhost:3000/bar.html`
   * - baseURL: `http://localhost:3000/foo/` and navigating to `./bar.html` results in `http://localhost:3000/foo/bar.html`
   * - baseURL: `http://localhost:3000/foo` (without trailing slash) and navigating to `./bar.html` results in
   *   `http://localhost:3000/bar.html`
   */
  baseURL: string | undefined;
  /**
   * Options used to create the context, as passed to
   * [browser.newContext([options])](https://playwright.dev/docs/api/class-browser#browser-new-context). Specific options
   * like [testOptions.viewport](https://playwright.dev/docs/api/class-testoptions#test-options-viewport) take priority over
   * this.
   */
  contextOptions: BrowserContextOptions;
  /**
   * Default timeout for each Playwright action in milliseconds, defaults to 0 (no timeout).
   *
   * This is a default timeout for all Playwright actions, same as configured via
   * [page.setDefaultTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-timeout).
   *
   * Learn more about [various timeouts](https://playwright.dev/docs/test-timeouts).
   */
  actionTimeout: number | undefined;
  /**
   * Timeout for each navigation action in milliseconds. Defaults to 0 (no timeout).
   *
   * This is a default navigation timeout, same as configured via
   * [page.setDefaultNavigationTimeout(timeout)](https://playwright.dev/docs/api/class-page#page-set-default-navigation-timeout).
   *
   * Learn more about [various timeouts](https://playwright.dev/docs/test-timeouts).
   */
  navigationTimeout: number | undefined;
  /**
   * Whether to allow sites to register Service workers. Defaults to `'allow'`.
   * - `'allow'`: [Service Workers](https://developer.mozilla.org/en-US/docs/Web/API/Service_Worker_API) can be registered.
   * - `'block'`: Playwright will block all registration of Service Workers.
   */
  serviceWorkers: ServiceWorkerPolicy | undefined;
}


/**
 * Playwright Test is based on the concept of the [test fixtures](https://playwright.dev/docs/test-fixtures). Test fixtures are used to establish
 * environment for each test, giving the test everything it needs and nothing else.
 *
 * Playwright Test looks at each test declaration, analyses the set of fixtures the test needs and prepares those fixtures
 * specifically for the test. Values prepared by the fixtures are merged into a single object that is available to the
 * `test`, hooks, annotations and other fixtures as a first parameter.
 *
 * ```js
 * import { test, expect } from '@playwright/test';
 *
 * test('basic test', async ({ page }) => {
 *   // ...
 * });
 * ```
 *
 * Given the test above, Playwright Test will set up the `page` fixture before running the test, and tear it down after the
 * test has finished. `page` fixture provides a [Page] object that is available to the test.
 *
 * Playwright Test comes with builtin fixtures listed below, and you can add your own fixtures as well. Playwright Test
 * also [provides options][TestOptions] to  configure
 * [fixtures.browser](https://playwright.dev/docs/api/class-fixtures#fixtures-browser),
 * [fixtures.context](https://playwright.dev/docs/api/class-fixtures#fixtures-context) and
 * [fixtures.page](https://playwright.dev/docs/api/class-fixtures#fixtures-page).
 */
export interface PlaywrightWorkerArgs {
  playwright: typeof import('playwright-core');
  /**
   * [Browser] instance is shared between all tests in the [same worker](https://playwright.dev/docs/test-parallel) - this makes testing efficient.
   * However, each test runs in an isolated [BrowserContext]  and gets a fresh environment.
   *
   * Learn how to [configure browser](https://playwright.dev/docs/test-configuration) and see [available options][TestOptions].
   */
  browser: Browser;
}

/**
 * Playwright Test is based on the concept of the [test fixtures](https://playwright.dev/docs/test-fixtures). Test fixtures are used to establish
 * environment for each test, giving the test everything it needs and nothing else.
 *
 * Playwright Test looks at each test declaration, analyses the set of fixtures the test needs and prepares those fixtures
 * specifically for the test. Values prepared by the fixtures are merged into a single object that is available to the
 * `test`, hooks, annotations and other fixtures as a first parameter.
 *
 * ```js
 * import { test, expect } from '@playwright/test';
 *
 * test('basic test', async ({ page }) => {
 *   // ...
 * });
 * ```
 *
 * Given the test above, Playwright Test will set up the `page` fixture before running the test, and tear it down after the
 * test has finished. `page` fixture provides a [Page] object that is available to the test.
 *
 * Playwright Test comes with builtin fixtures listed below, and you can add your own fixtures as well. Playwright Test
 * also [provides options][TestOptions] to  configure
 * [fixtures.browser](https://playwright.dev/docs/api/class-fixtures#fixtures-browser),
 * [fixtures.context](https://playwright.dev/docs/api/class-fixtures#fixtures-context) and
 * [fixtures.page](https://playwright.dev/docs/api/class-fixtures#fixtures-page).
 */
export interface PlaywrightTestArgs {
  /**
   * Isolated [BrowserContext] instance, created for each test. Since contexts are isolated between each other, every test
   * gets a fresh environment, even when multiple tests run in a single [Browser] for maximum efficiency.
   *
   * Learn how to [configure context](https://playwright.dev/docs/test-configuration) and see [available options][TestOptions].
   *
   * Default [fixtures.page](https://playwright.dev/docs/api/class-fixtures#fixtures-page) belongs to this context.
   */
  context: BrowserContext;
  /**
   * Isolated [Page] instance, created for each test. Pages are isolated between tests due to
   * [fixtures.context](https://playwright.dev/docs/api/class-fixtures#fixtures-context) isolation.
   *
   * This is the most common fixture used in a test.
   *
   * ```js
   * import { test, expect } from '@playwright/test';
   *
   * test('basic test', async ({ page }) => {
   *   await page.goto('/signin');
   *   await page.locator('#username').fill('User');
   *   await page.locator('#password').fill('pwd');
   *   await page.locator('text=Sign in').click();
   *   // ...
   * });
   * ```
   *
   */
  page: Page;
  /**
   * Isolated [APIRequestContext] instance for each test.
   *
   * ```js
   * import { test, expect } from '@playwright/test';
   *
   * test('basic test', async ({ request }) => {
   *   await request.post('/signin', {
   *     data: {
   *       username: 'user',
   *       password: 'password'
   *     }
   *   });
   *   // ...
   * });
   * ```
   *
   */
  request: APIRequestContext;
}

export type PlaywrightTestProject<TestArgs = {}, WorkerArgs = {}> = Project<PlaywrightTestOptions & TestArgs, PlaywrightWorkerOptions & WorkerArgs>;
export type PlaywrightTestConfig<TestArgs = {}, WorkerArgs = {}> = Config<PlaywrightTestOptions & TestArgs, PlaywrightWorkerOptions & WorkerArgs>;

import type * as expectType from './expect-types';
import type { Suite } from './testReporter';

type AsymmetricMatcher = Record<string, any>;

type AsymmetricMatchers = {
  any(sample: unknown): AsymmetricMatcher;
  anything(): AsymmetricMatcher;
  arrayContaining(sample: Array<unknown>): AsymmetricMatcher;
  closeTo(sample: number, precision?: number): AsymmetricMatcher;
  objectContaining(sample: Record<string, unknown>): AsymmetricMatcher;
  stringContaining(sample: string): AsymmetricMatcher;
  stringMatching(sample: string | RegExp): AsymmetricMatcher;
}

type Inverse<Matchers> = {
  /**
   * Inverse next matcher. If you know how to test something, `.not` lets you test its opposite.
   */
  not: Matchers;
};

type IfAny<T, Y, N> = 0 extends (1 & T) ? Y : N;
type ExtraMatchers<T, Type, Matchers> = T extends Type ? Matchers : IfAny<T, Matchers, {}>;

type BaseMatchers<R, T> = expectType.Matchers<R> & PlaywrightTest.Matchers<R, T>;

type MakeMatchers<R, T> = BaseMatchers<R, T> & {
    /**
     * If you know how to test something, `.not` lets you test its opposite.
     */
    not: MakeMatchers<R, T>;
    /**
     * Use resolves to unwrap the value of a fulfilled promise so any other
     * matcher can be chained. If the promise is rejected the assertion fails.
     */
    resolves: MakeMatchers<Promise<R>, Awaited<T>>;
    /**
    * Unwraps the reason of a rejected promise so any other matcher can be chained.
    * If the promise is fulfilled the assertion fails.
    */
    rejects: MakeMatchers<Promise<R>, Awaited<T>>;
  } & ScreenshotAssertions &
  ExtraMatchers<T, Page, PageAssertions> &
  ExtraMatchers<T, Locator, LocatorAssertions> &
  ExtraMatchers<T, APIResponse, APIResponseAssertions>;

type BaseExpect = {
  // Removed following methods because they rely on a test-runner integration from Jest which we don't support:
  // assertions(numberOfAssertions: number): void;
  // extractExpectedAssertionsErrors(): ExpectedAssertionsErrors;
  // hasAssertions(): void;
  extend(matchers: any): void;
  getState(): expectType.MatcherState;
  setState(state: Partial<expectType.MatcherState>): void;
}

export type Expect = {
  <T = unknown>(actual: T, messageOrOptions?: string | { message?: string }): MakeMatchers<void, T>;
  soft: <T = unknown>(actual: T, messageOrOptions?: string | { message?: string }) => MakeMatchers<void, T>;
  poll: <T = unknown>(actual: () => T | Promise<T>, messageOrOptions?: string | { message?: string, timeout?: number, intervals?: number[] }) => BaseMatchers<Promise<void>, T> & {
    /**
     * If you know how to test something, `.not` lets you test its opposite.
     */
     not: BaseMatchers<Promise<void>, T>;
  };
} & BaseExpect &
  AsymmetricMatchers &
  Inverse<Omit<AsymmetricMatchers, 'any' | 'anything'>>;

type Awaited<T> = T extends PromiseLike<infer U> ? U : T;

// --- BEGINGLOBAL ---
declare global {
  export namespace PlaywrightTest {
    export interface Matchers<R, T = unknown> {
    }
  }
}
// --- ENDGLOBAL ---

/**
 * These tests are executed in Playwright environment that launches the browser
 * and provides a fresh page to each test.
 */
export const test: TestType<PlaywrightTestArgs & PlaywrightTestOptions, PlaywrightWorkerArgs & PlaywrightWorkerOptions>;
export default test;

export const _baseTest: TestType<{}, {}>;
export const expect: Expect;

// This is required to not export everything by default. See https://github.com/Microsoft/TypeScript/issues/19545#issuecomment-340490459
export {};


/**
 * The [APIResponseAssertions] class provides assertion methods that can be used to make assertions about the [APIResponse]
 * in the tests. A new instance of [APIResponseAssertions] is created by calling
 * [expect(response)](https://playwright.dev/docs/api/class-playwrightassertions#playwright-assertions-expect-api-response):
 *
 * ```js
 * import { test, expect } from '@playwright/test';
 *
 * test('navigates to login', async ({ page }) => {
 *   // ...
 *   const response = await page.request.get('https://playwright.dev');
 *   await expect(response).toBeOK();
 * });
 * ```
 *
 */
interface APIResponseAssertions {
  /**
   * Makes the assertion check for the opposite condition. For example, this code tests that the response status is not
   * successful:
   *
   * ```js
   * await expect(response).not.toBeOK();
   * ```
   *
   */
  not: APIResponseAssertions;

  /**
   * Ensures the response status code is within [200..299] range.
   *
   * ```js
   * await expect(response).toBeOK();
   * ```
   *
   */
  toBeOK(): Promise<void>;
}

/**
 * The [LocatorAssertions] class provides assertion methods that can be used to make assertions about the [Locator] state
 * in the tests. A new instance of [LocatorAssertions] is created by calling
 * [expect(locator)](https://playwright.dev/docs/api/class-playwrightassertions#playwright-assertions-expect-locator):
 *
 * ```js
 * import { test, expect } from '@playwright/test';
 *
 * test('status becomes submitted', async ({ page }) => {
 *   // ...
 *   await page.locator('#submit-button').click();
 *   await expect(page.locator('.status')).toHaveText('Submitted');
 * });
 * ```
 *
 */
interface LocatorAssertions {
  /**
   * Makes the assertion check for the opposite condition. For example, this code tests that the Locator doesn't contain text
   * `"error"`:
   *
   * ```js
   * await expect(locator).not.toContainText('error');
   * ```
   *
   */
  not: LocatorAssertions;

  /**
   * Ensures the [Locator] points to a checked input.
   *
   * ```js
   * const locator = page.locator('.subscribe');
   * await expect(locator).toBeChecked();
   * ```
   *
   * @param options
   */
  toBeChecked(options?: {
    checked?: boolean;

    /**
     * Time to retry the assertion for. Defaults to `timeout` in `TestConfig.expect`.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Ensures the [Locator] points to a disabled element. Element is disabled if it has "disabled" attribute or is disabled
   * via ['aria-disabled'](https://developer.mozilla.org/en-US/docs/Web/Accessibility/ARIA/Attributes/aria-disabled). Note
   * that only native control elements such as HTML `button`, `input`, `select`, `textarea`, `option`, `optgroup` can be
   * disabled by setting "disabled" attribute. "disabled" attribute on other elements is ignored by the browser.
   *
   * ```js
   * const locator = page.locator('button.submit');
   * await expect(locator).toBeDisabled();
   * ```
   *
   * @param options
   */
  toBeDisabled(options?: {
    /**
     * Time to retry the assertion for. Defaults to `timeout` in `TestConfig.expect`.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Ensures the [Locator] points to an editable element.
   *
   * ```js
   * const locator = page.locator('input');
   * await expect(locator).toBeEditable();
   * ```
   *
   * @param options
   */
  toBeEditable(options?: {
    /**
     * Time to retry the assertion for. Defaults to `timeout` in `TestConfig.expect`.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Ensures the [Locator] points to an empty editable element or to a DOM node that has no text.
   *
   * ```js
   * const locator = page.locator('div.warning');
   * await expect(locator).toBeEmpty();
   * ```
   *
   * @param options
   */
  toBeEmpty(options?: {
    /**
     * Time to retry the assertion for. Defaults to `timeout` in `TestConfig.expect`.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Ensures the [Locator] points to an enabled element.
   *
   * ```js
   * const locator = page.locator('button.submit');
   * await expect(locator).toBeEnabled();
   * ```
   *
   * @param options
   */
  toBeEnabled(options?: {
    /**
     * Time to retry the assertion for. Defaults to `timeout` in `TestConfig.expect`.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Ensures the [Locator] points to a focused DOM node.
   *
   * ```js
   * const locator = page.locator('input');
   * await expect(locator).toBeFocused();
   * ```
   *
   * @param options
   */
  toBeFocused(options?: {
    /**
     * Time to retry the assertion for. Defaults to `timeout` in `TestConfig.expect`.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Ensures the [Locator] points to a hidden DOM node, which is the opposite of [visible](https://playwright.dev/docs/api/actionability#visible).
   *
   * ```js
   * const locator = page.locator('.my-element');
   * await expect(locator).toBeHidden();
   * ```
   *
   * @param options
   */
  toBeHidden(options?: {
    /**
     * Time to retry the assertion for. Defaults to `timeout` in `TestConfig.expect`.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Ensures the [Locator] points to a [visible](https://playwright.dev/docs/api/actionability#visible) DOM node.
   *
   * ```js
   * const locator = page.locator('.my-element');
   * await expect(locator).toBeVisible();
   * ```
   *
   * @param options
   */
  toBeVisible(options?: {
    /**
     * Time to retry the assertion for. Defaults to `timeout` in `TestConfig.expect`.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Ensures the [Locator] points to an element that contains the given text. You can use regular expressions for the value
   * as well.
   *
   * ```js
   * const locator = page.locator('.title');
   * await expect(locator).toContainText('substring');
   * await expect(locator).toContainText(/\d messages/);
   * ```
   *
   * If you pass an array as an expected value, the expectations are:
   * 1. Locator resolves to a list of elements.
   * 1. Elements from a **subset** of this list contain text from the expected array, respectively.
   * 1. The matching subset of elements has the same order as the expected array.
   * 1. Each text value from the expected array is matched by some element from the list.
   *
   * For example, consider the following list:
   *
   * ```html
   * <ul>
   *   <li>Item Text 1</li>
   *   <li>Item Text 2</li>
   *   <li>Item Text 3</li>
   * </ul>
   * ```
   *
   * Let's see how we can use the assertion:
   *
   * ```js
   * // ✓ Contains the right items in the right order
   * await expect(page.locator('ul > li')).toContainText(['Text 1', 'Text 3']);
   *
   * // ✖ Wrong order
   * await expect(page.locator('ul > li')).toContainText(['Text 3', 'Text 2']);
   *
   * // ✖ No item contains this text
   * await expect(page.locator('ul > li')).toContainText(['Some 33']);
   *
   * // ✖ Locator points to the outer list element, not to the list items
   * await expect(page.locator('ul')).toContainText(['Text 3']);
   * ```
   *
   * @param expected Expected substring or RegExp or a list of those.
   * @param options
   */
  toContainText(expected: string|RegExp|Array<string|RegExp>, options?: {
    /**
     * Whether to perform case-insensitive match. `ignoreCase` option takes precedence over the corresponding regular
     * expression flag if specified.
     */
    ignoreCase?: boolean;

    /**
     * Time to retry the assertion for. Defaults to `timeout` in `TestConfig.expect`.
     */
    timeout?: number;

    /**
     * Whether to use `element.innerText` instead of `element.textContent` when retrieving DOM node text.
     */
    useInnerText?: boolean;
  }): Promise<void>;

  /**
   * Ensures the [Locator] points to an element with given attribute.
   *
   * ```js
   * const locator = page.locator('input');
   * await expect(locator).toHaveAttribute('type', 'text');
   * ```
   *
   * @param name Attribute name.
   * @param value Expected attribute value.
   * @param options
   */
  toHaveAttribute(name: string, value: string|RegExp, options?: {
    /**
     * Time to retry the assertion for. Defaults to `timeout` in `TestConfig.expect`.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Ensures the [Locator] points to an element with given CSS classes. This needs to be a full match or using a relaxed
   * regular expression.
   *
   * ```html
   * <div class='selected row' id='component'></div>
   * ```
   *
   * ```js
   * const locator = page.locator('#component');
   * await expect(locator).toHaveClass(/selected/);
   * await expect(locator).toHaveClass('selected row');
   * ```
   *
   * Note that if array is passed as an expected value, entire lists of elements can be asserted:
   *
   * ```js
   * const locator = page.locator('list > .component');
   * await expect(locator).toHaveClass(['component', 'component selected', 'component']);
   * ```
   *
   * @param expected Expected class or RegExp or a list of those.
   * @param options
   */
  toHaveClass(expected: string|RegExp|Array<string|RegExp>, options?: {
    /**
     * Time to retry the assertion for. Defaults to `timeout` in `TestConfig.expect`.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Ensures the [Locator] resolves to an exact number of DOM nodes.
   *
   * ```js
   * const list = page.locator('list > .component');
   * await expect(list).toHaveCount(3);
   * ```
   *
   * @param count Expected count.
   * @param options
   */
  toHaveCount(count: number, options?: {
    /**
     * Time to retry the assertion for. Defaults to `timeout` in `TestConfig.expect`.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Ensures the [Locator] resolves to an element with the given computed CSS style.
   *
   * ```js
   * const locator = page.locator('button');
   * await expect(locator).toHaveCSS('display', 'flex');
   * ```
   *
   * @param name CSS property name.
   * @param value CSS property value.
   * @param options
   */
  toHaveCSS(name: string, value: string|RegExp, options?: {
    /**
     * Time to retry the assertion for. Defaults to `timeout` in `TestConfig.expect`.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Ensures the [Locator] points to an element with the given DOM Node ID.
   *
   * ```js
   * const locator = page.locator('input');
   * await expect(locator).toHaveId('lastname');
   * ```
   *
   * @param id Element id.
   * @param options
   */
  toHaveId(id: string|RegExp, options?: {
    /**
     * Time to retry the assertion for. Defaults to `timeout` in `TestConfig.expect`.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Ensures the [Locator] points to an element with given JavaScript property. Note that this property can be of a primitive
   * type as well as a plain serializable JavaScript object.
   *
   * ```js
   * const locator = page.locator('.component');
   * await expect(locator).toHaveJSProperty('loaded', true);
   * ```
   *
   * @param name Property name.
   * @param value Property value.
   * @param options
   */
  toHaveJSProperty(name: string, value: any, options?: {
    /**
     * Time to retry the assertion for. Defaults to `timeout` in `TestConfig.expect`.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * This function will wait until two consecutive locator screenshots yield the same result, and then compare the last
   * screenshot with the expectation.
   *
   * ```js
   * const locator = page.locator('button');
   * await expect(locator).toHaveScreenshot('image.png');
   * ```
   *
   * @param name Snapshot name.
   * @param options
   */
  toHaveScreenshot(name: string|Array<string>, options?: {
    /**
     * When set to `"disabled"`, stops CSS animations, CSS transitions and Web Animations. Animations get different treatment
     * depending on their duration:
     * - finite animations are fast-forwarded to completion, so they'll fire `transitionend` event.
     * - infinite animations are canceled to initial state, and then played over after the screenshot.
     *
     * Defaults to `"disabled"` that disables animations.
     */
    animations?: "disabled"|"allow";

    /**
     * When set to `"hide"`, screenshot will hide text caret. When set to `"initial"`, text caret behavior will not be changed.
     * Defaults to `"hide"`.
     */
    caret?: "hide"|"initial";

    /**
     * Specify locators that should be masked when the screenshot is taken. Masked elements will be overlaid with a pink box
     * `#FF00FF` that completely covers its bounding box.
     */
    mask?: Array<Locator>;

    /**
     * An acceptable ratio of pixels that are different to the total amount of pixels, between `0` and `1`. Default is
     * configurable with `TestConfig.expect`. Unset by default.
     */
    maxDiffPixelRatio?: number;

    /**
     * An acceptable amount of pixels that could be different. Default is configurable with `TestConfig.expect`. Unset by
     * default.
     */
    maxDiffPixels?: number;

    /**
     * Hides default white background and allows capturing screenshots with transparency. Not applicable to `jpeg` images.
     * Defaults to `false`.
     */
    omitBackground?: boolean;

    /**
     * When set to `"css"`, screenshot will have a single pixel per each css pixel on the page. For high-dpi devices, this will
     * keep screenshots small. Using `"device"` option will produce a single pixel per each device pixel, so screenhots of
     * high-dpi devices will be twice as large or even larger.
     *
     * Defaults to `"css"`.
     */
    scale?: "css"|"device";

    /**
     * An acceptable perceived color difference in the [YIQ color space](https://en.wikipedia.org/wiki/YIQ) between the same
     * pixel in compared images, between zero (strict) and one (lax), default is configurable with `TestConfig.expect`.
     * Defaults to `0.2`.
     */
    threshold?: number;

    /**
     * Time to retry the assertion for. Defaults to `timeout` in `TestConfig.expect`.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * This function will wait until two consecutive locator screenshots yield the same result, and then compare the last
   * screenshot with the expectation.
   *
   * ```js
   * const locator = page.locator('button');
   * await expect(locator).toHaveScreenshot();
   * ```
   *
   * @param options
   */
  toHaveScreenshot(options?: {
    /**
     * When set to `"disabled"`, stops CSS animations, CSS transitions and Web Animations. Animations get different treatment
     * depending on their duration:
     * - finite animations are fast-forwarded to completion, so they'll fire `transitionend` event.
     * - infinite animations are canceled to initial state, and then played over after the screenshot.
     *
     * Defaults to `"disabled"` that disables animations.
     */
    animations?: "disabled"|"allow";

    /**
     * When set to `"hide"`, screenshot will hide text caret. When set to `"initial"`, text caret behavior will not be changed.
     * Defaults to `"hide"`.
     */
    caret?: "hide"|"initial";

    /**
     * Specify locators that should be masked when the screenshot is taken. Masked elements will be overlaid with a pink box
     * `#FF00FF` that completely covers its bounding box.
     */
    mask?: Array<Locator>;

    /**
     * An acceptable ratio of pixels that are different to the total amount of pixels, between `0` and `1`. Default is
     * configurable with `TestConfig.expect`. Unset by default.
     */
    maxDiffPixelRatio?: number;

    /**
     * An acceptable amount of pixels that could be different. Default is configurable with `TestConfig.expect`. Unset by
     * default.
     */
    maxDiffPixels?: number;

    /**
     * Hides default white background and allows capturing screenshots with transparency. Not applicable to `jpeg` images.
     * Defaults to `false`.
     */
    omitBackground?: boolean;

    /**
     * When set to `"css"`, screenshot will have a single pixel per each css pixel on the page. For high-dpi devices, this will
     * keep screenshots small. Using `"device"` option will produce a single pixel per each device pixel, so screenhots of
     * high-dpi devices will be twice as large or even larger.
     *
     * Defaults to `"css"`.
     */
    scale?: "css"|"device";

    /**
     * An acceptable perceived color difference in the [YIQ color space](https://en.wikipedia.org/wiki/YIQ) between the same
     * pixel in compared images, between zero (strict) and one (lax), default is configurable with `TestConfig.expect`.
     * Defaults to `0.2`.
     */
    threshold?: number;

    /**
     * Time to retry the assertion for. Defaults to `timeout` in `TestConfig.expect`.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Ensures the [Locator] points to an element with the given text. You can use regular expressions for the value as well.
   *
   * ```js
   * const locator = page.locator('.title');
   * await expect(locator).toHaveText(/Welcome, Test User/);
   * await expect(locator).toHaveText(/Welcome, .*\/);
   * ```
   *
   * If you pass an array as an expected value, the expectations are:
   * 1. Locator resolves to a list of elements.
   * 1. The number of elements equals the number of expected values in the array.
   * 1. Elements from the list have text matching expected array values, one by one, in order.
   *
   * For example, consider the following list:
   *
   * ```html
   * <ul>
   *   <li>Text 1</li>
   *   <li>Text 2</li>
   *   <li>Text 3</li>
   * </ul>
   * ```
   *
   * Let's see how we can use the assertion:
   *
   * ```js
   * // ✓ Has the right items in the right order
   * await expect(page.locator('ul > li')).toHaveText(['Text 1', 'Text 2', 'Text 3']);
   *
   * // ✖ Wrong order
   * await expect(page.locator('ul > li')).toHaveText(['Text 3', 'Text 2', 'Text 1']);
   *
   * // ✖ Last item does not match
   * await expect(page.locator('ul > li')).toHaveText(['Text 1', 'Text 2', 'Text']);
   *
   * // ✖ Locator points to the outer list element, not to the list items
   * await expect(page.locator('ul')).toHaveText(['Text 1', 'Text 2', 'Text 3']);
   * ```
   *
   * @param expected Expected substring or RegExp or a list of those.
   * @param options
   */
  toHaveText(expected: string|RegExp|Array<string|RegExp>, options?: {
    /**
     * Whether to perform case-insensitive match. `ignoreCase` option takes precedence over the corresponding regular
     * expression flag if specified.
     */
    ignoreCase?: boolean;

    /**
     * Time to retry the assertion for. Defaults to `timeout` in `TestConfig.expect`.
     */
    timeout?: number;

    /**
     * Whether to use `element.innerText` instead of `element.textContent` when retrieving DOM node text.
     */
    useInnerText?: boolean;
  }): Promise<void>;

  /**
   * Ensures the [Locator] points to an element with the given input value. You can use regular expressions for the value as
   * well.
   *
   * ```js
   * const locator = page.locator('input[type=number]');
   * await expect(locator).toHaveValue(/[0-9]/);
   * ```
   *
   * @param value Expected value.
   * @param options
   */
  toHaveValue(value: string|RegExp, options?: {
    /**
     * Time to retry the assertion for. Defaults to `timeout` in `TestConfig.expect`.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Ensures the [Locator] points to multi-select/combobox (i.e. a `select` with the `multiple` attribute) and the specified
   * values are selected.
   *
   * For example, given the following element:
   *
   * ```html
   * <select id="favorite-colors" multiple>
   *   <option value="R">Red</option>
   *   <option value="G">Green</option>
   *   <option value="B">Blue</option>
   * </select>
   * ```
   *
   * ```js
   * const locator = page.locator("id=favorite-colors");
   * await locator.selectOption(["R", "G"]);
   * await expect(locator).toHaveValues([/R/, /G/]);
   * ```
   *
   * @param values Expected options currently selected.
   * @param options
   */
  toHaveValues(values: Array<string|RegExp>, options?: {
    /**
     * Time to retry the assertion for. Defaults to `timeout` in `TestConfig.expect`.
     */
    timeout?: number;
  }): Promise<void>;
}

/**
 * The [PageAssertions] class provides assertion methods that can be used to make assertions about the [Page] state in the
 * tests. A new instance of [PageAssertions] is created by calling
 * [expect(page)](https://playwright.dev/docs/api/class-playwrightassertions#playwright-assertions-expect-page):
 *
 * ```js
 * import { test, expect } from '@playwright/test';
 *
 * test('navigates to login', async ({ page }) => {
 *   // ...
 *   await page.locator('#login').click();
 *   await expect(page).toHaveURL(/.*\/login/);
 * });
 * ```
 *
 */
interface PageAssertions {
  /**
   * Makes the assertion check for the opposite condition. For example, this code tests that the page URL doesn't contain
   * `"error"`:
   *
   * ```js
   * await expect(page).not.toHaveURL('error');
   * ```
   *
   */
  not: PageAssertions;

  /**
   * This function will wait until two consecutive page screenshots yield the same result, and then compare the last
   * screenshot with the expectation.
   *
   * ```js
   * await expect(page).toHaveScreenshot('image.png');
   * ```
   *
   * @param name Snapshot name.
   * @param options
   */
  toHaveScreenshot(name: string|Array<string>, options?: {
    /**
     * When set to `"disabled"`, stops CSS animations, CSS transitions and Web Animations. Animations get different treatment
     * depending on their duration:
     * - finite animations are fast-forwarded to completion, so they'll fire `transitionend` event.
     * - infinite animations are canceled to initial state, and then played over after the screenshot.
     *
     * Defaults to `"disabled"` that disables animations.
     */
    animations?: "disabled"|"allow";

    /**
     * When set to `"hide"`, screenshot will hide text caret. When set to `"initial"`, text caret behavior will not be changed.
     * Defaults to `"hide"`.
     */
    caret?: "hide"|"initial";

    /**
     * An object which specifies clipping of the resulting image. Should have the following fields:
     */
    clip?: {
      /**
       * x-coordinate of top-left corner of clip area
       */
      x: number;

      /**
       * y-coordinate of top-left corner of clip area
       */
      y: number;

      /**
       * width of clipping area
       */
      width: number;

      /**
       * height of clipping area
       */
      height: number;
    };

    /**
     * When true, takes a screenshot of the full scrollable page, instead of the currently visible viewport. Defaults to
     * `false`.
     */
    fullPage?: boolean;

    /**
     * Specify locators that should be masked when the screenshot is taken. Masked elements will be overlaid with a pink box
     * `#FF00FF` that completely covers its bounding box.
     */
    mask?: Array<Locator>;

    /**
     * An acceptable ratio of pixels that are different to the total amount of pixels, between `0` and `1`. Default is
     * configurable with `TestConfig.expect`. Unset by default.
     */
    maxDiffPixelRatio?: number;

    /**
     * An acceptable amount of pixels that could be different. Default is configurable with `TestConfig.expect`. Unset by
     * default.
     */
    maxDiffPixels?: number;

    /**
     * Hides default white background and allows capturing screenshots with transparency. Not applicable to `jpeg` images.
     * Defaults to `false`.
     */
    omitBackground?: boolean;

    /**
     * When set to `"css"`, screenshot will have a single pixel per each css pixel on the page. For high-dpi devices, this will
     * keep screenshots small. Using `"device"` option will produce a single pixel per each device pixel, so screenhots of
     * high-dpi devices will be twice as large or even larger.
     *
     * Defaults to `"css"`.
     */
    scale?: "css"|"device";

    /**
     * An acceptable perceived color difference in the [YIQ color space](https://en.wikipedia.org/wiki/YIQ) between the same
     * pixel in compared images, between zero (strict) and one (lax), default is configurable with `TestConfig.expect`.
     * Defaults to `0.2`.
     */
    threshold?: number;

    /**
     * Time to retry the assertion for. Defaults to `timeout` in `TestConfig.expect`.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * This function will wait until two consecutive page screenshots yield the same result, and then compare the last
   * screenshot with the expectation.
   *
   * ```js
   * await expect(page).toHaveScreenshot();
   * ```
   *
   * @param options
   */
  toHaveScreenshot(options?: {
    /**
     * When set to `"disabled"`, stops CSS animations, CSS transitions and Web Animations. Animations get different treatment
     * depending on their duration:
     * - finite animations are fast-forwarded to completion, so they'll fire `transitionend` event.
     * - infinite animations are canceled to initial state, and then played over after the screenshot.
     *
     * Defaults to `"disabled"` that disables animations.
     */
    animations?: "disabled"|"allow";

    /**
     * When set to `"hide"`, screenshot will hide text caret. When set to `"initial"`, text caret behavior will not be changed.
     * Defaults to `"hide"`.
     */
    caret?: "hide"|"initial";

    /**
     * An object which specifies clipping of the resulting image. Should have the following fields:
     */
    clip?: {
      /**
       * x-coordinate of top-left corner of clip area
       */
      x: number;

      /**
       * y-coordinate of top-left corner of clip area
       */
      y: number;

      /**
       * width of clipping area
       */
      width: number;

      /**
       * height of clipping area
       */
      height: number;
    };

    /**
     * When true, takes a screenshot of the full scrollable page, instead of the currently visible viewport. Defaults to
     * `false`.
     */
    fullPage?: boolean;

    /**
     * Specify locators that should be masked when the screenshot is taken. Masked elements will be overlaid with a pink box
     * `#FF00FF` that completely covers its bounding box.
     */
    mask?: Array<Locator>;

    /**
     * An acceptable ratio of pixels that are different to the total amount of pixels, between `0` and `1`. Default is
     * configurable with `TestConfig.expect`. Unset by default.
     */
    maxDiffPixelRatio?: number;

    /**
     * An acceptable amount of pixels that could be different. Default is configurable with `TestConfig.expect`. Unset by
     * default.
     */
    maxDiffPixels?: number;

    /**
     * Hides default white background and allows capturing screenshots with transparency. Not applicable to `jpeg` images.
     * Defaults to `false`.
     */
    omitBackground?: boolean;

    /**
     * When set to `"css"`, screenshot will have a single pixel per each css pixel on the page. For high-dpi devices, this will
     * keep screenshots small. Using `"device"` option will produce a single pixel per each device pixel, so screenhots of
     * high-dpi devices will be twice as large or even larger.
     *
     * Defaults to `"css"`.
     */
    scale?: "css"|"device";

    /**
     * An acceptable perceived color difference in the [YIQ color space](https://en.wikipedia.org/wiki/YIQ) between the same
     * pixel in compared images, between zero (strict) and one (lax), default is configurable with `TestConfig.expect`.
     * Defaults to `0.2`.
     */
    threshold?: number;

    /**
     * Time to retry the assertion for. Defaults to `timeout` in `TestConfig.expect`.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Ensures the page has the given title.
   *
   * ```js
   * await expect(page).toHaveTitle(/.*checkout/);
   * ```
   *
   * @param titleOrRegExp Expected title or RegExp.
   * @param options
   */
  toHaveTitle(titleOrRegExp: string|RegExp, options?: {
    /**
     * Time to retry the assertion for. Defaults to `timeout` in `TestConfig.expect`.
     */
    timeout?: number;
  }): Promise<void>;

  /**
   * Ensures the page is navigated to the given URL.
   *
   * ```js
   * await expect(page).toHaveURL(/.*checkout/);
   * ```
   *
   * @param urlOrRegExp Expected URL string or RegExp.
   * @param options
   */
  toHaveURL(urlOrRegExp: string|RegExp, options?: {
    /**
     * Time to retry the assertion for. Defaults to `timeout` in `TestConfig.expect`.
     */
    timeout?: number;
  }): Promise<void>;
}

/**
 * Playwright provides methods for comparing page and element screenshots with expected values stored in files.
 *
 * ```js
 * expect(screenshot).toMatchSnapshot('landing-page.png');
 * ```
 *
 */
interface ScreenshotAssertions {
  /**
   * Ensures that passed value, either a [string] or a [Buffer], matches the expected snapshot stored in the test snapshots
   * directory.
   *
   * ```js
   * // Basic usage.
   * expect(await page.screenshot()).toMatchSnapshot('landing-page.png');
   *
   * // Pass options to customize the snapshot comparison and have a generated name.
   * expect(await page.screenshot()).toMatchSnapshot('landing-page.png', {
   *   maxDiffPixels: 27, // allow no more than 27 different pixels.
   * });
   *
   * // Configure image matching threshold.
   * expect(await page.screenshot()).toMatchSnapshot('landing-page.png', { threshold: 0.3 });
   *
   * // Bring some structure to your snapshot files by passing file path segments.
   * expect(await page.screenshot()).toMatchSnapshot(['landing', 'step2.png']);
   * expect(await page.screenshot()).toMatchSnapshot(['landing', 'step3.png']);
   * ```
   *
   * Learn more about [visual comparisons](https://playwright.dev/docs/api/test-snapshots).
   * @param name Snapshot name.
   * @param options
   */
  toMatchSnapshot(name: string|Array<string>, options?: {
    /**
     * An acceptable ratio of pixels that are different to the total amount of pixels, between `0` and `1`. Default is
     * configurable with `TestConfig.expect`. Unset by default.
     */
    maxDiffPixelRatio?: number;

    /**
     * An acceptable amount of pixels that could be different. Default is configurable with `TestConfig.expect`. Unset by
     * default.
     */
    maxDiffPixels?: number;

    /**
     * An acceptable perceived color difference in the [YIQ color space](https://en.wikipedia.org/wiki/YIQ) between the same
     * pixel in compared images, between zero (strict) and one (lax), default is configurable with `TestConfig.expect`.
     * Defaults to `0.2`.
     */
    threshold?: number;
  }): void;

  /**
   * Ensures that passed value, either a [string] or a [Buffer], matches the expected snapshot stored in the test snapshots
   * directory.
   *
   * ```js
   * // Basic usage and the file name is derived from the test name.
   * expect(await page.screenshot()).toMatchSnapshot();
   *
   * // Pass options to customize the snapshot comparison and have a generated name.
   * expect(await page.screenshot()).toMatchSnapshot({
   *   maxDiffPixels: 27, // allow no more than 27 different pixels.
   * });
   *
   * // Configure image matching threshold and snapshot name.
   * expect(await page.screenshot()).toMatchSnapshot({
   *   name: 'landing-page.png',
   *   threshold: 0.3,
   * });
   * ```
   *
   * Learn more about [visual comparisons](https://playwright.dev/docs/api/test-snapshots).
   * @param options
   */
  toMatchSnapshot(options?: {
    /**
     * An acceptable ratio of pixels that are different to the total amount of pixels, between `0` and `1`. Default is
     * configurable with `TestConfig.expect`. Unset by default.
     */
    maxDiffPixelRatio?: number;

    /**
     * An acceptable amount of pixels that could be different. Default is configurable with `TestConfig.expect`. Unset by
     * default.
     */
    maxDiffPixels?: number;

    /**
     * Snapshot name. If not passed, the test name and ordinals are used when called multiple times.
     */
    name?: string|Array<string>;

    /**
     * An acceptable perceived color difference in the [YIQ color space](https://en.wikipedia.org/wiki/YIQ) between the same
     * pixel in compared images, between zero (strict) and one (lax), default is configurable with `TestConfig.expect`.
     * Defaults to `0.2`.
     */
    threshold?: number;
  }): void;
}

/**
 * Information about an error thrown during test execution.
 */
export interface TestError {
  /**
   * Error message. Set when [Error] (or its subclass) has been thrown.
   */
  message?: string;

  /**
   * Error stack. Set when [Error] (or its subclass) has been thrown.
   */
  stack?: string;

  /**
   * The value that was thrown. Set when anything except the [Error] (or its subclass) has been thrown.
   */
  value?: string;
}

/**
 * Playwright Test supports running multiple test projects at the same time. This is useful for running tests in multiple
 * configurations. For example, consider running tests against multiple browsers.
 *
 * `TestProject` encapsulates configuration specific to a single project. Projects are configured in
 * [testConfig.projects](https://playwright.dev/docs/api/class-testconfig#test-config-projects) specified in the
 * [configuration file](https://playwright.dev/docs/test-configuration). Note that all properties of [TestProject] are available in the top-level
 * [TestConfig], in which case they are shared between all projects.
 *
 * Here is an example configuration that runs every test in Chromium, Firefox and WebKit, both Desktop and Mobile versions.
 *
 * ```js
 * // playwright.config.ts
 * import { type PlaywrightTestConfig, devices } from '@playwright/test';
 *
 * const config: PlaywrightTestConfig = {
 *   // Options shared for all projects.
 *   timeout: 30000,
 *   use: {
 *     ignoreHTTPSErrors: true,
 *   },
 *
 *   // Options specific to each project.
 *   projects: [
 *     {
 *       name: 'Desktop Chromium',
 *       use: {
 *         browserName: 'chromium',
 *         viewport: { width: 1280, height: 720 },
 *       },
 *     },
 *     {
 *       name: 'Desktop Safari',
 *       use: {
 *         browserName: 'webkit',
 *         viewport: { width: 1280, height: 720 },
 *       }
 *     },
 *     {
 *       name: 'Desktop Firefox',
 *       use: {
 *         browserName: 'firefox',
 *         viewport: { width: 1280, height: 720 },
 *       }
 *     },
 *     {
 *       name: 'Mobile Chrome',
 *       use: devices['Pixel 5'],
 *     },
 *     {
 *       name: 'Mobile Safari',
 *       use: devices['iPhone 12'],
 *     },
 *   ],
 * };
 * export default config;
 * ```
 *
 */
interface TestProject {
  /**
   * Configuration for the `expect` assertion library.
   *
   * Use [testConfig.expect](https://playwright.dev/docs/api/class-testconfig#test-config-expect) to change this option for
   * all projects.
   */
  expect?: {
    /**
     * Default timeout for async expect matchers in milliseconds, defaults to 5000ms.
     */
    timeout?: number;

    /**
     * Configuration for the
     * [pageAssertions.toHaveScreenshot(name[, options])](https://playwright.dev/docs/api/class-pageassertions#page-assertions-to-have-screenshot-1)
     * method.
     */
    toHaveScreenshot?: {
      /**
       * an acceptable perceived color difference in the [YIQ color space](https://en.wikipedia.org/wiki/YIQ) between the same
       * pixel in compared images, between zero (strict) and one (lax). Defaults to `0.2`.
       */
      threshold?: number;

      /**
       * an acceptable amount of pixels that could be different, unset by default.
       */
      maxDiffPixels?: number;

      /**
       * an acceptable ratio of pixels that are different to the total amount of pixels, between `0` and `1` , unset by default.
       */
      maxDiffPixelRatio?: number;

      /**
       * See `animations` in [page.screenshot([options])](https://playwright.dev/docs/api/class-page#page-screenshot). Defaults
       * to `"disabled"`.
       */
      animations?: "allow"|"disabled";

      /**
       * See `caret` in [page.screenshot([options])](https://playwright.dev/docs/api/class-page#page-screenshot). Defaults to
       * `"hide"`.
       */
      caret?: "hide"|"initial";

      /**
       * See `scale` in [page.screenshot([options])](https://playwright.dev/docs/api/class-page#page-screenshot). Defaults to
       * `"css"`.
       */
      scale?: "css"|"device";
    };

    /**
     * Configuration for the
     * [screenshotAssertions.toMatchSnapshot(name[, options])](https://playwright.dev/docs/api/class-screenshotassertions#screenshot-assertions-to-match-snapshot-1)
     * method.
     */
    toMatchSnapshot?: {
      /**
       * an acceptable perceived color difference in the [YIQ color space](https://en.wikipedia.org/wiki/YIQ) between the same
       * pixel in compared images, between zero (strict) and one (lax). Defaults to `0.2`.
       */
      threshold?: number;

      /**
       * an acceptable amount of pixels that could be different, unset by default.
       */
      maxDiffPixels?: number;

      /**
       * an acceptable ratio of pixels that are different to the total amount of pixels, between `0` and `1` , unset by default.
       */
      maxDiffPixelRatio?: number;
    };
  };

  /**
   * Playwright Test runs tests in parallel. In order to achieve that, it runs several worker processes that run at the same
   * time. By default, **test files** are run in parallel. Tests in a single file are run in order, in the same worker
   * process.
   *
   * You can configure entire test project to concurrently run all tests in all files using this option.
   */
  fullyParallel?: boolean;

  /**
   * Filter to only run tests with a title matching one of the patterns. For example, passing `grep: /cart/` should only run
   * tests with "cart" in the title. Also available globally and in the [command line](https://playwright.dev/docs/test-cli) with the `-g` option.
   *
   * `grep` option is also useful for [tagging tests](https://playwright.dev/docs/test-annotations#tag-tests).
   */
  grep?: RegExp|Array<RegExp>;

  /**
   * Filter to only run tests with a title **not** matching one of the patterns. This is the opposite of
   * [testProject.grep](https://playwright.dev/docs/api/class-testproject#test-project-grep). Also available globally and in
   * the [command line](https://playwright.dev/docs/test-cli) with the `--grep-invert` option.
   *
   * `grepInvert` option is also useful for [tagging tests](https://playwright.dev/docs/test-annotations#tag-tests).
   */
  grepInvert?: RegExp|Array<RegExp>;

  /**
   * Metadata that will be put directly to the test report serialized as JSON.
   */
  metadata?: Metadata;

  /**
   * Project name is visible in the report and during test execution.
   */
  name?: string;

  /**
   * The base directory, relative to the config file, for snapshot files created with `toMatchSnapshot`. Defaults to
   * [testProject.testDir](https://playwright.dev/docs/api/class-testproject#test-project-test-dir).
   *
   * The directory for each test can be accessed by
   * [testInfo.snapshotDir](https://playwright.dev/docs/api/class-testinfo#test-info-snapshot-dir) and
   * [testInfo.snapshotPath(...pathSegments)](https://playwright.dev/docs/api/class-testinfo#test-info-snapshot-path).
   *
   * This path will serve as the base directory for each test file snapshot directory. Setting `snapshotDir` to
   * `'snapshots'`, the [testInfo.snapshotDir](https://playwright.dev/docs/api/class-testinfo#test-info-snapshot-dir) would
   * resolve to `snapshots/a.spec.js-snapshots`.
   */
  snapshotDir?: string;

  /**
   * The output directory for files created during test execution. Defaults to `<package.json-directory>/test-results`.
   *
   * This directory is cleaned at the start. When running a test, a unique subdirectory inside the
   * [testProject.outputDir](https://playwright.dev/docs/api/class-testproject#test-project-output-dir) is created,
   * guaranteeing that test running in parallel do not conflict. This directory can be accessed by
   * [testInfo.outputDir](https://playwright.dev/docs/api/class-testinfo#test-info-output-dir) and
   * [testInfo.outputPath(...pathSegments)](https://playwright.dev/docs/api/class-testinfo#test-info-output-path).
   *
   * Here is an example that uses
   * [testInfo.outputPath(...pathSegments)](https://playwright.dev/docs/api/class-testinfo#test-info-output-path) to create a
   * temporary file.
   *
   * ```js
   * import { test, expect } from '@playwright/test';
   * import fs from 'fs';
   *
   * test('example test', async ({}, testInfo) => {
   *   const file = testInfo.outputPath('temporary-file.txt');
   *   await fs.promises.writeFile(file, 'Put some data to the file', 'utf8');
   * });
   * ```
   *
   * Use [testConfig.outputDir](https://playwright.dev/docs/api/class-testconfig#test-config-output-dir) to change this
   * option for all projects.
   */
  outputDir?: string;

  /**
   * The number of times to repeat each test, useful for debugging flaky tests.
   *
   * Use [testConfig.repeatEach](https://playwright.dev/docs/api/class-testconfig#test-config-repeat-each) to change this
   * option for all projects.
   */
  repeatEach?: number;

  /**
   * The maximum number of retry attempts given to failed tests. Learn more about [test retries](https://playwright.dev/docs/test-retries#retries).
   *
   * Use [testConfig.retries](https://playwright.dev/docs/api/class-testconfig#test-config-retries) to change this option for
   * all projects.
   */
  retries?: number;

  /**
   * Directory that will be recursively scanned for test files. Defaults to the directory of the configuration file.
   *
   * Each project can use a different directory. Here is an example that runs smoke tests in three browsers and all other
   * tests in stable Chrome browser.
   *
   * ```js
   * // playwright.config.ts
   * import type { PlaywrightTestConfig } from '@playwright/test';
   *
   * const config: PlaywrightTestConfig = {
   *   projects: [
   *     {
   *       name: 'Smoke Chromium',
   *       testDir: './smoke-tests',
   *       use: {
   *         browserName: 'chromium',
   *       }
   *     },
   *     {
   *       name: 'Smoke WebKit',
   *       testDir: './smoke-tests',
   *       use: {
   *         browserName: 'webkit',
   *       }
   *     },
   *     {
   *       name: 'Smoke Firefox',
   *       testDir: './smoke-tests',
   *       use: {
   *         browserName: 'firefox',
   *       }
   *     },
   *     {
   *       name: 'Chrome Stable',
   *       testDir: './',
   *       use: {
   *         browserName: 'chromium',
   *         channel: 'chrome',
   *       }
   *     },
   *   ],
   * };
   * export default config;
   * ```
   *
   * Use [testConfig.testDir](https://playwright.dev/docs/api/class-testconfig#test-config-test-dir) to change this option
   * for all projects.
   */
  testDir?: string;

  /**
   * Files matching one of these patterns are not executed as test files. Matching is performed against the absolute file
   * path. Strings are treated as glob patterns.
   *
   * For example, `'**\/test-assets/**'` will ignore any files in the `test-assets` directory.
   *
   * Use [testConfig.testIgnore](https://playwright.dev/docs/api/class-testconfig#test-config-test-ignore) to change this
   * option for all projects.
   */
  testIgnore?: string|RegExp|Array<string|RegExp>;

  /**
   * Only the files matching one of these patterns are executed as test files. Matching is performed against the absolute
   * file path. Strings are treated as glob patterns.
   *
   * By default, Playwright Test looks for files matching `.*(test|spec)\.(js|ts|mjs)`.
   *
   * Use [testConfig.testMatch](https://playwright.dev/docs/api/class-testconfig#test-config-test-match) to change this
   * option for all projects.
   */
  testMatch?: string|RegExp|Array<string|RegExp>;

  /**
   * Timeout for each test in milliseconds. Defaults to 30 seconds.
   *
   * This is a base timeout for all tests. In addition, each test can configure its own timeout with
   * [test.setTimeout(timeout)](https://playwright.dev/docs/api/class-test#test-set-timeout).
   *
   * Use [testConfig.timeout](https://playwright.dev/docs/api/class-testconfig#test-config-timeout) to change this option for
   * all projects.
   */
  timeout?: number;
}

interface TestConfigWebServer {
  /**
   * Shell command to start. For example `npm run start`..
   */
  command: string;

  /**
   * The port that your http server is expected to appear on. It does wait until it accepts connections. Exactly one of
   * `port` or `url` is required.
   */
  port?: number;

  /**
   * The url on your http server that is expected to return a 2xx, 3xx, 400, 401, 402, or 403 status code when the server is
   * ready to accept connections. Exactly one of `port` or `url` is required.
   */
  url?: string;

  /**
   * Whether to ignore HTTPS errors when fetching the `url`. Defaults to `false`.
   */
  ignoreHTTPSErrors?: boolean;

  /**
   * How long to wait for the process to start up and be available in milliseconds. Defaults to 60000.
   */
  timeout?: number;

  /**
   * If true, it will re-use an existing server on the `port` or `url` when available. If no server is running on that `port`
   * or `url`, it will run the command to start a new server. If `false`, it will throw if an existing process is listening
   * on the `port` or `url`. This should be commonly set to `!process.env.CI` to allow the local dev server when running
   * tests locally.
   */
  reuseExistingServer?: boolean;

  /**
   * Current working directory of the spawned process, defaults to the directory of the configuration file.
   */
  cwd?: string;

  /**
   * Environment variables to set for the command, `process.env` by default.
   */
  env?: { [key: string]: string; };
}

