const body = document.body;
const toggleButton = document.querySelector('[data-ui-toggle="sidebar"]');
const dismissButton = document.querySelector('[data-ui-dismiss="sidebar"]');
const actionModal = document.querySelector('[data-ui="action-modal"]');
const actionTitle = actionModal?.querySelector('[data-action-title]');
const actionSub = actionModal?.querySelector('[data-action-sub]');
const actionForm = actionModal?.querySelector('[data-action-form]');
const actionFields = actionModal?.querySelector('[data-action-fields]');
const actionCloseButtons = actionModal?.querySelectorAll('[data-action-close]') ?? [];
const toast = document.querySelector('[data-ui="toast"]');
const chatbot = document.querySelector('[data-chatbot]');
const chatbotOpen = document.querySelector('[data-chatbot-open]');
const chatbotClose = document.querySelector('[data-chatbot-close]');
const chatbotBody = document.querySelector('[data-chatbot-body]');
const chatbotForm = document.querySelector('[data-chatbot-form]');
const chatbotText = document.querySelector('[data-chatbot-text]');
const chatbotFile = document.querySelector('[data-chatbot-file]');
const chatHistory = [];
let conversationId = localStorage.getItem('adm_chat_conversation') || '';
let pendingUploads = [];
const chatRoutes = [
    { label: 'Dashboard', path: '/ui', keywords: ['dashboard', 'command center', 'home'] },
    { label: 'Assets', path: '/ui/assets', keywords: ['assets', 'asset register', 'asset portfolio'] },
    { label: 'Fleet', path: '/ui/vehicles', keywords: ['fleet', 'vehicles', 'vehicle operations'] },
    { label: 'Trips', path: '/ui/trips', keywords: ['trips', 'trip intelligence', 'trip'] },
    { label: 'Maintenance', path: '/ui/maintenance', keywords: ['maintenance overview', 'maintenance'] },
    { label: 'Work Orders', path: '/ui/work-orders', keywords: ['work orders', 'work order', 'job card'] },
    { label: 'PM Schedules', path: '/ui/pm-schedules', keywords: ['pm schedule', 'preventive maintenance', 'pm'] },
    { label: 'Checklists', path: '/ui/checklists', keywords: ['checklists', 'checklist'] },
    { label: 'Downtime', path: '/ui/downtime', keywords: ['downtime'] },
    { label: 'Fuel', path: '/ui/fuel', keywords: ['fuel', 'fuel logs'] },
    { label: 'Tyres', path: '/ui/tyres', keywords: ['tyres', 'tyre', 'tire'] },
    { label: 'Telemetry', path: '/ui/telemetry', keywords: ['telemetry', 'sensor pulse', 'sensors'] },
    { label: 'Alerts', path: '/ui/alerts', keywords: ['alerts', 'alert', 'alarms'] },
    { label: 'Alarm Rules', path: '/ui/alarm-rules', keywords: ['alarm rules', 'alert rules', 'alarm rule'] },
    { label: 'KPI Studio', path: '/ui/kpis', keywords: ['kpi studio', 'kpis', 'performance metrics', 'kpi'] },
    { label: 'AI Studio', path: '/ui/ai', keywords: ['ai', 'automation studio', 'automation'] },
    { label: 'Reports', path: '/ui/reports', keywords: ['reports', 'reporting'] },
    { label: 'Tenants', path: '/ui/tenants', keywords: ['tenants', 'tenant directory', 'tenant'] },
    { label: 'Users', path: '/ui/users', keywords: ['users', 'team access', 'team', 'user'] },
    { label: 'Roles', path: '/ui/roles', keywords: ['roles', 'rbac'] },
    { label: 'Notifications', path: '/ui/notifications', keywords: ['notifications', 'channels'] },
    { label: 'Audit Log', path: '/ui/audit', keywords: ['audit', 'audit log'] },
    { label: 'Reconciliation', path: '/ui/reconciliation', keywords: ['reconciliation', 'linking', 'matching', 'link suggestions'] },
    { label: 'Settings', path: '/ui/settings', keywords: ['settings', 'system configuration'] },
    { label: 'Inventory', path: '/ui/inventory', keywords: ['inventory', 'parts', 'stores'] },
    { label: 'Vendors', path: '/ui/vendors', keywords: ['vendors', 'suppliers'] },
    { label: 'Purchase Requests', path: '/ui/purchase-requests', keywords: ['purchase requests', 'purchase request', 'pr'] },
    { label: 'Purchase Orders', path: '/ui/purchase-orders', keywords: ['purchase orders', 'purchase order', 'po'] },
    { label: 'Payments', path: '/ui/payments', keywords: ['payments', 'payment', 'invoices'] },
    { label: 'Imports', path: '/ui/imports', keywords: ['imports', 'import', 'uploads', 'upload'] },
    { label: 'Devices', path: '/ui/devices', keywords: ['devices', 'device', 'sensors'] },
    { label: 'Sites', path: '/ui/sites', keywords: ['sites', 'locations'] },
];
const navToggles = document.querySelectorAll('[data-nav-toggle]');
const topSearchInput = document.querySelector('.search input');

if (toggleButton) {
    toggleButton.addEventListener('click', () => {
        body.classList.toggle('sidebar-open');
    });
}

if (dismissButton) {
    dismissButton.addEventListener('click', () => {
        body.classList.remove('sidebar-open');
    });
}

navToggles.forEach((toggle) => {
    toggle.addEventListener('click', () => {
        const group = toggle.closest('.nav-group');
        if (!group) {
            return;
        }

        group.classList.toggle('is-collapsed');
    });
});

function focusTopSearch() {
    if (!topSearchInput) {
        return;
    }
    topSearchInput.focus();
    topSearchInput.select();
}

if (topSearchInput) {
    topSearchInput.addEventListener('keydown', (event) => {
        if (event.key !== 'Enter') {
            return;
        }

        event.preventDefault();
        const query = topSearchInput.value.trim().toLowerCase();
        if (!query) {
            return;
        }

        const route = resolveChatRoute(query);
        if (!route) {
            showToast('No matching page found.');
            return;
        }

        window.location.href = route.path;
    });
}

document.addEventListener('keydown', (event) => {
    if (event.key !== '/') {
        return;
    }

    const target = event.target;
    if (target && (target.tagName === 'INPUT' || target.tagName === 'TEXTAREA' || target.isContentEditable)) {
        return;
    }

    event.preventDefault();
    focusTopSearch();
});

function buildFields(fields) {
    if (!actionFields) {
        return;
    }

    actionFields.innerHTML = '';
    fields.forEach((field) => {
        const label = document.createElement('label');
        label.textContent = field.label;
        actionFields.appendChild(label);

        if (field.type === 'select') {
            const select = document.createElement('select');
            select.className = 'select';
            if (field.key) {
                select.dataset.fieldKey = field.key;
            }
            (field.options || []).forEach((optionValue) => {
                const option = document.createElement('option');
                option.value = optionValue;
                option.textContent = optionValue;
                select.appendChild(option);
            });
            if (field.value) {
                select.value = field.value;
            }
            actionFields.appendChild(select);
            return;
        }

        if (field.type === 'textarea') {
            const textarea = document.createElement('textarea');
            textarea.className = 'input';
            textarea.rows = field.rows || 3;
            textarea.placeholder = field.placeholder || '';
            if (field.key) {
                textarea.dataset.fieldKey = field.key;
            }
            if (field.value) {
                textarea.value = field.value;
            }
            actionFields.appendChild(textarea);
            return;
        }

        const input = document.createElement('input');
        input.className = 'input';
        input.type = field.type || 'text';
        input.placeholder = field.placeholder || '';
        if (field.key) {
            input.dataset.fieldKey = field.key;
        }
        if (field.value) {
            input.value = field.value;
        }
        actionFields.appendChild(input);
    });
}

function normalizeLabel(label) {
    return label.toLowerCase().trim();
}

function randomCode(prefix) {
    const digits = Math.floor(100 + Math.random() * 900);
    return `${prefix}-${digits}`;
}

function resolveAction(label) {
    const normalized = normalizeLabel(label);
    const config = {
        title: label || 'Action',
        description: 'Complete the form to proceed.',
        fields: [
            { label: 'Notes', type: 'textarea', placeholder: 'Add a short note...' },
        ],
        successMessage: `${label} submitted.`,
        redirect: null,
    };

    if (normalized.includes('quick add')) {
        config.title = 'Quick Add';
        config.fields = [
            { label: 'Create', type: 'select', options: ['Asset', 'Work Order', 'Trip', 'Fuel Log', 'Device'] },
            { label: 'Notes', type: 'textarea', placeholder: 'Optional notes' },
        ];
        config.redirect = '';
        config.successMessage = 'Quick Add queued.';
        return config;
    }

    if (normalized.includes('work order')) {
        config.title = 'Create Work Order';
        config.fields = [
            { label: 'Asset', type: 'text', placeholder: 'Asset ID or name' },
            { label: 'Type', type: 'select', options: ['Corrective', 'Preventive', 'Inspection'] },
            { label: 'Priority', type: 'select', options: ['Low', 'Medium', 'High'] },
            { label: 'Description', type: 'textarea', placeholder: 'Describe the issue...' },
        ];
        config.redirect = `/ui/work-orders/${randomCode('WO')}`;
        return config;
    }

    if (normalized.includes('asset')) {
        config.title = 'Create Asset';
        config.fields = [
            { label: 'Asset Tag', type: 'text', placeholder: 'A-000' },
            { label: 'Name', type: 'text', placeholder: 'Asset name' },
            { label: 'Category', type: 'select', options: ['Mobile Plant', 'Vehicle', 'Fixed Plant', 'Tool'] },
            { label: 'Site', type: 'text', placeholder: 'Site name' },
        ];
        config.redirect = `/ui/assets/${randomCode('A')}`;
        return config;
    }

    if (normalized.includes('vehicle') || normalized.includes('fleet')) {
        config.title = 'Add Vehicle';
        config.fields = [
            { label: 'Vehicle Name', type: 'text', placeholder: 'Hauler 10' },
            { label: 'Plate', type: 'text', placeholder: 'KX-220-ZN' },
            { label: 'Type', type: 'select', options: ['Hauler', 'Loader', 'Excavator'] },
            { label: 'Site', type: 'text', placeholder: 'Site name' },
        ];
        config.redirect = `/ui/vehicles/${randomCode('V')}`;
        return config;
    }

    if (normalized.includes('trip')) {
        config.title = 'Create Trip';
        config.fields = [
            { label: 'Start Location', type: 'text', placeholder: 'Origin' },
            { label: 'End Location', type: 'text', placeholder: 'Destination' },
            { label: 'Vehicle', type: 'text', placeholder: 'Vehicle ID' },
        ];
        config.redirect = `/ui/trips/${randomCode('T')}`;
        return config;
    }

    if (normalized.includes('pm') || normalized.includes('schedule')) {
        config.title = 'Create PM Schedule';
        config.fields = [
            { label: 'Asset', type: 'text', placeholder: 'Asset ID' },
            { label: 'Interval', type: 'text', placeholder: '250' },
            { label: 'Unit', type: 'select', options: ['Hours', 'KM', 'Days'] },
        ];
        config.redirect = `/ui/pm-schedules/${randomCode('PM')}`;
        return config;
    }

    if (normalized.includes('checklist')) {
        config.title = 'Create Checklist';
        config.fields = [
            { label: 'Checklist Name', type: 'text', placeholder: 'Safety Inspection' },
            { label: 'Type', type: 'select', options: ['Pre-start', 'Safety', 'Inspection'] },
        ];
        config.redirect = `/ui/checklists/${randomCode('CL')}`;
        return config;
    }

    if (normalized.includes('downtime')) {
        config.title = 'Log Downtime';
        config.fields = [
            { label: 'Asset', type: 'text', placeholder: 'Asset ID' },
            { label: 'Reason', type: 'select', options: ['Breakdown', 'Waiting Parts', 'No Operator'] },
            { label: 'Start Time', type: 'text', placeholder: '08:30' },
        ];
        config.redirect = `/ui/downtime/${randomCode('DT')}`;
        return config;
    }

    if (normalized.includes('fuel')) {
        config.title = 'Log Fuel';
        config.fields = [
            { label: 'Asset', type: 'text', placeholder: 'Asset ID' },
            { label: 'Quantity (L)', type: 'text', placeholder: '120' },
            { label: 'Source', type: 'select', options: ['Bowser', 'Station'] },
        ];
        config.redirect = `/ui/fuel/${randomCode('FL')}`;
        return config;
    }

    if (normalized.includes('tyre') || normalized.includes('tire')) {
        config.title = 'Add Tyre';
        config.fields = [
            { label: 'Asset', type: 'text', placeholder: 'Asset ID' },
            { label: 'Position', type: 'text', placeholder: 'Front Left' },
            { label: 'Tread Depth', type: 'text', placeholder: '22 mm' },
        ];
        config.redirect = `/ui/tyres/${randomCode('TY')}`;
        return config;
    }

    if (normalized.includes('inventory') || normalized.includes('part')) {
        config.title = 'Add Inventory Item';
        config.fields = [
            { label: 'SKU', type: 'text', placeholder: 'FLT-100' },
            { label: 'Quantity', type: 'text', placeholder: '10' },
            { label: 'Location', type: 'text', placeholder: 'Main Store' },
        ];
        config.redirect = `/ui/inventory/${randomCode('PRT')}`;
        return config;
    }

    if (normalized.includes('vendor') || normalized.includes('quote')) {
        config.title = 'Add Vendor';
        config.fields = [
            { label: 'Vendor Name', type: 'text', placeholder: 'Summit Industrial' },
            { label: 'Category', type: 'text', placeholder: 'Filters' },
            { label: 'Lead Time', type: 'text', placeholder: '5 days' },
        ];
        config.redirect = `/ui/vendors/${randomCode('VND')}`;
        return config;
    }

    if (normalized.includes('service sheet')) {
        config.title = 'Upload Service Sheets';
        config.fields = [
            { label: 'File Name', type: 'text', placeholder: 'service_sheets.csv' },
            { label: 'Asset Identifier', type: 'select', options: ['Asset Tag', 'Asset Name'] },
            { label: 'Notes', type: 'textarea', placeholder: 'Optional notes' },
        ];
        config.redirect = `/ui/imports/${randomCode('IMP')}`;
        config.successMessage = 'Service sheets queued for processing.';
        return config;
    }

    if (normalized.includes('invoice')) {
        config.title = 'Upload Invoices';
        config.fields = [
            { label: 'File Name', type: 'text', placeholder: 'invoices.csv' },
            { label: 'Match By', type: 'select', options: ['Work Order Ref', 'Asset Tag'] },
            { label: 'Notes', type: 'textarea', placeholder: 'Optional notes' },
        ];
        config.redirect = `/ui/imports/${randomCode('IMP')}`;
        config.successMessage = 'Invoices queued for posting.';
        return config;
    }

    if (normalized.includes('import') || normalized.includes('upload')) {
        config.title = 'Upload Import';
        config.fields = [
            { label: 'File Name', type: 'text', placeholder: 'fleet_register.csv' },
            { label: 'Type', type: 'select', options: ['CSV', 'Excel'] },
            { label: 'Notes', type: 'textarea', placeholder: 'Optional notes' },
        ];
        config.redirect = `/ui/imports/${randomCode('IMP')}`;
        return config;
    }

    if (normalized.includes('device') || normalized.includes('sensor')) {
        config.title = 'Add Device';
        config.fields = [
            { label: 'Device ID', type: 'text', placeholder: 'DEV-102Q' },
            { label: 'Type', type: 'select', options: ['CAN', 'GPS', 'Vibration'] },
            { label: 'Asset', type: 'text', placeholder: 'Asset ID' },
        ];
        config.redirect = `/ui/devices/${randomCode('DEV')}`;
        return config;
    }

    if (normalized.includes('alert rule') || normalized.includes('alarm rule')) {
        config.title = 'Create Alarm Rule';
        config.fields = [
            { label: 'Metric', type: 'text', placeholder: 'coolant_temp' },
            { label: 'Threshold', type: 'text', placeholder: '> 95' },
            { label: 'Severity', type: 'select', options: ['Info', 'Warning', 'Critical'] },
        ];
        config.redirect = `/ui/alarm-rules/${randomCode('AR')}`;
        return config;
    }

    if (normalized.includes('alert')) {
        config.title = 'Acknowledge Alert';
        config.fields = [
            { label: 'Owner', type: 'text', placeholder: 'Maintenance Manager' },
            { label: 'Notes', type: 'textarea', placeholder: 'Add a response note...' },
        ];
        config.redirect = `/ui/alerts/${randomCode('AL')}`;
        return config;
    }

    if (normalized.includes('notification') || normalized.includes('channel')) {
        config.title = 'Configure Notifications';
        config.fields = [
            { label: 'Channel', type: 'select', options: ['WhatsApp', 'SMS', 'Telegram', 'Email'] },
            { label: 'Recipient', type: 'text', placeholder: '+1 555 010' },
        ];
        config.redirect = `/ui/notifications/${randomCode('NT')}`;
        return config;
    }

    if (normalized.includes('role')) {
        config.title = 'Create Role';
        config.fields = [
            { label: 'Role Name', type: 'text', placeholder: 'Maintenance' },
            { label: 'Description', type: 'textarea', placeholder: 'Role summary' },
        ];
        config.redirect = `/ui/roles/${encodeURIComponent('New Role')}`;
        return config;
    }

    if (normalized.includes('tenant')) {
        config.title = 'Create Tenant';
        config.fields = [
            { label: 'Tenant Name', type: 'text', placeholder: 'Iron Ridge Mining' },
            { label: 'Plan', type: 'select', options: ['Basic', 'Pro', 'Enterprise'] },
        ];
        config.redirect = `/ui/tenants/${encodeURIComponent('New Tenant')}`;
        return config;
    }

    if (normalized.includes('user') || normalized.includes('invite')) {
        config.title = 'Invite User';
        config.fields = [
            { label: 'Name', type: 'text', placeholder: 'Jane Doe' },
            { label: 'Role', type: 'select', options: ['Admin', 'Maintenance', 'Operator', 'Viewer'] },
            { label: 'Email', type: 'text', placeholder: 'user@example.com' },
        ];
        config.redirect = `/ui/users/${encodeURIComponent('New User')}`;
        return config;
    }

    if (normalized.includes('report')) {
        config.title = 'Generate Report';
        config.fields = [
            { label: 'Report Type', type: 'select', options: ['Weekly Ops', 'Compliance', 'Costing'] },
            { label: 'Period', type: 'text', placeholder: 'Last 7 days' },
        ];
        config.redirect = `/ui/reports/${encodeURIComponent('New Report')}`;
        return config;
    }

    if (normalized.includes('kpi')) {
        config.title = 'Create KPI';
        config.fields = [
            { label: 'KPI Name', type: 'text', placeholder: 'Utilization' },
            { label: 'Target', type: 'text', placeholder: '85%' },
        ];
        config.redirect = `/ui/kpis/${encodeURIComponent('New KPI')}`;
        return config;
    }

    if (normalized.includes('site')) {
        config.title = 'Add Site';
        config.fields = [
            { label: 'Site Name', type: 'text', placeholder: 'Plant A' },
            { label: 'Type', type: 'select', options: ['Quarry', 'Plant', 'Workshop', 'Yard'] },
        ];
        config.redirect = `/ui/sites/${randomCode('S')}`;
        return config;
    }

    if (normalized.includes('support')) {
        config.title = 'Open Support';
        config.fields = [
            { label: 'Topic', type: 'text', placeholder: 'Describe your issue' },
            { label: 'Priority', type: 'select', options: ['Low', 'Medium', 'High'] },
        ];
        config.redirect = '';
        config.successMessage = 'Support ticket opened.';
        return config;
    }

    if (normalized.includes('export') || normalized.includes('download') || normalized.includes('share')) {
        config.title = label;
        config.fields = [
            { label: 'Format', type: 'select', options: ['PDF', 'CSV'] },
        ];
        config.redirect = null;
        config.successMessage = `${label} queued.`;
        return config;
    }

    return config;
}

function openActionModal(label) {
    if (!actionModal) {
        return;
    }

    const definition = resolveAction(label);
    actionModal.dataset.actionLabel = definition.title || label || 'Action';
    actionModal.dataset.actionRedirect = definition.redirect || '';
    actionModal.dataset.actionMessage = definition.successMessage || `${label} submitted.`;

    if (actionTitle) {
        actionTitle.textContent = actionModal.dataset.actionLabel;
    }
    if (actionSub) {
        actionSub.textContent = definition.description || 'Preview flow with sample fields.';
    }

    buildFields(definition.fields || []);

    body.classList.add('action-open');
    actionModal.setAttribute('aria-hidden', 'false');
}

function closeActionModal() {
    if (!actionModal) {
        return;
    }

    body.classList.remove('action-open');
    actionModal.setAttribute('aria-hidden', 'true');
    actionModal.dataset.actionType = '';
    actionModal.dataset.actionId = '';
}

function showToast(message) {
    if (!toast) {
        return;
    }

    toast.textContent = message;
    toast.classList.add('is-visible');
    window.setTimeout(() => {
        toast.classList.remove('is-visible');
    }, 2200);
}

function openChatbot() {
    if (!chatbot) {
        return;
    }

    body.classList.add('chatbot-open');
    chatbot.setAttribute('aria-hidden', 'false');
    if (chatbotText) {
        chatbotText.focus();
    }
}

function closeChatbot() {
    if (!chatbot) {
        return;
    }

    body.classList.remove('chatbot-open');
    chatbot.setAttribute('aria-hidden', 'true');
}

function appendChatBubble(text, role, options = {}) {
    if (!chatbotBody) {
        return null;
    }

    const bubble = document.createElement('div');
    bubble.className = `chatbot-bubble ${role}`;
    if (options.isTyping) {
        bubble.classList.add('is-typing');
    }
    if (options.html) {
        bubble.innerHTML = text;
    } else {
        bubble.textContent = text;
    }
    chatbotBody.appendChild(bubble);
    chatbotBody.scrollTop = chatbotBody.scrollHeight;
    return bubble;
}

function resolveChatRoute(input) {
    return chatRoutes.find((route) => route.keywords.some((keyword) => input.includes(keyword))) || null;
}

function resolveChatRouteByPath(path) {
    if (!path) {
        return null;
    }
    return chatRoutes.find((route) => route.path === path) || null;
}

function resolveChatRouteByLabel(label) {
    if (!label) {
        return null;
    }
    const normalized = label.toLowerCase().trim();
    return chatRoutes.find((route) => route.label.toLowerCase() === normalized) || null;
}

function pushChatHistory(role, content) {
    if (!content) {
        return;
    }
    chatHistory.push({ role, content });
    if (chatHistory.length > 12) {
        chatHistory.shift();
    }
}

async function requestAiResponse(message, files = []) {
    const context = {
        current_path: window.location.pathname,
        routes: chatRoutes.map((route) => ({ label: route.label, path: route.path })),
    };

    let response;
    if (files.length) {
        const formData = new FormData();
        formData.append('message', message || '');
        if (conversationId) {
            formData.append('conversation_id', conversationId);
        }
        formData.append('history', JSON.stringify(chatHistory));
        formData.append('context', JSON.stringify(context));
        files.forEach((file) => {
            formData.append('files[]', file);
        });

        response = await fetch('/api/ai/chat', {
            method: 'POST',
            headers: {
                'Accept': 'application/json',
            },
            body: formData,
        });
    } else {
        response = await fetch('/api/ai/chat', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'Accept': 'application/json',
            },
            body: JSON.stringify({
                message,
                conversation_id: conversationId || null,
                history: chatHistory,
                context,
            }),
        });
    }

    if (!response.ok) {
        throw new Error('AI request failed');
    }

    return response.json();
}

function resolveChatActionLabel(input) {
    if (input.includes('quick add')) {
        return 'Quick Add';
    }
    if (input.includes('work order') || input.includes('job card')) {
        return 'Create Work Order';
    }
    if (input.includes('asset register') || input.includes('upload asset')) {
        return 'Upload Asset Register';
    }
    if (input.includes('asset')) {
        return 'Create Asset';
    }
    if (input.includes('vehicle') || input.includes('fleet')) {
        return 'Add Vehicle';
    }
    if (input.includes('trip')) {
        return 'Create Trip';
    }
    if (input.includes('pm') || input.includes('schedule')) {
        return 'Create PM Schedule';
    }
    if (input.includes('checklist')) {
        return 'Create Checklist';
    }
    if (input.includes('downtime')) {
        return 'Log Downtime';
    }
    if (input.includes('fuel')) {
        return 'Log Fuel';
    }
    if (input.includes('tyre') || input.includes('tire')) {
        return 'Add Tyre';
    }
    if (input.includes('inventory') || input.includes('part')) {
        return 'Add Inventory Item';
    }
    if (input.includes('vendor') || input.includes('supplier')) {
        return 'Add Vendor';
    }
    if (input.includes('service sheet')) {
        return 'Upload Service Sheets';
    }
    if (input.includes('invoice')) {
        return 'Upload Invoices';
    }
    if (input.includes('import') || input.includes('upload')) {
        return 'Upload Import';
    }
    if (input.includes('device') || input.includes('sensor')) {
        return 'Add Device';
    }
    if (input.includes('alarm rule') || input.includes('alert rule')) {
        return 'Create Alarm Rule';
    }
    if (input.includes('alert')) {
        return 'Acknowledge Alert';
    }
    if (input.includes('notification') || input.includes('channel')) {
        return 'Configure Notifications';
    }
    if (input.includes('role')) {
        return 'Create Role';
    }
    if (input.includes('tenant')) {
        return 'Create Tenant';
    }
    if (input.includes('user') || input.includes('invite')) {
        return 'Invite User';
    }
    if (input.includes('report')) {
        return 'Generate Report';
    }
    if (input.includes('kpi')) {
        return 'Create KPI';
    }
    if (input.includes('site') || input.includes('location')) {
        return 'Add Site';
    }
    if (input.includes('support')) {
        return 'Open Support';
    }
    if (input.includes('export') || input.includes('snapshot')) {
        return 'Export Snapshot';
    }
    return 'Start Workflow';
}

function buildIntakePreview(intake) {
    if (!intake) {
        return '';
    }

    const rows = intake.preview || [];
    const warnings = intake.warnings || [];
    const summary = intake.summary || {};
    const docType = intake.doc_type || 'document';
    const warningCount = Array.isArray(warnings) ? warnings.length : 0;

    let tableHtml = '';
    rows.forEach((row) => {
        const cells = Object.entries(row.data || {}).slice(0, 4).map(([key, value]) => {
            return `<div><strong>${key}</strong>: ${value ?? '-'}</div>`;
        }).join('');
        tableHtml += `<div class="chatbot-row">${cells}</div>`;
    });

    return `
        <div class="chatbot-card">
            <div class="chatbot-card-title">Intake Preview (${docType})</div>
            <div class="chatbot-card-sub">Rows: ${summary.rows || 0} · Errors: ${summary.errors || 0} · Warnings: ${summary.warnings || warningCount}</div>
            <div class="chatbot-card-body">${tableHtml || '<div class="chatbot-row">No preview rows available.</div>'}</div>
            <div class="chatbot-card-actions">
                <button class="btn btn-primary" data-intake-apply="${intake.batch_id}">Review & Apply</button>
            </div>
        </div>
    `;
}

async function handleChatCommand(input) {
    const normalized = input.toLowerCase().trim();
    if (!normalized && pendingUploads.length === 0) {
        return;
    }

    appendChatBubble(input, 'user');
    pushChatHistory('user', input);

    if (normalized === 'help' || normalized === '?' || normalized.includes('help me')) {
        appendChatBubble('Try: "open assets", "create work order", or "upload asset register".', 'bot');
        return;
    }

    const navigationVerbs = ['open', 'go to', 'show', 'view', 'browse', 'navigate'];
    const actionVerbs = ['create', 'add', 'log', 'launch', 'start', 'upload', 'import', 'generate', 'invite', 'configure', 'acknowledge'];
    const wantsNavigation = navigationVerbs.some((verb) => normalized.includes(verb));
    const wantsAction = actionVerbs.some((verb) => normalized.includes(verb));
    const fallbackRoute = resolveChatRoute(normalized);

    const typingBubble = appendChatBubble('Thinking...', 'bot', { isTyping: true });
    let aiPayload = null;
    const attachments = pendingUploads;
    pendingUploads = [];

    try {
        aiPayload = await requestAiResponse(input, attachments);
    } catch (error) {
        aiPayload = null;
    }

    const aiReply = aiPayload && typeof aiPayload.reply === 'string' ? aiPayload.reply.trim() : '';
    if (typingBubble) {
        typingBubble.classList.remove('is-typing');
        typingBubble.textContent = aiReply;
    } else if (aiReply) {
        appendChatBubble(aiReply, 'bot');
    }

    if (aiReply) {
        pushChatHistory('assistant', aiReply);
    }

    if (aiPayload && aiPayload.conversation_id) {
        conversationId = String(aiPayload.conversation_id);
        localStorage.setItem('adm_chat_conversation', conversationId);
    }

    if (aiPayload && aiPayload.intake) {
        appendChatBubble(buildIntakePreview(aiPayload.intake), 'bot', { html: true });
    }

    let handled = false;
    const aiIntent = (aiPayload && aiPayload.intent ? String(aiPayload.intent).toLowerCase() : '');

    if (aiIntent === 'navigate') {
        const route = resolveChatRouteByPath(aiPayload.route) || resolveChatRouteByLabel(aiPayload.route);
        if (route) {
            handled = true;
            if (!aiReply && typingBubble) {
                typingBubble.textContent = `Opening ${route.label}...`;
            }
            window.setTimeout(() => {
                window.location.href = route.path;
            }, 400);
        }
    }

    if (!handled && aiIntent === 'action') {
        const label = aiPayload && aiPayload.action_label ? String(aiPayload.action_label).trim() : '';
        const actionLabel = label || resolveChatActionLabel(normalized);
        handled = true;
        if (!aiReply && typingBubble) {
            typingBubble.textContent = `Starting ${actionLabel.toLowerCase()}...`;
        }
        openActionModal(actionLabel);
    }

    if (!handled && aiIntent === 'upload') {
        handled = true;
        if (!aiReply && typingBubble) {
            typingBubble.textContent = 'Starting upload...';
        }
        openActionModal('Upload Import');
    }

    if (handled) {
        return;
    }

    if (wantsNavigation && fallbackRoute) {
        const message = `Opening ${fallbackRoute.label}...`;
        if (!aiReply && typingBubble) {
            typingBubble.textContent = message;
        } else {
            appendChatBubble(message, 'bot');
        }
        window.setTimeout(() => {
            window.location.href = fallbackRoute.path;
        }, 400);
        return;
    }

    if (wantsAction) {
        const label = resolveChatActionLabel(normalized);
        const message = `Starting ${label.toLowerCase()}...`;
        if (!aiReply && typingBubble) {
            typingBubble.textContent = message;
        } else {
            appendChatBubble(message, 'bot');
        }
        openActionModal(label);
        return;
    }

    if (fallbackRoute) {
        const message = `Opening ${fallbackRoute.label}...`;
        if (!aiReply && typingBubble) {
            typingBubble.textContent = message;
        } else {
            appendChatBubble(message, 'bot');
        }
        window.setTimeout(() => {
            window.location.href = fallbackRoute.path;
        }, 400);
        return;
    }

    if (!aiReply) {
        const fallbackMessage = 'I can open pages or start workflows. Try "open reports", "create work order", or "upload service sheets".';
        if (typingBubble) {
            typingBubble.textContent = fallbackMessage;
        } else {
            appendChatBubble(fallbackMessage, 'bot');
        }
    }
}

function resolveQuickAddRedirect() {
    if (!actionFields) {
        return '';
    }

    const select = actionFields.querySelector('select');
    const value = select ? select.value : '';
    if (!value) {
        return '';
    }

    if (value === 'Asset') {
        return `/ui/assets/${randomCode('A')}`;
    }
    if (value === 'Work Order') {
        return `/ui/work-orders/${randomCode('WO')}`;
    }
    if (value === 'Trip') {
        return `/ui/trips/${randomCode('T')}`;
    }
    if (value === 'Fuel Log') {
        return `/ui/fuel/${randomCode('FL')}`;
    }
    if (value === 'Device') {
        return `/ui/devices/${randomCode('DEV')}`;
    }

    return '';
}

if (actionModal) {
    actionModal.addEventListener('click', (event) => {
        if (event.target === actionModal) {
            closeActionModal();
        }
    });
}

actionCloseButtons.forEach((button) => {
    button.addEventListener('click', () => {
        closeActionModal();
    });
});

if (actionForm) {
    actionForm.addEventListener('submit', (event) => {
        event.preventDefault();
        const label = actionModal?.dataset.actionLabel || 'Action';
        let redirect = actionModal?.dataset.actionRedirect || '';
        const message = actionModal?.dataset.actionMessage || `${label} submitted.`;
        const actionType = actionModal?.dataset.actionType || '';

        if (actionType === 'purchase-request-item') {
            const itemId = actionModal?.dataset.actionId;
            const payload = {};
            actionFields?.querySelectorAll('[data-field-key]').forEach((input) => {
                payload[input.dataset.fieldKey] = input.value;
            });

            fetch(`/purchase-request-items/${itemId}`, {
                method: 'PATCH',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': csrfToken,
                    'Accept': 'application/json',
                },
                body: JSON.stringify(payload),
            })
                .then((response) => response.json())
                .then((data) => {
                    const row = document.querySelector(`[data-pr-item][data-item-id="${itemId}"]`);
                    if (row && data) {
                        row.dataset.itemName = data.item_name || '';
                        row.dataset.itemQty = data.quantity || '';
                        row.dataset.itemUnit = data.unit || '';
                        row.dataset.itemPurpose = data.purpose_cost_center || '';
                        row.dataset.itemSupplier = data.supplier_name || '';
                        row.dataset.itemSelected = data.selected_supplier || '';
                        row.dataset.itemUsd = data.quote_amount_usd || '';
                        row.dataset.itemZwl = data.quote_amount_zwl || '';
                        row.dataset.itemComments = data.comments || '';

                        const cells = row.querySelectorAll('div');
                        if (cells.length >= 6) {
                            cells[0].textContent = data.item_name || 'Item';
                            cells[1].textContent = data.quantity ? Number(data.quantity).toLocaleString() : '-';
                            cells[2].textContent = data.unit || '-';
                            const hasUsd = data.quote_amount_usd !== null && data.quote_amount_usd !== '';
                            const hasZwl = data.quote_amount_zwl !== null && data.quote_amount_zwl !== '';
                            let costText = '-';
                            if (hasUsd) {
                                costText = `$${Number(data.quote_amount_usd).toLocaleString()}`;
                            } else if (hasZwl) {
                                costText = `ZWL ${Number(data.quote_amount_zwl).toLocaleString()}`;
                            }
                            cells[3].textContent = costText;
                            cells[4].textContent = data.selected_supplier || data.supplier_name || '-';
                            const statusBadge = row.querySelector('.badge');
                            if (statusBadge) {
                                statusBadge.textContent = hasUsd || hasZwl ? 'Quoted' : 'Pending';
                                statusBadge.className = `badge ${hasUsd || hasZwl ? 'badge-live' : 'badge'}`;
                            }
                        }
                    }
                    closeActionModal();
                    showToast(message);
                })
                .catch(() => {
                    closeActionModal();
                    showToast('Update failed.');
                });
            return;
        }

        closeActionModal();
        showToast(message);
        if (label === 'Quick Add') {
            const quickRedirect = resolveQuickAddRedirect();
            if (quickRedirect) {
                redirect = quickRedirect;
            }
        }
        if (redirect) {
            window.setTimeout(() => {
                window.location.href = redirect;
            }, 400);
        }
    });
}

if (chatbotOpen) {
    chatbotOpen.addEventListener('click', () => {
        openChatbot();
    });
}

if (chatbotClose) {
    chatbotClose.addEventListener('click', () => {
        closeChatbot();
    });
}

if (chatbot) {
    chatbot.addEventListener('click', (event) => {
        if (event.target === chatbot) {
            closeChatbot();
        }
    });
}

if (chatbotBody) {
    chatbotBody.addEventListener('click', (event) => {
        const button = event.target.closest('[data-chatbot-quick]');
        if (!button) {
            const applyButton = event.target.closest('[data-intake-apply]');
            if (applyButton) {
                const batchId = applyButton.dataset.intakeApply;
                if (!batchId) {
                    return;
                }
                applyButton.disabled = true;
                fetch(`/api/ai/intake/${batchId}/apply`, { method: 'POST' })
                    .then((response) => response.json())
                    .then((payload) => {
                        appendChatBubble(`Applied intake batch ${batchId}.`, 'bot');
                        if (payload && payload.summary) {
                            appendChatBubble(`Applied ${payload.summary.applied || 0} rows.`, 'bot');
                        }
                    })
                    .catch(() => {
                        appendChatBubble('Unable to apply intake batch. Try again.', 'bot');
                    })
                    .finally(() => {
                        applyButton.disabled = false;
                    });
            }
            return;
        }

        void handleChatCommand(button.dataset.chatbotQuick || button.textContent.trim());
    });
}

if (chatbotForm) {
    chatbotForm.addEventListener('submit', (event) => {
        event.preventDefault();
        if (!chatbotText) {
            return;
        }

        const value = chatbotText.value.trim();
        chatbotText.value = '';

        if (!value && pendingUploads.length) {
            void handleChatCommand('Process these uploads');
            return;
        }

        if (!value) {
            return;
        }

        void handleChatCommand(value);
    });
}

if (chatbotFile) {
    chatbotFile.addEventListener('change', (event) => {
        const files = event.target.files ? Array.from(event.target.files) : [];
        if (!files.length) {
            return;
        }

        files.forEach((file) => {
            pendingUploads.push(file);
        });

        const names = files.map((file) => file.name).join(', ');
        appendChatBubble(`Attached file(s): ${names}`, 'user');
        appendChatBubble('Got it. Add a note or press Send to process.', 'bot');
        chatbotFile.value = '';
    });
}

document.addEventListener('click', (event) => {
    const button = event.target.closest('button.btn');
    if (!button) {
        return;
    }

    if (button.closest('[data-ui="action-modal"]') || button.closest('[data-chatbot]')) {
        return;
    }

    if (button.matches('[data-export-button], [data-pr-item-view], [data-pr-item-edit], [data-pr-item-attach]')) {
        return;
    }

    if (button.type === 'submit' && button.closest('form')) {
        return;
    }

    const label = button.dataset.actionLabel || button.textContent.trim() || 'Action';
    openActionModal(label);
    event.preventDefault();
});

const filterScopes = document.querySelectorAll('[data-filter-scope]');
filterScopes.forEach((scope) => {
    const input = scope.querySelector('[data-filter-input]');
    const statusSelect = scope.querySelector('[data-filter-status]');
    const siteSelect = scope.querySelector('[data-filter-site]');
    const rows = document.querySelectorAll('[data-row]');

    const applyFilters = () => {
        const query = (input?.value || '').toLowerCase().trim();
        const status = (statusSelect?.value || 'all').toLowerCase();
        const site = (siteSelect?.value || 'all').toLowerCase();

        rows.forEach((row) => {
            const search = (row.dataset.search || '').toLowerCase();
            const rowStatus = (row.dataset.status || '').toLowerCase();
            const rowSite = (row.dataset.site || '').toLowerCase();
            const matchesQuery = !query || search.includes(query);
            const matchesStatus = status === 'all' || rowStatus === status;
            const matchesSite = site === 'all' || rowSite === site;

            row.style.display = matchesQuery && matchesStatus && matchesSite ? '' : 'none';
        });
    };

    input?.addEventListener('input', applyFilters);
    statusSelect?.addEventListener('change', applyFilters);
    siteSelect?.addEventListener('change', applyFilters);
});

const exportButtons = document.querySelectorAll('[data-export-button]');
exportButtons.forEach((button) => {
    button.addEventListener('click', () => {
        const type = button.dataset.exportType || '';
        const selected = Array.from(document.querySelectorAll('[data-row-select]:checked')).map((input) => input.value);

        if (!selected.length) {
            showToast('Select rows to export.');
            return;
        }

        const params = new URLSearchParams({ type, ids: selected.join(',') });
        window.location.href = `/exports/supply-chain?${params.toString()}`;
    });
});

const csrfToken = document.querySelector('meta[name="csrf-token"]')?.getAttribute('content') || '';
const prAttachInput = document.createElement('input');
prAttachInput.type = 'file';
prAttachInput.multiple = true;
prAttachInput.style.display = 'none';
document.body.appendChild(prAttachInput);

document.addEventListener('click', (event) => {
    const viewBtn = event.target.closest('[data-pr-item-view]');
    if (viewBtn) {
        const itemId = viewBtn.dataset.prItemView;
        const details = document.querySelector(`[data-pr-item-details="${itemId}"]`);
        if (details) {
            details.hidden = !details.hidden;
        }
        return;
    }

    const editBtn = event.target.closest('[data-pr-item-edit]');
    if (editBtn) {
        const itemId = editBtn.dataset.prItemEdit;
        const row = document.querySelector(`[data-pr-item][data-item-id="${itemId}"]`);
        if (!row || !actionModal) {
            return;
        }

        actionModal.dataset.actionType = 'purchase-request-item';
        actionModal.dataset.actionId = itemId;
        actionModal.dataset.actionLabel = 'Edit Line Item';
        actionModal.dataset.actionMessage = 'Line item updated.';
        if (actionTitle) {
            actionTitle.textContent = 'Edit Line Item';
        }
        if (actionSub) {
            actionSub.textContent = 'Update item details and costs.';
        }

        buildFields([
            { label: 'Item', type: 'text', placeholder: 'Item name', key: 'item_name', value: row.dataset.itemName || '' },
            { label: 'Quantity', type: 'text', placeholder: '1', key: 'quantity', value: row.dataset.itemQty || '' },
            { label: 'Unit', type: 'text', placeholder: 'pcs', key: 'unit', value: row.dataset.itemUnit || '' },
            { label: 'Cost Center', type: 'text', placeholder: 'Operations', key: 'purpose_cost_center', value: row.dataset.itemPurpose || '' },
            { label: 'Supplier', type: 'text', placeholder: 'Supplier', key: 'supplier_name', value: row.dataset.itemSupplier || '' },
            { label: 'Selected Supplier', type: 'text', placeholder: 'Supplier', key: 'selected_supplier', value: row.dataset.itemSelected || '' },
            { label: 'Quote USD', type: 'text', placeholder: '0', key: 'quote_amount_usd', value: row.dataset.itemUsd || '' },
            { label: 'Quote ZWL', type: 'text', placeholder: '0', key: 'quote_amount_zwl', value: row.dataset.itemZwl || '' },
            { label: 'Comments', type: 'textarea', placeholder: 'Notes', key: 'comments', value: row.dataset.itemComments || '' },
        ]);

        body.classList.add('action-open');
        actionModal.setAttribute('aria-hidden', 'false');
        return;
    }

    const attachBtn = event.target.closest('[data-pr-item-attach]');
    if (attachBtn) {
        prAttachInput.dataset.itemId = attachBtn.dataset.prItemAttach;
        prAttachInput.click();
    }
});

prAttachInput.addEventListener('change', () => {
    const itemId = prAttachInput.dataset.itemId;
    if (!itemId || !prAttachInput.files.length) {
        return;
    }

    const files = Array.from(prAttachInput.files);
    files.forEach((file) => {
        const formData = new FormData();
        formData.append('file', file);

        fetch(`/purchase-request-items/${itemId}/documents`, {
            method: 'POST',
            headers: { 'X-CSRF-TOKEN': csrfToken },
            body: formData,
        })
            .then((response) => response.json())
            .then((payload) => {
                const list = document.querySelector(`[data-pr-doc-list="${itemId}"]`);
                if (!list || !payload?.download_url) {
                    return;
                }
                const item = document.createElement('li');
                const link = document.createElement('a');
                link.href = payload.download_url;
                link.textContent = payload.name || 'Attachment';
                item.appendChild(link);
                list.appendChild(item);
            })
            .catch(() => {
                showToast('Upload failed.');
            });
    });

    prAttachInput.value = '';
});
