<?php

$databasePath = __DIR__ . '/../database/database.sqlite';
$outPath = __DIR__ . '/../database/full_dump_mysql.sql';

if (!is_file($databasePath)) {
    fwrite(STDERR, "Database not found: {$databasePath}\n");
    exit(1);
}

$pdo = new PDO('sqlite:' . $databasePath, null, null, [
    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
]);

function normalizeDefault(?string $default): ?string
{
    if ($default === null) {
        return null;
    }

    $trimmed = trim($default, " \t\n\r\0\x0B()");
    if (stripos($trimmed, 'current_timestamp') !== false) {
        return 'CURRENT_TIMESTAMP';
    }

    return $default;
}

function mapType(string $type, bool $isPrimaryKey): string
{
    $type = strtolower(trim($type));
    if ($type === '') {
        return $isPrimaryKey ? 'BIGINT UNSIGNED' : 'TEXT';
    }

    if (preg_match('/tinyint/i', $type)) {
        return 'TINYINT';
    }
    if (preg_match('/smallint/i', $type)) {
        return 'SMALLINT';
    }
    if (preg_match('/bigint/i', $type)) {
        return $isPrimaryKey ? 'BIGINT UNSIGNED' : 'BIGINT';
    }
    if (preg_match('/int/i', $type)) {
        return $isPrimaryKey ? 'BIGINT UNSIGNED' : 'BIGINT';
    }
    if (preg_match('/bool/i', $type)) {
        return 'TINYINT(1)';
    }
    if (preg_match('/decimal\((\d+),\s*(\d+)\)/i', $type, $matches)) {
        return 'DECIMAL(' . $matches[1] . ',' . $matches[2] . ')';
    }
    if (preg_match('/numeric\((\d+),\s*(\d+)\)/i', $type, $matches)) {
        return 'DECIMAL(' . $matches[1] . ',' . $matches[2] . ')';
    }
    if (preg_match('/float|double|real/i', $type)) {
        return 'DOUBLE';
    }
    if (preg_match('/date/i', $type) && preg_match('/time/i', $type)) {
        return 'DATETIME';
    }
    if (preg_match('/datetime|timestamp/i', $type)) {
        return 'DATETIME';
    }
    if (preg_match('/date/i', $type)) {
        return 'DATE';
    }
    if (preg_match('/json/i', $type)) {
        return 'JSON';
    }
    if (preg_match('/char\\s*\\((\\d+)\\)/i', $type, $matches)) {
        return 'CHAR(' . $matches[1] . ')';
    }
    if (preg_match('/varchar\\s*\\((\\d+)\\)/i', $type, $matches)) {
        return 'VARCHAR(' . $matches[1] . ')';
    }
    if (preg_match('/text|clob/i', $type)) {
        return 'TEXT';
    }

    return 'TEXT';
}

function quoteIdentifier(string $value): string
{
    return '`' . str_replace('`', '``', $value) . '`';
}

$tables = $pdo->query("SELECT name FROM sqlite_master WHERE type = 'table' AND name NOT LIKE 'sqlite_%' ORDER BY name")
    ->fetchAll();

$objects = $pdo->query("SELECT type, name, sql FROM sqlite_master WHERE type IN ('index','trigger','view') AND sql IS NOT NULL ORDER BY type, name")
    ->fetchAll();

$fh = fopen($outPath, 'w');
if (!$fh) {
    fwrite(STDERR, "Unable to write to {$outPath}\n");
    exit(1);
}

fwrite($fh, "-- MySQL-compatible dump (source: SQLite)\n");
fwrite($fh, "SET FOREIGN_KEY_CHECKS=0;\n");
fwrite($fh, "START TRANSACTION;\n\n");

foreach ($tables as $table) {
    $name = $table['name'];
    $columns = $pdo->query("PRAGMA table_info(\"{$name}\")")->fetchAll();
    $foreignKeys = $pdo->query("PRAGMA foreign_key_list(\"{$name}\")")->fetchAll();

    $primaryKeys = [];
    foreach ($columns as $column) {
        if ((int) $column['pk'] > 0) {
            $primaryKeys[(int) $column['pk']] = $column['name'];
        }
    }
    ksort($primaryKeys);
    $primaryKeys = array_values($primaryKeys);

    $lines = [];
    foreach ($columns as $column) {
        $colName = quoteIdentifier($column['name']);
        $isPrimary = in_array($column['name'], $primaryKeys, true);
        $type = mapType((string) $column['type'], $isPrimary);
        $line = "{$colName} {$type}";

        if ((int) $column['notnull'] === 1) {
            $line .= ' NOT NULL';
        }

        $default = normalizeDefault($column['dflt_value'] ?? null);
        if ($default !== null) {
            $line .= ' DEFAULT ' . $default;
        }

        if ($isPrimary && count($primaryKeys) === 1 && preg_match('/int/i', (string) $column['type'])) {
            $line .= ' AUTO_INCREMENT';
        }

        $lines[] = $line;
    }

    if ($primaryKeys) {
        $pkCols = implode(', ', array_map('quoteIdentifier', $primaryKeys));
        $lines[] = "PRIMARY KEY ({$pkCols})";
    }

    if ($foreignKeys) {
        $fkGroups = [];
        foreach ($foreignKeys as $fk) {
            $fkGroups[$fk['id']][] = $fk;
        }
        foreach ($fkGroups as $group) {
            $fromCols = array_map(fn ($row) => quoteIdentifier($row['from']), $group);
            $toCols = array_map(fn ($row) => quoteIdentifier($row['to'] ?: $row['from']), $group);
            $target = quoteIdentifier($group[0]['table']);
            $constraint = "FOREIGN KEY (" . implode(', ', $fromCols) . ") REFERENCES {$target} (" . implode(', ', $toCols) . ")";
            if ($group[0]['on_delete']) {
                $constraint .= " ON DELETE " . strtoupper($group[0]['on_delete']);
            }
            if ($group[0]['on_update']) {
                $constraint .= " ON UPDATE " . strtoupper($group[0]['on_update']);
            }
            $lines[] = $constraint;
        }
    }

    fwrite($fh, "-- Table: {$name}\n");
    fwrite($fh, "DROP TABLE IF EXISTS " . quoteIdentifier($name) . ";\n");
    fwrite($fh, "CREATE TABLE " . quoteIdentifier($name) . " (\n    " . implode(",\n    ", $lines) . "\n) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;\n\n");

    $stmt = $pdo->query("SELECT * FROM \"{$name}\"");
    while ($row = $stmt->fetch()) {
        $columns = array_keys($row);
        $values = array_map(function ($value) use ($pdo) {
            if ($value === null) {
                return 'NULL';
            }
            if (is_numeric($value) && !preg_match('/^0[0-9]/', (string) $value)) {
                return (string) $value;
            }
            return $pdo->quote((string) $value);
        }, array_values($row));

        $colList = implode(', ', array_map('quoteIdentifier', $columns));
        $valList = implode(', ', $values);
        fwrite($fh, "INSERT INTO " . quoteIdentifier($name) . " ({$colList}) VALUES ({$valList});\n");
    }

    fwrite($fh, "\n");
}

if ($objects) {
    fwrite($fh, "-- Indexes / triggers / views (SQLite SQL)\n");
    foreach ($objects as $object) {
        fwrite($fh, "-- " . $object['type'] . ' ' . $object['name'] . "\n");
        fwrite($fh, $object['sql'] . ";\n");
    }
    fwrite($fh, "\n");
}

fwrite($fh, "COMMIT;\n");
fwrite($fh, "SET FOREIGN_KEY_CHECKS=1;\n");
fclose($fh);

echo "Wrote {$outPath}\n";
