<?php
$databasePath = __DIR__ . '/../database/database.sqlite';
$outPath = __DIR__ . '/../database/full_dump.sql';

if (!is_file($databasePath)) {
    fwrite(STDERR, "Database not found: {$databasePath}\n");
    exit(1);
}

$pdo = new PDO('sqlite:' . $databasePath, null, null, [
    PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
    PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
]);

$tables = $pdo->query("SELECT name, sql FROM sqlite_master WHERE type = 'table' AND name NOT LIKE 'sqlite_%' ORDER BY name")
    ->fetchAll();

$objects = $pdo->query("SELECT type, name, sql FROM sqlite_master WHERE type IN ('index','trigger','view') AND sql IS NOT NULL ORDER BY type, name")
    ->fetchAll();

$fh = fopen($outPath, 'w');
if (!$fh) {
    fwrite(STDERR, "Unable to write to {$outPath}\n");
    exit(1);
}

fwrite($fh, "-- SQLite full dump\n");
fwrite($fh, "PRAGMA foreign_keys=OFF;\n");
fwrite($fh, "BEGIN TRANSACTION;\n\n");

foreach ($tables as $table) {
    $name = $table['name'];
    $sql = $table['sql'];
    if (!$sql) {
        continue;
    }

    fwrite($fh, "-- Table: {$name}\n");
    fwrite($fh, $sql . ";\n\n");

    $stmt = $pdo->query("SELECT * FROM \"{$name}\"");
    while ($row = $stmt->fetch()) {
        $columns = array_keys($row);
        $values = array_map(function ($value) use ($pdo) {
            if ($value === null) {
                return 'NULL';
            }
            return $pdo->quote((string) $value);
        }, array_values($row));

        $colList = implode(', ', array_map(fn ($col) => '"' . $col . '"', $columns));
        $valList = implode(', ', $values);
        fwrite($fh, "INSERT INTO \"{$name}\" ({$colList}) VALUES ({$valList});\n");
    }

    fwrite($fh, "\n");
}

if ($objects) {
    fwrite($fh, "-- Indexes / triggers / views\n");
    foreach ($objects as $object) {
        fwrite($fh, $object['sql'] . ";\n");
    }
    fwrite($fh, "\n");
}

fwrite($fh, "COMMIT;\n");
fclose($fh);

echo "Wrote {$outPath}\n";
