<?php

require __DIR__ . '/../vendor/autoload.php';

use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\Worksheet\Drawing;

$outputDir = __DIR__ . '/../public/exports/upload-templates';
if (!is_dir($outputDir)) {
    mkdir($outputDir, 0777, true);
}

$logoPath = __DIR__ . '/../public/ui-assets/images/fourways-logo.png';

function columnLetter(int $index): string
{
    $result = '';
    while ($index > 0) {
        $index--;
        $result = chr(65 + ($index % 26)) . $result;
        $index = intdiv($index, 26);
    }
    return $result;
}

function addBranding($sheet, ?string $logoPath, bool $withWatermark = true): void
{
    if (!$logoPath || !is_file($logoPath)) {
        return;
    }

    $logo = new Drawing();
    $logo->setName('Logo');
    $logo->setPath($logoPath);
    $logo->setHeight(48);
    $logo->setCoordinates('A1');
    $logo->setWorksheet($sheet);

    if (!$withWatermark) {
        return;
    }

    $watermark = new Drawing();
    $watermark->setName('Watermark');
    $watermark->setPath($logoPath);
    $watermark->setHeight(260);
    $watermark->setCoordinates('C6');
    if (method_exists($watermark, 'setOpacity')) {
        $watermark->setOpacity(0.08);
    }
    $watermark->setWorksheet($sheet);
}

function createTemplate(
    string $filename,
    string $title,
    array $headers,
    array $rows,
    array $notes,
    ?string $logoPath,
    bool $withWatermark
): void
{
    $spreadsheet = new Spreadsheet();
    $sheet = $spreadsheet->getActiveSheet();
    $sheet->setTitle(substr($title, 0, 31));

    addBranding($sheet, $logoPath, $withWatermark);

    $endCol = columnLetter(count($headers));
    $sheet->setCellValue('A4', $title);
    $sheet->mergeCells("A4:{$endCol}4");
    $sheet->getStyle('A4')->getFont()->setBold(true)->setSize(14);

    $noteRow = 6;
    foreach ($notes as $note) {
        $sheet->setCellValue("A{$noteRow}", $note);
        $sheet->mergeCells("A{$noteRow}:{$endCol}{$noteRow}");
        $sheet->getStyle("A{$noteRow}")->getFont()->setSize(10)->setItalic(true);
        $noteRow++;
    }

    $headerRow = $noteRow + 1;
    $sheet->fromArray($headers, null, "A{$headerRow}");
    $sheet->getStyle("A{$headerRow}:{$endCol}{$headerRow}")->getFont()->setBold(true);
    $sheet->getStyle("A{$headerRow}:{$endCol}{$headerRow}")->getFill()->setFillType('solid')->getStartColor()->setARGB('FF0B1220');
    $sheet->getStyle("A{$headerRow}:{$endCol}{$headerRow}")->getFont()->getColor()->setARGB('FFE9EEF6');

    $rowIndex = $headerRow + 1;
    foreach ($rows as $row) {
        $sheet->fromArray($row, null, "A{$rowIndex}");
        $rowIndex++;
    }

    foreach (range('A', $endCol) as $column) {
        $sheet->getColumnDimension($column)->setAutoSize(true);
    }

    $writer = new Xlsx($spreadsheet);
    $writer->save($filename);
}

$templates = [
    [
        'file' => $outputDir . '/asset-register-template.xlsx',
        'title' => 'Asset Register Upload Template',
        'headers' => ['asset_tag', 'name', 'make', 'model', 'category', 'location', 'serial_number', 'purchase_date', 'current_value'],
        'rows' => [
            ['A-100', 'Hauler 10', 'Caterpillar', '797F', 'Vehicle', 'Fourways Quarry', 'CAT797F-0001', '2022-04-10', '420000'],
            ['A-101', 'Crusher 6', 'Metso', 'HP400', 'Fixed Plant', 'Fourways Quarry', 'MET-HP400-22', '2020-11-02', '210000'],
        ],
        'notes' => ['Provide one asset per row. Dates should be YYYY-MM-DD.'],
    ],
    [
        'file' => $outputDir . '/purchase-request-template.xlsx',
        'title' => 'Purchase Request Upload Template',
        'headers' => ['item_ordered', 'quantity', 'purpose_cost_centre', 'date_initiated', 'suppliers', 'quote_number', 'quote_amount_usd', 'quote_amount_zwl', 'selected_supplier', 'comments'],
        'rows' => [
            ['Hydraulic Pump Kit', '2', 'Maintenance', '2026-01-12', 'Summit Industrial', 'Q-4481', '14200', '191700000', 'Summit Industrial', 'Urgent'],
            ['Conveyor Belt Set', '1', 'Operations', '2026-01-13', 'Prime Spares', 'Q-4510', '8200', '110700000', 'Prime Spares', ''],
        ],
        'notes' => ['Use USD or ZWL columns. Supplier names should match your vendor list.'],
    ],
    [
        'file' => $outputDir . '/service-sheet-template.xlsx',
        'title' => 'Service Sheet Upload Template',
        'headers' => ['asset_tag', 'asset_name', 'description', 'completed_at', 'meter_reading', 'labor_cost', 'parts_cost', 'contractor_cost', 'work_order_ref', 'root_cause'],
        'rows' => [
            ['A-100', 'Hauler 10', 'Replace hydraulic seals', '2026-01-10', '12450', '1200', '3400', '0', 'WO-221', 'Seal wear'],
        ],
        'notes' => ['Include either asset_tag or asset_name. Costs should be numeric.'],
    ],
    [
        'file' => $outputDir . '/invoice-template.xlsx',
        'title' => 'Invoice Upload Template',
        'headers' => ['amount', 'work_order_ref', 'asset_tag', 'asset_name', 'cost_type', 'description', 'invoice_number'],
        'rows' => [
            ['74200', 'WO-221', 'A-100', 'Hauler 10', 'contractor', 'Hydraulic repair invoice', 'INV-4481'],
        ],
        'notes' => ['Use work_order_ref when available. cost_type: labor, parts, contractor.'],
    ],
    [
        'file' => $outputDir . '/fuel-sheet-template.xlsx',
        'title' => 'Fuel Sheet Upload Template',
        'headers' => ['quantity', 'asset_tag', 'asset_name', 'source', 'fuel_type', 'unit_cost', 'total_cost', 'logged_at'],
        'rows' => [
            ['420', 'A-100', 'Hauler 10', 'Bowser', 'Diesel', '1.40', '588', '2026-01-12'],
        ],
        'notes' => ['Quantity in liters. logged_at should be YYYY-MM-DD.'],
    ],
    [
        'file' => $outputDir . '/meter-readings-template.xlsx',
        'title' => 'Meter Readings Upload Template',
        'headers' => ['reading', 'asset_tag', 'asset_name', 'meter_type', 'reading_at'],
        'rows' => [
            ['12540', 'A-100', 'Hauler 10', 'hours', '2026-01-12'],
        ],
        'notes' => ['meter_type examples: hours, km, odometer.'],
    ],
    [
        'file' => $outputDir . '/grn-template.xlsx',
        'title' => 'Goods Received Note Upload Template',
        'headers' => ['reference', 'supplier', 'received_at', 'items'],
        'rows' => [
            ['GRN-114', 'Summit Industrial', '2026-01-12', 'Hydraulic Pump Kit x2; Seal Set x4'],
        ],
        'notes' => ['List items in a readable format or JSON.'],
    ],
    [
        'file' => $outputDir . '/issue-voucher-template.xlsx',
        'title' => 'Issue Voucher Upload Template',
        'headers' => ['reference', 'issued_to', 'issued_at', 'items'],
        'rows' => [
            ['IV-3001', 'Workshop - Bay 2', '2026-01-12', 'Hydraulic Oil 20L x3'],
        ],
        'notes' => ['issued_to can be a person, site, or department.'],
    ],
];

foreach ($templates as $template) {
    createTemplate(
        $template['file'],
        $template['title'],
        $template['headers'],
        $template['rows'],
        $template['notes'],
        $logoPath,
        false
    );
}

echo "Templates generated in {$outputDir}\n";
