const fs = require('fs');
const path = require('path');

const root = path.resolve(__dirname, '..');
const viewsDir = path.join(root, 'resources', 'views', 'ui');
const layoutPath = path.join(viewsDir, 'layouts', 'app.blade.php');
const sidebarPath = path.join(viewsDir, 'partials', 'sidebar.blade.php');
const topbarPath = path.join(viewsDir, 'partials', 'topbar.blade.php');
const cssPath = path.join(root, 'resources', 'css', 'app.css');
const jsPath = path.join(root, 'resources', 'js', 'app.js');

const outputDir = path.join(root, 'public', 'ui-previews');

const layout = fs.readFileSync(layoutPath, 'utf8');
const sidebarTemplate = fs.readFileSync(sidebarPath, 'utf8');
const topbar = fs.readFileSync(topbarPath, 'utf8');
const css = fs.readFileSync(cssPath, 'utf8');
const js = fs.readFileSync(jsPath, 'utf8');

function extractTitle(view) {
    const match = view.match(/@section\\('title',\\s*'([^']+)'\\)/);
    return match ? match[1] : 'Fleet OS';
}

function extractActive(view) {
    const match = view.match(/@php\\(\\$active\\s*=\\s*'([^']+)'\\)/);
    return match ? match[1] : 'dashboard';
}

function extractContent(view) {
    const start = "@section('content')";
    const end = '@endsection';
    const startIndex = view.indexOf(start);
    if (startIndex === -1) {
        return '';
    }
    const endIndex = view.indexOf(end, startIndex);
    if (endIndex === -1) {
        return '';
    }
    return view.slice(startIndex + start.length, endIndex).trim();
}

function renderSidebar(active) {
    return sidebarTemplate.replace(
        /{{ \\(\\$active \\?\\? ''\\) === '([^']+)' \\? 'is-active' : '' }}/g,
        (_, name) => (name === active ? 'is-active' : '')
    );
}

function renderHtml(viewFile) {
    const viewPath = path.join(viewsDir, viewFile);
    const view = fs.readFileSync(viewPath, 'utf8');
    const title = extractTitle(view);
    const active = extractActive(view);
    const content = extractContent(view);
    const sidebar = renderSidebar(active);

    let html = layout;
    html = html.replace("@yield('title', 'Fleet OS')", title);
    html = html.replace("@include('ui.partials.sidebar', ['active' => $active ?? 'dashboard'])", sidebar);
    html = html.replace("@include('ui.partials.topbar')", topbar);
    html = html.replace("@yield('content')", content);
    html = html.replace(
        "@vite(['resources/css/app.css', 'resources/js/app.js'])",
        `<style>${css}</style><script>${js}</script>`
    );

    return html;
}

function writePreview(viewFile) {
    const name = viewFile.replace('.blade.php', '');
    const outputPath = path.join(outputDir, `${name}.html`);
    const html = renderHtml(viewFile);
    fs.mkdirSync(outputDir, { recursive: true });
    fs.writeFileSync(outputPath, html, 'utf8');
    return outputPath;
}

const files = fs
    .readdirSync(viewsDir)
    .filter((file) => file.endsWith('.blade.php'));

const rendered = files.map(writePreview);
console.log(`Rendered ${rendered.length} preview files to ${outputDir}`);
