<?php

namespace Tests\Feature;

use App\Models\Asset;
use App\Models\Part;
use App\Models\Site;
use App\Models\Tenant;
use App\Models\User;
use App\Models\Vehicle;
use App\Jobs\ProcessImportBatch;
use App\Jobs\RunCostControlAgent;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Queue;
use Laravel\Sanctum\Sanctum;
use Tests\TestCase;

class ApiPostOnlyWorkflowTest extends TestCase
{
    use RefreshDatabase;

    public function test_post_only_core_module_create_workflows(): void
    {
        $tenant = Tenant::factory()->create();
        $user = $this->createUser($tenant);
        Sanctum::actingAs($user);

        $siteResponse = $this->postJson('/api/sites', [
            'name' => 'North Pit',
            'code' => 'NP-01',
            'active' => true,
        ]);
        $siteResponse->assertCreated()->assertJsonPath('name', 'North Pit');
        $this->assertDatabaseHas('sites', [
            'tenant_id' => $tenant->id,
            'name' => 'North Pit',
            'code' => 'NP-01',
        ]);

        $checklistResponse = $this->postJson('/api/checklists', [
            'name' => 'Pre-start Inspection',
            'type' => 'daily',
            'items' => ['Hydraulics', 'Tyres'],
            'active' => true,
        ]);
        $checklistResponse->assertCreated()->assertJsonPath('name', 'Pre-start Inspection');
        $this->assertDatabaseHas('checklists', [
            'tenant_id' => $tenant->id,
            'name' => 'Pre-start Inspection',
            'type' => 'daily',
        ]);

        $reasonResponse = $this->postJson('/api/downtime-reasons', [
            'code' => 'HYD',
            'label' => 'Hydraulic Failure',
            'active' => true,
        ]);
        $reasonResponse->assertCreated()->assertJsonPath('label', 'Hydraulic Failure');
        $this->assertDatabaseHas('downtime_reason_codes', [
            'tenant_id' => $tenant->id,
            'code' => 'HYD',
            'label' => 'Hydraulic Failure',
        ]);

        $partResponse = $this->postJson('/api/parts', [
            'sku' => 'SEAL-100',
            'name' => 'Hydraulic Seal Kit',
            'unit' => 'kit',
        ]);
        $partResponse->assertCreated()->assertJsonPath('name', 'Hydraulic Seal Kit');
        $this->assertDatabaseHas('parts', [
            'tenant_id' => $tenant->id,
            'sku' => 'SEAL-100',
            'name' => 'Hydraulic Seal Kit',
        ]);

        $vendorResponse = $this->postJson('/api/vendors', [
            'name' => 'Industrial Spares Co',
            'email' => 'sales@spares.example',
            'lead_time_days' => 4,
        ]);
        $vendorResponse->assertCreated()->assertJsonPath('name', 'Industrial Spares Co');
        $this->assertDatabaseHas('vendors', [
            'tenant_id' => $tenant->id,
            'name' => 'Industrial Spares Co',
            'lead_time_days' => 4,
        ]);

        $alarmRuleResponse = $this->postJson('/api/alarm-rules', [
            'name' => 'High Engine Temp',
            'metric' => 'engine_temp',
            'comparison' => '>',
            'threshold' => 98.5,
            'severity' => 'critical',
            'enabled' => true,
        ]);
        $alarmRuleResponse->assertCreated()->assertJsonPath('name', 'High Engine Temp');
        $this->assertDatabaseHas('alarm_rules', [
            'tenant_id' => $tenant->id,
            'name' => 'High Engine Temp',
            'metric' => 'engine_temp',
        ]);

        $channelResponse = $this->postJson('/api/notifications/channels', [
            'type' => 'telegram',
            'name' => 'Ops Telegram',
            'enabled' => true,
        ]);
        $channelResponse->assertCreated()->assertJsonPath('type', 'telegram');
        $this->assertDatabaseHas('notification_channels', [
            'tenant_id' => $tenant->id,
            'type' => 'telegram',
            'name' => 'Ops Telegram',
        ]);

        $routeResponse = $this->postJson('/api/notifications/routes', [
            'event_type' => 'alarm.triggered',
            'channel_order' => ['telegram'],
            'recipients' => ['ops-team'],
            'escalation_minutes' => 15,
        ]);
        $routeResponse->assertCreated()->assertJsonPath('event_type', 'alarm.triggered');
        $this->assertDatabaseHas('notification_routes', [
            'tenant_id' => $tenant->id,
            'event_type' => 'alarm.triggered',
            'escalation_minutes' => 15,
        ]);

        $roleResponse = $this->postJson('/api/roles', [
            'name' => 'Maintenance Planner',
            'description' => 'Plans work and schedules',
        ]);
        $roleResponse->assertCreated()->assertJsonPath('name', 'Maintenance Planner');
        $roleId = $roleResponse->json('id');
        $this->assertDatabaseHas('roles', [
            'id' => $roleId,
            'tenant_id' => $tenant->id,
            'name' => 'Maintenance Planner',
        ]);

        $assignResponse = $this->postJson('/api/roles/assign/' . $user->id, [
            'role_ids' => [$roleId],
        ]);
        $assignResponse->assertOk();
        $this->assertDatabaseHas('role_user', [
            'role_id' => $roleId,
            'user_id' => $user->id,
        ]);

        $deviceResponse = $this->postJson('/api/devices', [
            'name' => 'Engine Sensor',
            'type' => 'temperature',
            'identifier' => 'eng-temp-01',
            'meta' => ['unit' => 'celsius'],
        ]);
        $deviceResponse->assertCreated()->assertJsonPath('identifier', 'eng-temp-01');
        $this->assertDatabaseHas('devices', [
            'tenant_id' => $tenant->id,
            'identifier' => 'eng-temp-01',
        ]);
    }

    public function test_post_only_dependent_module_create_workflows(): void
    {
        Queue::fake();

        $tenant = Tenant::factory()->create();
        $user = $this->createUser($tenant);
        $site = Site::create([
            'tenant_id' => $tenant->id,
            'name' => 'Main Yard',
            'code' => 'MY-01',
            'active' => true,
        ]);
        $asset = Asset::factory()->create([
            'tenant_id' => $tenant->id,
            'site_id' => $site->id,
            'status' => 'active',
        ]);
        $part = Part::create([
            'tenant_id' => $tenant->id,
            'sku' => 'FILTER-001',
            'name' => 'Fuel Filter',
            'unit' => 'ea',
        ]);
        $vehicle = Vehicle::factory()->create([
            'tenant_id' => $tenant->id,
            'asset_id' => $asset->id,
            'site_id' => $site->id,
        ]);

        Sanctum::actingAs($user);

        $pmResponse = $this->postJson('/api/pm-schedules', [
            'asset_id' => $asset->id,
            'name' => '500 Hour Service',
            'schedule_type' => 'meter',
            'interval_value' => 500,
            'interval_unit' => 'hours',
            'active' => true,
        ]);
        $pmResponse->assertCreated()->assertJsonPath('name', '500 Hour Service');
        $this->assertDatabaseHas('pm_schedules', [
            'tenant_id' => $tenant->id,
            'asset_id' => $asset->id,
            'name' => '500 Hour Service',
        ]);

        $fuelResponse = $this->postJson('/api/fuel-logs', [
            'asset_id' => $asset->id,
            'site_id' => $site->id,
            'operator_id' => $user->id,
            'fuel_type' => 'diesel',
            'quantity' => 125.5,
            'unit_cost' => 1.48,
        ]);
        $fuelResponse->assertCreated()->assertJsonPath('fuel_type', 'diesel');
        $this->assertDatabaseHas('fuel_logs', [
            'tenant_id' => $tenant->id,
            'asset_id' => $asset->id,
            'site_id' => $site->id,
            'operator_id' => $user->id,
        ]);
        Queue::assertPushed(RunCostControlAgent::class);

        $tyreResponse = $this->postJson('/api/tyres', [
            'asset_id' => $asset->id,
            'position' => 'front-left',
            'serial_number' => 'TY-44551',
            'status' => 'active',
        ]);
        $tyreResponse->assertCreated()->assertJsonPath('serial_number', 'TY-44551');
        $this->assertDatabaseHas('tyres', [
            'tenant_id' => $tenant->id,
            'asset_id' => $asset->id,
            'serial_number' => 'TY-44551',
        ]);

        $locationResponse = $this->postJson('/api/inventory-locations', [
            'site_id' => $site->id,
            'name' => 'Container A',
            'code' => 'CON-A',
        ]);
        $locationResponse->assertCreated()->assertJsonPath('name', 'Container A');
        $locationId = $locationResponse->json('id');

        $itemResponse = $this->postJson('/api/inventory-items', [
            'part_id' => $part->id,
            'location_id' => $locationId,
            'quantity' => 40,
            'reorder_point' => 10,
        ]);
        $itemResponse->assertCreated()->assertJsonPath('part_id', $part->id);
        $this->assertDatabaseHas('inventory_items', [
            'tenant_id' => $tenant->id,
            'part_id' => $part->id,
            'location_id' => $locationId,
            'quantity' => 40,
        ]);

        $telemetryResponse = $this->postJson('/api/telemetry', [
            'vehicle_id' => $vehicle->id,
            'data' => [[
                'timestamp' => now()->toIso8601String(),
                'usage_hours' => 222.75,
                'asset_id' => $asset->id,
            ]],
        ]);
        $telemetryResponse->assertCreated()->assertJsonPath('message', 'Telemetry data ingested successfully.');
        $this->assertDatabaseHas('telemetries', [
            'tenant_id' => $tenant->id,
            'vehicle_id' => $vehicle->id,
            'asset_id' => $asset->id,
            'usage_hours' => 222.75,
        ]);
    }

    public function test_import_module_post_workflow_creates_batch(): void
    {
        Queue::fake();

        $tenant = Tenant::factory()->create();
        $user = $this->createUser($tenant);
        Sanctum::actingAs($user);

        $file = UploadedFile::fake()->createWithContent(
            'assets.csv',
            "name,asset_tag\nExcavator A,AST-1001\n"
        );

        $response = $this->post('/api/imports', [
            'file' => $file,
            'import_type' => 'assets',
        ], [
            'Accept' => 'application/json',
        ]);

        $response->assertCreated();
        $batchId = $response->json('id');

        $this->assertDatabaseHas('import_batches', [
            'id' => $batchId,
            'tenant_id' => $tenant->id,
            'created_by' => $user->id,
            'import_type' => 'assets',
        ]);
        Queue::assertPushed(ProcessImportBatch::class);
    }

    public function test_post_only_core_module_validation_failures(): void
    {
        $tenant = Tenant::factory()->create();
        $user = $this->createUser($tenant);
        Sanctum::actingAs($user);

        $this->postJson('/api/sites', [])->assertStatus(422)->assertJsonValidationErrors(['name']);
        $this->postJson('/api/checklists', [])->assertStatus(422)->assertJsonValidationErrors(['name']);
        $this->postJson('/api/downtime-reasons', [])->assertStatus(422)->assertJsonValidationErrors(['label']);
        $this->postJson('/api/parts', [])->assertStatus(422)->assertJsonValidationErrors(['name']);
        $this->postJson('/api/vendors', [])->assertStatus(422)->assertJsonValidationErrors(['name']);
        $this->postJson('/api/alarm-rules', [])->assertStatus(422)->assertJsonValidationErrors(['name']);
        $this->postJson('/api/notifications/channels', [])->assertStatus(422)->assertJsonValidationErrors(['type']);
        $this->postJson('/api/notifications/routes', [])->assertStatus(422)->assertJsonValidationErrors(['event_type']);
        $this->postJson('/api/roles', [])->assertStatus(422)->assertJsonValidationErrors(['name']);
        $this->postJson('/api/devices', ['meta' => 'invalid'])->assertStatus(422)->assertJsonValidationErrors(['meta']);

        $this->postJson('/api/roles/assign/' . $user->id, [])
            ->assertStatus(422)
            ->assertJsonValidationErrors(['role_ids']);
    }

    public function test_post_only_modules_enforce_tenant_isolation_on_foreign_references(): void
    {
        $tenantA = Tenant::factory()->create();
        $tenantB = Tenant::factory()->create();

        $userA = $this->createUser($tenantA);
        $userB = $this->createUser($tenantB);

        $siteB = Site::create([
            'tenant_id' => $tenantB->id,
            'name' => 'Foreign Site',
            'code' => 'FS-01',
        ]);
        $assetB = Asset::factory()->create([
            'tenant_id' => $tenantB->id,
            'site_id' => $siteB->id,
            'status' => 'active',
        ]);
        $partB = Part::create([
            'tenant_id' => $tenantB->id,
            'name' => 'Foreign Part',
        ]);
        $vehicleB = Vehicle::factory()->create([
            'tenant_id' => $tenantB->id,
            'asset_id' => $assetB->id,
            'site_id' => $siteB->id,
        ]);

        Sanctum::actingAs($userA);

        $this->postJson('/api/pm-schedules', [
            'asset_id' => $assetB->id,
            'name' => 'Foreign PM',
        ])->assertStatus(422)->assertJsonValidationErrors(['asset_id']);

        $this->postJson('/api/fuel-logs', [
            'asset_id' => $assetB->id,
        ])->assertStatus(422)->assertJsonValidationErrors(['asset_id']);

        $this->postJson('/api/tyres', [
            'asset_id' => $assetB->id,
        ])->assertStatus(422)->assertJsonValidationErrors(['asset_id']);

        $this->postJson('/api/inventory-locations', [
            'site_id' => $siteB->id,
            'name' => 'Foreign Location',
        ])->assertStatus(422)->assertJsonValidationErrors(['site_id']);

        $this->postJson('/api/inventory-items', [
            'part_id' => $partB->id,
            'quantity' => 5,
        ])->assertStatus(422)->assertJsonValidationErrors(['part_id']);

        $this->postJson('/api/devices', [
            'asset_id' => $assetB->id,
            'identifier' => 'foreign-device',
        ])->assertStatus(422)->assertJsonValidationErrors(['asset_id']);

        $this->postJson('/api/telemetry', [
            'vehicle_id' => $vehicleB->id,
            'data' => [[
                'timestamp' => now()->toIso8601String(),
            ]],
        ])->assertStatus(404);

        $roleAResponse = $this->postJson('/api/roles', [
            'name' => 'Tenant A Role',
        ])->assertCreated();
        $roleAId = $roleAResponse->json('id');

        $roleB = \App\Models\Role::create([
            'tenant_id' => $tenantB->id,
            'name' => 'Tenant B Role',
        ]);

        $this->postJson('/api/roles/assign/' . $userB->id, [
            'role_ids' => [$roleAId],
        ])->assertStatus(403);

        $this->postJson('/api/roles/assign/' . $userA->id, [
            'role_ids' => [$roleB->id],
        ])->assertOk();

        $this->assertDatabaseMissing('role_user', [
            'role_id' => $roleB->id,
            'user_id' => $userA->id,
        ]);
    }

    public function test_import_module_validation_failures(): void
    {
        $tenant = Tenant::factory()->create();
        $user = $this->createUser($tenant);
        Sanctum::actingAs($user);

        $this->post('/api/imports', [], ['Accept' => 'application/json'])
            ->assertStatus(422)
            ->assertJsonValidationErrors(['file']);

        $file = UploadedFile::fake()->createWithContent(
            'assets.csv',
            "name,asset_tag\nExcavator B,AST-2002\n"
        );

        $this->post('/api/imports', [
            'file' => $file,
            'import_type' => 'invalid-type',
        ], ['Accept' => 'application/json'])
            ->assertStatus(422)
            ->assertJsonValidationErrors(['import_type']);
    }

    private function createUser(Tenant $tenant): User
    {
        static $counter = 1;

        $user = User::create([
            'name' => 'Post Workflow Tester ' . $counter,
            'email' => 'post-workflow-tester-' . $counter . '@example.com',
            'password' => Hash::make('password123'),
            'tenant_id' => $tenant->id,
        ]);

        $counter++;

        return $user;
    }
}
