<?php

namespace Tests\Feature;

use App\Models\Asset;
use App\Models\Tenant;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Hash;
use Laravel\Sanctum\Sanctum;
use Tests\TestCase;

class ApiPostPutWorkflowTest extends TestCase
{
    use RefreshDatabase;

    public function test_tenant_module_create_and_update_workflow(): void
    {
        $user = $this->createUser();

        Sanctum::actingAs($user);

        $createPayload = [
            'name' => 'Workflow Tenant',
            'email' => 'workflow-tenant@example.com',
            'subdomain' => 'workflow-tenant',
            'password' => 'secret123',
        ];

        $createResponse = $this->postJson('/api/tenants', $createPayload);
        $createResponse->assertCreated()
            ->assertJsonPath('name', 'Workflow Tenant');

        $tenantId = $createResponse->json('id');

        $this->assertDatabaseHas('tenants', [
            'id' => $tenantId,
            'email' => 'workflow-tenant@example.com',
            'subdomain' => 'workflow-tenant',
        ]);

        $user->tenant_id = $tenantId;
        $user->save();

        Sanctum::actingAs($user->fresh());

        $updateResponse = $this->putJson('/api/tenants/' . $tenantId, [
            'name' => 'Workflow Tenant Updated',
            'email' => 'workflow-tenant-updated@example.com',
            'subdomain' => 'workflow-tenant-updated',
        ]);

        $updateResponse->assertOk()
            ->assertJsonPath('name', 'Workflow Tenant Updated')
            ->assertJsonPath('email', 'workflow-tenant-updated@example.com');

        $this->assertDatabaseHas('tenants', [
            'id' => $tenantId,
            'name' => 'Workflow Tenant Updated',
            'email' => 'workflow-tenant-updated@example.com',
            'subdomain' => 'workflow-tenant-updated',
        ]);
    }

    public function test_asset_module_create_and_update_workflow(): void
    {
        $tenant = Tenant::factory()->create();
        $user = $this->createUser($tenant);

        Sanctum::actingAs($user);

        $createResponse = $this->postJson('/api/assets', [
            'name' => 'Workflow Crusher',
            'status' => 'active',
            'description' => 'Initial commissioning',
        ]);

        $createResponse->assertCreated()
            ->assertJsonPath('name', 'Workflow Crusher')
            ->assertJsonPath('status', 'active');

        $assetId = $createResponse->json('id');

        $this->assertDatabaseHas('assets', [
            'id' => $assetId,
            'tenant_id' => $tenant->id,
            'name' => 'Workflow Crusher',
            'status' => 'active',
        ]);

        $updateResponse = $this->putJson('/api/assets/' . $assetId, [
            'status' => 'inactive',
            'description' => 'Awaiting spares',
        ]);

        $updateResponse->assertOk()
            ->assertJsonPath('id', $assetId)
            ->assertJsonPath('status', 'inactive')
            ->assertJsonPath('description', 'Awaiting spares');

        $this->assertDatabaseHas('assets', [
            'id' => $assetId,
            'status' => 'inactive',
            'description' => 'Awaiting spares',
        ]);
    }

    public function test_vehicle_module_create_and_update_workflow(): void
    {
        $tenant = Tenant::factory()->create();
        $user = $this->createUser($tenant);

        Sanctum::actingAs($user);

        $createResponse = $this->postJson('/api/vehicles', [
            'name' => 'Workflow Hauler',
            'license_plate' => 'WF-1001',
            'status' => 'active',
        ]);

        $createResponse->assertCreated()
            ->assertJsonPath('name', 'Workflow Hauler')
            ->assertJsonPath('license_plate', 'WF-1001');

        $vehicleId = $createResponse->json('id');

        $this->assertDatabaseHas('vehicles', [
            'id' => $vehicleId,
            'tenant_id' => $tenant->id,
            'name' => 'Workflow Hauler',
            'license_plate' => 'WF-1001',
        ]);

        $updateResponse = $this->putJson('/api/vehicles/' . $vehicleId, [
            'name' => 'Workflow Hauler Rev A',
            'status' => 'maintenance',
            'license_plate' => 'WF-1001A',
        ]);

        $updateResponse->assertOk()
            ->assertJsonPath('id', $vehicleId)
            ->assertJsonPath('name', 'Workflow Hauler Rev A')
            ->assertJsonPath('status', 'maintenance')
            ->assertJsonPath('license_plate', 'WF-1001A');

        $this->assertDatabaseHas('vehicles', [
            'id' => $vehicleId,
            'name' => 'Workflow Hauler Rev A',
            'status' => 'maintenance',
            'license_plate' => 'WF-1001A',
        ]);
    }

    public function test_work_order_module_create_and_update_workflow(): void
    {
        $tenant = Tenant::factory()->create();
        $user = $this->createUser($tenant);
        $asset = Asset::factory()->create([
            'tenant_id' => $tenant->id,
            'status' => 'active',
        ]);

        Sanctum::actingAs($user);

        $createResponse = $this->postJson('/api/work-orders', [
            'asset_id' => $asset->id,
            'type' => 'corrective',
            'status' => 'open',
            'priority' => 'high',
            'description' => 'Hydraulic leak at boom cylinder',
        ]);

        $createResponse->assertCreated()
            ->assertJsonPath('asset_id', $asset->id)
            ->assertJsonPath('status', 'open')
            ->assertJsonPath('priority', 'high');

        $workOrderId = $createResponse->json('id');

        $this->assertDatabaseHas('work_orders', [
            'id' => $workOrderId,
            'tenant_id' => $tenant->id,
            'asset_id' => $asset->id,
            'status' => 'open',
            'priority' => 'high',
        ]);

        $updateResponse = $this->putJson('/api/work-orders/' . $workOrderId, [
            'status' => 'completed',
            'resolution' => 'Replaced hose and pressure-tested circuit',
        ]);

        $updateResponse->assertOk()
            ->assertJsonPath('id', $workOrderId)
            ->assertJsonPath('status', 'completed')
            ->assertJsonPath('resolution', 'Replaced hose and pressure-tested circuit');

        $this->assertDatabaseHas('work_orders', [
            'id' => $workOrderId,
            'status' => 'completed',
            'resolution' => 'Replaced hose and pressure-tested circuit',
        ]);
    }

    private function createUser(?Tenant $tenant = null): User
    {
        static $counter = 1;

        $user = User::create([
            'name' => 'Workflow Tester ' . $counter,
            'email' => 'workflow-tester-' . $counter . '@example.com',
            'password' => Hash::make('password123'),
            'tenant_id' => $tenant?->id,
        ]);

        $counter++;

        return $user;
    }
}
