<?php

namespace Tests\Feature;

use App\Models\AgentNotification;
use App\Models\Tenant;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Tests\TestCase;

class PublicLaunchHardeningAuditTest extends TestCase
{
    use RefreshDatabase;

    public function test_public_ui_pages_links_and_images_are_resolvable(): void
    {
        $tenant = Tenant::factory()->create();
        $notification = AgentNotification::create([
            'tenant_id' => $tenant->id,
            'agent_key' => 'ops',
            'agent_name' => 'Ops Agent',
            'severity' => 'info',
            'title' => 'Launch Audit',
            'summary' => 'Audit notification',
            'payload' => [],
            'recipients' => ['ops'],
            'status' => 'sent',
            'sent_at' => now(),
        ]);

        $seedPages = [
            '/ui',
            '/ui/assets',
            '/ui/assets/1',
            '/ui/vehicles',
            '/ui/vehicles/1',
            '/ui/trips',
            '/ui/trips/1',
            '/ui/maintenance',
            '/ui/work-orders',
            '/ui/work-orders/1',
            '/ui/pm-schedules',
            '/ui/pm-schedules/1',
            '/ui/checklists',
            '/ui/checklists/1',
            '/ui/downtime',
            '/ui/downtime/1',
            '/ui/fuel',
            '/ui/fuel/1',
            '/ui/tyres',
            '/ui/tyres/1',
            '/ui/telemetry',
            '/ui/telemetry/1',
            '/ui/sites',
            '/ui/sites/1',
            '/ui/inventory',
            '/ui/inventory/1',
            '/ui/vendors',
            '/ui/vendors/1',
            '/ui/purchase-requests',
            '/ui/purchase-requests/1',
            '/ui/purchase-orders',
            '/ui/purchase-orders/1',
            '/ui/payments',
            '/ui/payments/1',
            '/ui/imports',
            '/ui/imports/1',
            '/ui/devices',
            '/ui/devices/1',
            '/ui/alerts',
            '/ui/alerts/1',
            '/ui/alarm-rules',
            '/ui/alarm-rules/1',
            '/ui/kpis',
            '/ui/kpis/1',
            '/ui/ai',
            '/ui/ai/1',
            '/ui/agent-notifications',
            '/ui/agent-notifications/' . $notification->id,
            '/ui/reports',
            '/ui/reports/1',
            '/ui/tenants',
            '/ui/tenants/1',
            '/ui/users',
            '/ui/users/1',
            '/ui/roles',
            '/ui/roles/1',
            '/ui/notifications',
            '/ui/notifications/1',
            '/ui/audit',
            '/ui/audit/1',
            '/ui/reconciliation',
            '/ui/settings',
        ];

        $visitedPages = [];
        $visitedAssets = [];
        $pagesToCheck = $seedPages;

        while ($pagesToCheck) {
            $path = array_shift($pagesToCheck);
            if (isset($visitedPages[$path])) {
                continue;
            }
            $visitedPages[$path] = true;

            $response = $this->get($path);
            $status = $response->getStatusCode();
            $this->assertNotSame(404, $status, "Broken page link [$path] returned 404.");
            $this->assertLessThan(500, $status, "Page [$path] returned 5xx.");

            $html = $response->getContent();
            if (!is_string($html) || $html === '') {
                continue;
            }

            [$links, $images] = $this->extractInternalReferences($html);

            foreach ($links as $link) {
                if (str_starts_with($link, '/api/')) {
                    continue;
                }

                $linkResponse = $this->get($link);
                $linkStatus = $linkResponse->getStatusCode();
                $this->assertNotSame(404, $linkStatus, "Broken link [$link] discovered on [$path].");
                $this->assertLessThan(500, $linkStatus, "Link [$link] discovered on [$path] returned 5xx.");

                if (str_starts_with($link, '/ui')) {
                    $pagesToCheck[] = $link;
                }
            }

            foreach ($images as $src) {
                if (isset($visitedAssets[$src])) {
                    continue;
                }
                $visitedAssets[$src] = true;

                if (str_starts_with($src, '/ui-assets/') || str_starts_with($src, '/build/')) {
                    $this->assertFileExists(
                        public_path(ltrim($src, '/')),
                        "Broken image/asset file [$src] discovered on [$path]."
                    );
                    continue;
                }

                $assetResponse = $this->get($src);
                $assetStatus = $assetResponse->getStatusCode();
                $this->assertNotSame(404, $assetStatus, "Broken image/asset [$src] discovered on [$path].");
                $this->assertLessThan(400, $assetStatus, "Image/asset [$src] discovered on [$path] is not directly accessible.");
            }
        }
    }

    private function extractInternalReferences(string $html): array
    {
        preg_match_all('/href="([^"]+)"/', $html, $hrefMatches);
        preg_match_all('/src="([^"]+)"/', $html, $srcMatches);

        $links = [];
        foreach (($hrefMatches[1] ?? []) as $href) {
            $value = trim((string) $href);
            if ($value === '' || str_contains($value, '{{') || str_starts_with($value, '#') || str_starts_with($value, 'http')) {
                continue;
            }
            if (!str_starts_with($value, '/')) {
                continue;
            }
            $links[] = strtok($value, '?') ?: $value;
        }

        $images = [];
        foreach (($srcMatches[1] ?? []) as $src) {
            $value = trim((string) $src);
            if ($value === '' || str_contains($value, '{{') || str_starts_with($value, 'http')) {
                continue;
            }
            if (!str_starts_with($value, '/')) {
                continue;
            }
            $images[] = strtok($value, '?') ?: $value;
        }

        return [array_values(array_unique($links)), array_values(array_unique($images))];
    }
}
