<?php

namespace Tests\Feature;

use App\Models\AgentNotification;
use App\Models\Tenant;
use App\Models\User;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Facades\Hash;
use Laravel\Sanctum\Sanctum;
use Tests\TestCase;

class SmokeCoverageTest extends TestCase
{
    use RefreshDatabase;

    public function test_public_ui_routes_do_not_error(): void
    {
        $tenant = Tenant::factory()->create();
        $notification = AgentNotification::create([
            'tenant_id' => $tenant->id,
            'agent_key' => 'ops',
            'agent_name' => 'Ops Agent',
            'severity' => 'info',
            'title' => 'Smoke Test',
            'summary' => 'Smoke test notification',
            'payload' => [],
            'recipients' => ['ops'],
            'status' => 'sent',
            'sent_at' => now(),
        ]);

        $paths = [
            '/ui',
            '/ui/assets',
            '/ui/assets/1',
            '/ui/vehicles',
            '/ui/vehicles/1',
            '/ui/trips',
            '/ui/trips/1',
            '/ui/maintenance',
            '/ui/work-orders',
            '/ui/work-orders/1',
            '/ui/pm-schedules',
            '/ui/pm-schedules/1',
            '/ui/checklists',
            '/ui/checklists/1',
            '/ui/downtime',
            '/ui/downtime/1',
            '/ui/fuel',
            '/ui/fuel/1',
            '/ui/tyres',
            '/ui/tyres/1',
            '/ui/telemetry',
            '/ui/telemetry/1',
            '/ui/sites',
            '/ui/sites/1',
            '/ui/inventory',
            '/ui/inventory/1',
            '/ui/vendors',
            '/ui/vendors/1',
            '/ui/purchase-requests',
            '/ui/purchase-requests/1',
            '/ui/purchase-orders',
            '/ui/purchase-orders/1',
            '/ui/payments',
            '/ui/payments/1',
            '/ui/imports',
            '/ui/imports/1',
            '/ui/devices',
            '/ui/devices/1',
            '/ui/alerts',
            '/ui/alerts/1',
            '/ui/alarm-rules',
            '/ui/alarm-rules/1',
            '/ui/kpis',
            '/ui/kpis/1',
            '/ui/ai',
            '/ui/ai/1',
            '/ui/agent-notifications',
            '/ui/agent-notifications/' . $notification->id,
            '/ui/reports',
            '/ui/reports/1',
            '/ui/tenants',
            '/ui/tenants/1',
            '/ui/users',
            '/ui/users/1',
            '/ui/roles',
            '/ui/roles/1',
            '/ui/notifications',
            '/ui/notifications/1',
            '/ui/audit',
            '/ui/audit/1',
            '/ui/reconciliation',
            '/ui/settings',
        ];

        foreach ($paths as $path) {
            $response = $this->get($path);
            $this->assertLessThan(
                500,
                $response->getStatusCode(),
                "Expected non-5xx response for [{$path}], got [{$response->getStatusCode()}]"
            );
        }
    }

    public function test_authenticated_api_get_routes_do_not_error(): void
    {
        $tenant = Tenant::factory()->create();
        $user = User::create([
            'name' => 'API Tester',
            'email' => 'api-smoke@example.com',
            'password' => Hash::make('password123'),
            'tenant_id' => $tenant->id,
        ]);

        Sanctum::actingAs($user);

        $response = $this->postJson('/api/auth/login', [
            'email' => $user->email,
            'password' => 'password123',
        ]);
        $this->assertLessThan(500, $response->getStatusCode(), 'Login endpoint returned 5xx');

        $response = $this->postJson('/api/auth/register', [
            'name' => 'API Register',
            'email' => 'api-register@example.com',
            'password' => 'password123',
            'password_confirmation' => 'password123',
            'tenant_id' => $tenant->id,
        ]);
        $this->assertLessThan(500, $response->getStatusCode(), 'Register endpoint returned 5xx');

        $paths = [
            '/api/tenants',
            '/api/sites',
            '/api/assets',
            '/api/vehicles',
            '/api/telemetry',
            '/api/work-orders',
            '/api/pm-schedules',
            '/api/checklists',
            '/api/downtime-reasons',
            '/api/fuel-logs',
            '/api/tyres',
            '/api/parts',
            '/api/vendors',
            '/api/inventory-locations',
            '/api/inventory-items',
            '/api/imports',
            '/api/devices',
            '/api/alarms',
            '/api/alarm-rules',
            '/api/notifications/channels',
            '/api/notifications/routes',
            '/api/notifications/logs',
            '/api/roles',
            '/api/ai/insights',
            '/api/ai/reports',
            '/api/audit-logs',
        ];

        foreach ($paths as $path) {
            $response = $this->getJson($path);
            $this->assertLessThan(
                500,
                $response->getStatusCode(),
                "Expected non-5xx response for [{$path}], got [{$response->getStatusCode()}]"
            );
        }
    }
}
