<?php

namespace Tests\Feature;

use App\Domain\Telegram\TelegramApiClient;
use App\Domain\Telegram\TelegramNotifier;
use App\Jobs\ProcessNotificationOutbox;
use App\Models\Asset;
use App\Models\NotificationOutbox;
use App\Models\PurchaseRequest;
use App\Models\PurchaseRequestApproval;
use App\Models\Role;
use App\Models\TelegramGroup;
use App\Models\TelegramLink;
use App\Models\TelegramLinkCode;
use App\Models\Tenant;
use App\Models\User;
use App\Models\WorkOrder;
use App\Services\TelegramReminderService;
use App\Services\TelegramEventEscalationService;
use App\Services\TelegramEventService;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\Hash;
use Mockery;
use Tests\TestCase;

class TelegramWorkflowTest extends TestCase
{
    use RefreshDatabase;

    protected function tearDown(): void
    {
        Mockery::close();
        parent::tearDown();
    }

    public function test_telegram_link_endpoint_validates_and_links_account(): void
    {
        $tenant = Tenant::factory()->create();
        $user = $this->createUser($tenant);

        $this->postJson('/api/telegram/link', [])
            ->assertStatus(422)
            ->assertJsonValidationErrors(['code', 'telegram_user_id']);

        $this->postJson('/api/telegram/link', [
            'code' => 'NOPE',
            'telegram_user_id' => 'tg-100',
        ])->assertStatus(422)->assertJsonPath('message', 'Invalid or expired link code.');

        TelegramLinkCode::create([
            'tenant_id' => $tenant->id,
            'user_id' => $user->id,
            'code' => 'EXPIRED1',
            'expires_at' => now()->subMinute(),
        ]);

        $this->postJson('/api/telegram/link', [
            'code' => 'EXPIRED1',
            'telegram_user_id' => 'tg-100',
        ])->assertStatus(422)->assertJsonPath('message', 'Link code has expired.');

        TelegramLinkCode::create([
            'tenant_id' => $tenant->id,
            'user_id' => $user->id,
            'code' => 'VALID123',
            'expires_at' => now()->addMinutes(10),
        ]);

        $this->postJson('/api/telegram/link', [
            'code' => 'VALID123',
            'telegram_user_id' => 'tg-100',
            'chat_id' => 'chat-100',
        ])->assertOk()->assertJsonPath('message', 'Linked successfully. You can now use Telegram commands.');

        $this->assertDatabaseHas('telegram_links', [
            'tenant_id' => $tenant->id,
            'user_id' => $user->id,
            'telegram_user_id' => 'tg-100',
            'telegram_chat_id' => 'chat-100',
        ]);
        $this->assertDatabaseHas('telegram_link_codes', [
            'code' => 'VALID123',
        ]);
        $this->assertNotNull(TelegramLinkCode::where('code', 'VALID123')->value('used_at'));
        $this->assertDatabaseHas('audit_logs', [
            'tenant_id' => $tenant->id,
            'user_id' => $user->id,
            'channel' => 'telegram',
            'action' => 'telegram.link',
        ]);
    }

    public function test_telegram_webhook_secret_and_command_flow(): void
    {
        config([
            'services.telegram.webhook_secret' => 'secret-test',
        ]);

        $payload = [
            'update_id' => 2001,
            'message' => [
                'message_id' => 1,
                'chat' => ['id' => 'chat-1', 'type' => 'private'],
                'from' => ['id' => 'tg-1'],
                'text' => '/help',
            ],
        ];

        $this->postJson('/api/telegram/webhook', $payload)->assertForbidden();
        $this->postJson('/api/telegram/webhook', $payload, [
            'X-Telegram-Bot-Api-Secret-Token' => 'bad',
        ])->assertForbidden();

        $client = Mockery::mock(TelegramApiClient::class);
        $client->shouldReceive('sendMessage')
            ->once()
            ->withArgs(function (string $chatId, string $text) {
                return $chatId === 'chat-1' && str_contains($text, 'Not linked');
            })
            ->andReturn(['message_id' => 'm-1']);
        $this->app->instance(TelegramApiClient::class, $client);

        $this->postJson('/api/telegram/webhook', $payload, [
            'X-Telegram-Bot-Api-Secret-Token' => 'secret-test',
        ])->assertOk()->assertJsonPath('ok', true);

        $this->assertDatabaseHas('audit_logs', [
            'channel' => 'telegram',
            'action' => 'telegram.update',
        ]);
    }

    public function test_telegram_webhook_handles_linked_help_and_callback_ack(): void
    {
        config([
            'services.telegram.webhook_secret' => 'secret-test',
        ]);

        $tenant = Tenant::factory()->create();
        $user = $this->createUser($tenant);
        TelegramLink::create([
            'tenant_id' => $tenant->id,
            'user_id' => $user->id,
            'telegram_user_id' => 'tg-2',
            'telegram_chat_id' => 'chat-2',
            'linked_at' => now(),
        ]);

        $client = Mockery::mock(TelegramApiClient::class);
        $client->shouldReceive('sendMessage')
            ->once()
            ->withArgs(function (string $chatId, string $text) {
                return $chatId === 'chat-2' && str_contains($text, 'Commands:');
            })
            ->andReturn(['message_id' => 'm-2']);
        $client->shouldReceive('answerCallbackQuery')
            ->once()
            ->with('cb-1', 'Done')
            ->andReturn(['ok' => true]);
        $this->app->instance(TelegramApiClient::class, $client);

        $messagePayload = [
            'update_id' => 2101,
            'message' => [
                'message_id' => 2,
                'chat' => ['id' => 'chat-2', 'type' => 'private'],
                'from' => ['id' => 'tg-2'],
                'text' => '/help',
            ],
        ];

        $this->postJson('/api/telegram/webhook', $messagePayload, [
            'X-Telegram-Bot-Api-Secret-Token' => 'secret-test',
        ])->assertOk()->assertJsonPath('ok', true);

        $callbackPayload = [
            'update_id' => 2102,
            'callback_query' => [
                'id' => 'cb-1',
                'from' => ['id' => 'tg-2'],
                'data' => 'ACCEPT WO-0001',
                'message' => [
                    'message_id' => 3,
                    'chat' => ['id' => 'chat-2', 'type' => 'private'],
                ],
            ],
        ];

        $this->postJson('/api/telegram/webhook', $callbackPayload, [
            'X-Telegram-Bot-Api-Secret-Token' => 'secret-test',
        ])->assertOk()->assertJsonPath('ok', true);
    }

    public function test_notification_outbox_dispatches_and_tracks_failures(): void
    {
        $tenant = Tenant::factory()->create();
        $user = $this->createUser($tenant);

        TelegramLink::create([
            'tenant_id' => $tenant->id,
            'user_id' => $user->id,
            'telegram_user_id' => 'tg-outbox',
            'telegram_chat_id' => 'chat-user',
            'linked_at' => now(),
        ]);

        TelegramGroup::create([
            'tenant_id' => $tenant->id,
            'name' => 'FW-Quarry-Maint',
            'type' => 'maintenance',
            'chat_id' => 'chat-group',
            'enabled' => true,
        ]);

        $okItem = NotificationOutbox::create([
            'tenant_id' => $tenant->id,
            'channel' => 'telegram',
            'message' => 'Outbox success',
            'recipients' => [$user->id, 'maintenance'],
            'status' => 'queued',
        ]);

        $failItem = NotificationOutbox::create([
            'tenant_id' => $tenant->id,
            'channel' => 'telegram',
            'message' => 'Outbox fail',
            'recipients' => [$user->id],
            'status' => 'queued',
        ]);

        $notifier = Mockery::mock(TelegramNotifier::class);
        $notifier->shouldReceive('notifyUser')->once()->with($user->id, 'Outbox success')->andReturn(true);
        $notifier->shouldReceive('notifyChat')->once()->with('chat-group', 'Outbox success')->andReturn('msg-1');
        $notifier->shouldReceive('notifyUser')->once()->with($user->id, 'Outbox fail')->andThrow(new \RuntimeException('send failed'));

        $job = new ProcessNotificationOutbox();
        $job->handle($notifier);

        $this->assertSame('sent', $okItem->fresh()->status);
        $this->assertNotNull($okItem->fresh()->sent_at);

        $this->assertSame('failed', $failItem->fresh()->status);
        $this->assertNotNull($failItem->fresh()->failed_at);
        $this->assertStringContainsString('send failed', (string) $failItem->fresh()->error_message);
    }

    public function test_telegram_reminder_service_sends_pr_and_work_order_notifications(): void
    {
        Carbon::setTestNow(Carbon::create(2026, 2, 4, 10, 0, 0, 'UTC'));

        $tenant = Tenant::factory()->create();
        $requester = $this->createUser($tenant, 'requester@example.com', 'Requester');
        $approver = $this->createUser($tenant, 'approver@example.com', 'Approver');
        $engineering = $this->createUser($tenant, 'engineer@example.com', 'Engineering');

        $role = Role::create([
            'tenant_id' => $tenant->id,
            'name' => 'Engineering Manager',
        ]);
        $engineering->roles()->attach($role->id);

        foreach ([$requester, $approver, $engineering] as $index => $linkedUser) {
            TelegramLink::create([
                'tenant_id' => $tenant->id,
                'user_id' => $linkedUser->id,
                'telegram_user_id' => 'tg-rem-' . $index,
                'telegram_chat_id' => 'chat-rem-' . $index,
                'linked_at' => now(),
            ]);
        }

        $request = PurchaseRequest::create([
            'tenant_id' => $tenant->id,
            'request_code' => 'PR-1001',
            'status' => 'submitted',
            'requested_by' => $requester->id,
            'last_reminded_at' => now()->subHours(10),
        ]);

        PurchaseRequestApproval::create([
            'tenant_id' => $tenant->id,
            'purchase_request_id' => $request->id,
            'step' => 1,
            'approver_id' => $approver->id,
            'status' => 'pending',
        ]);

        $asset = Asset::factory()->create(['tenant_id' => $tenant->id]);
        WorkOrder::create([
            'tenant_id' => $tenant->id,
            'asset_id' => $asset->id,
            'reference_code' => 'WO-1001',
            'status' => 'pending_approval',
            'approval_status' => 'pending',
            'reported_by' => $requester->id,
            'last_reminded_at' => now()->subHours(10),
        ]);
        WorkOrder::create([
            'tenant_id' => $tenant->id,
            'asset_id' => $asset->id,
            'reference_code' => 'WO-1002',
            'status' => 'open',
            'assigned_to' => $approver->id,
            'reported_by' => $requester->id,
            'due_at' => now()->subHours(30),
            'last_reminded_at' => now()->subHours(10),
        ]);

        $calls = [];
        $notifier = Mockery::mock(TelegramNotifier::class);
        $notifier->shouldReceive('notifyUser')
            ->andReturnUsing(function (int $userId, string $message, ?array $markup = null) use (&$calls) {
                $calls[] = compact('userId', 'message', 'markup');
                return true;
            });

        $service = new TelegramReminderService($notifier);
        $service->run();

        $this->assertGreaterThanOrEqual(5, count($calls));
        $messages = implode("\n", array_column($calls, 'message'));

        $this->assertStringContainsString('PR PR-1001 awaiting', $messages);
        $this->assertStringContainsString('WO WO-1001 awaiting Engineering approval', $messages);
        $this->assertStringContainsString('Escalation: WO WO-1002 is overdue', $messages);

        $recipientIds = array_column($calls, 'userId');
        $this->assertContains($requester->id, $recipientIds);
        $this->assertContains($approver->id, $recipientIds);
        $this->assertContains($engineering->id, $recipientIds);

        $this->assertNotNull($request->fresh()->last_reminded_at);
        $this->assertNotNull(WorkOrder::where('reference_code', 'WO-1001')->first()->last_reminded_at);
        $this->assertNotNull(WorkOrder::where('reference_code', 'WO-1002')->first()->last_reminded_at);

        Carbon::setTestNow();
    }

    public function test_telegram_event_service_dispatches_and_deduplicates(): void
    {
        $tenant = Tenant::factory()->create();
        $asset = Asset::factory()->create(['tenant_id' => $tenant->id]);
        TelegramGroup::create([
            'tenant_id' => $tenant->id,
            'name' => 'FW-Quarry-Maint',
            'type' => 'maintenance',
            'chat_id' => 'chat-maint',
            'enabled' => true,
        ]);

        $notifier = Mockery::mock(TelegramNotifier::class);
        $notifier->shouldReceive('notifyChat')->once()->with('chat-maint', Mockery::type('string'))->andReturn('evt-msg-1');
        $notifier->shouldReceive('notifyUser')->never();
        $this->app->instance(TelegramNotifier::class, $notifier);

        $service = $this->app->make(TelegramEventService::class);
        $event = $service->createEvent($tenant->id, 'breakdown', 'alarm', [
            'asset_id' => $asset->id,
            'details' => ['Reply: ACK <event_code>'],
        ], [
            'title' => 'BREAKDOWN',
            'dedupe_key' => 'breakdown|asset:' . $asset->id,
        ]);

        $this->assertNotNull($event->id);
        $this->assertSame('open', $event->status);
        $this->assertDatabaseHas('telegram_event_messages', [
            'tenant_id' => $tenant->id,
            'telegram_event_id' => $event->id,
            'chat_id' => 'chat-maint',
            'message_id' => 'evt-msg-1',
        ]);

        $duplicate = $service->createEvent($tenant->id, 'breakdown', 'alarm', [
            'asset_id' => $asset->id,
        ], [
            'dedupe_key' => 'breakdown|asset:' . $asset->id,
        ]);

        $this->assertTrue($duplicate->is($event));
        $this->assertSame(1, \App\Models\TelegramEvent::where('tenant_id', $tenant->id)->count());
    }

    public function test_telegram_event_escalation_dispatches_when_threshold_met(): void
    {
        Carbon::setTestNow(Carbon::create(2026, 2, 4, 12, 0, 0, 'UTC'));

        $tenant = Tenant::factory()->create();
        TelegramGroup::create([
            'tenant_id' => $tenant->id,
            'name' => 'FW-Management',
            'type' => 'management',
            'chat_id' => 'chat-mgmt',
            'enabled' => true,
        ]);

        $event = \App\Models\TelegramEvent::create([
            'tenant_id' => $tenant->id,
            'event_code' => 'EVT-00001',
            'event_type' => 'breakdown',
            'severity' => 'alarm',
            'title' => 'BREAKDOWN',
            'message' => 'Escalate me',
            'status' => 'open',
        ]);
        \Illuminate\Support\Facades\DB::table('telegram_events')
            ->where('id', $event->id)
            ->update([
                'created_at' => now()->subMinutes(30),
                'updated_at' => now()->subMinutes(30),
            ]);

        $notifier = Mockery::mock(TelegramNotifier::class);
        $notifier->shouldReceive('notifyChat')->once()->with('chat-mgmt', 'Escalate me')->andReturn('evt-msg-2');
        $notifier->shouldReceive('notifyUser')->never();
        $this->app->instance(TelegramNotifier::class, $notifier);

        $service = $this->app->make(TelegramEventEscalationService::class);
        $service->run();

        $fresh = $event->fresh();
        $this->assertSame(1, (int) $fresh->escalation_level);
        $this->assertNotNull($fresh->last_escalated_at);
        $this->assertDatabaseHas('telegram_event_messages', [
            'tenant_id' => $tenant->id,
            'telegram_event_id' => $event->id,
            'chat_id' => 'chat-mgmt',
            'message_id' => 'evt-msg-2',
        ]);

        Carbon::setTestNow();
    }

    private function createUser(Tenant $tenant, ?string $email = null, ?string $name = null): User
    {
        static $counter = 1;

        $user = User::create([
            'name' => $name ?: ('Telegram Tester ' . $counter),
            'email' => $email ?: ('telegram-tester-' . $counter . '@example.com'),
            'password' => Hash::make('password123'),
            'tenant_id' => $tenant->id,
        ]);

        $counter++;

        return $user;
    }
}
