const { test, expect } = require('@playwright/test');

const sidebarLinks = [
    '/ui',
    '/ui/assets',
    '/ui/vehicles',
    '/ui/work-orders',
    '/ui/fuel',
    '/ui/inventory',
    '/ui/imports',
    '/ui/reports',
    '/ui/notifications',
    '/ui/settings',
];

async function ensureSidebarVisible(page) {
    const sidebar = page.locator('.sidebar');
    if (await sidebar.isVisible()) {
        return;
    }

    await page.locator('[data-ui-toggle="sidebar"]').click();
    await expect(sidebar).toBeVisible();
}

async function expandSidebarGroups(page) {
    const toggles = await page.$$('.nav-group.is-collapsed [data-nav-toggle]');
    for (const toggle of toggles) {
        await toggle.click({ force: true });
    }
}

async function gotoUi(page, path = '/ui') {
    await page.goto(path, { waitUntil: 'domcontentloaded', timeout: 30_000 });
}

test('sidebar links click-through across modules', async ({ page }, testInfo) => {
    test.setTimeout(120_000);
    test.skip(testInfo.project.name.includes('mobile'), 'Sidebar click-through is validated on desktop where sidebar is always interactive.');

    for (const href of sidebarLinks) {
        await gotoUi(page, '/ui');
        await ensureSidebarVisible(page);
        await expandSidebarGroups(page);

        const link = page.locator(`.sidebar .nav-item[href="${href}"]`).first();
        await link.evaluate((el) => el.scrollIntoView({ block: 'center' }));
        await expect(link).toBeVisible();
        await link.click({ force: true });
        await expect(page).toHaveURL(new RegExp(`${href.replace(/[.*+?^${}()|[\]\\]/g, '\\$&')}$`));
        await expect(page.locator('main.content')).toBeVisible();
    }
});

test('topbar links, quick actions, and modal flows work', async ({ page }, testInfo) => {
    await gotoUi(page, '/ui');

    await page.locator('.top-actions .btn', { hasText: 'Quick Add' }).click({ force: true });
    await expect(page.locator('[data-ui="action-modal"]')).toHaveAttribute('aria-hidden', 'false');
    await expect(page.locator('[data-action-title]')).toContainText('Quick Add');
    await page.locator('[data-ui="action-modal"]').click({ position: { x: 8, y: 8 }, force: true });
    await expect(page.locator('body')).not.toHaveClass(/action-open/);

    await page.locator('.top-actions .btn', { hasText: 'Create Work Order' }).click({ force: true });
    await expect(page.locator('[data-action-title]')).toContainText('Create Work Order');
    await page.locator('[data-ui="action-modal"]').click({ position: { x: 8, y: 8 }, force: true });

    if (!testInfo.project.name.includes('mobile')) {
        await ensureSidebarVisible(page);
        await expandSidebarGroups(page);
        await page.locator('.support-card .btn', { hasText: 'Open support' }).click({ force: true });
        await expect(page.locator('[data-action-title]')).toContainText('Open Support');
        await page.locator('[data-ui="action-modal"]').click({ position: { x: 8, y: 8 }, force: true });
    }

    if (!testInfo.project.name.includes('mobile')) {
        await page.locator('.top-actions a[href="/ui/notifications"]').click();
        await expect(page).toHaveURL(/\/ui\/notifications$/);
    }
});

test.skip('search shortcut and enter-navigation work', async () => {});

test('chatbot open, close, and quick navigation chip works', async ({ page }) => {
    await gotoUi(page, '/ui');

    await page.locator('[data-chatbot-open]').click({ force: true });
    await expect(page.locator('body')).toHaveClass(/chatbot-open/);
    await expect(page.locator('[data-chatbot]')).toHaveAttribute('aria-hidden', 'false');

    await page.locator('[data-chatbot-quick="open assets"]').click({ force: true });
    await expect(page).toHaveURL(/\/ui\/assets$/);

    await gotoUi(page, '/ui');
    await page.locator('[data-chatbot-open]').click({ force: true });
    await page.locator('[data-chatbot]').click({ position: { x: 8, y: 8 }, force: true });
    await expect(page.locator('body')).not.toHaveClass(/chatbot-open/);
    await expect(page.locator('[data-chatbot]')).toHaveAttribute('aria-hidden', 'true');
});

test('detail links are clickable from list pages', async ({ page }) => {
    const cases = [
        { list: '/ui/assets', detailPattern: /\/ui\/assets\/\d+$/ },
        { list: '/ui/vehicles', detailPattern: /\/ui\/vehicles\/\d+$/ },
        { list: '/ui/work-orders', detailPattern: /\/ui\/work-orders\/\d+$/ },
        { list: '/ui/imports', detailPattern: /\/ui\/imports\/[A-Z]+-\d+$/ },
        { list: '/ui/reports', detailPattern: /\/ui\/reports\/[^/]+$/ },
    ];

    for (const item of cases) {
        await gotoUi(page, item.list);
        const firstDetailLink = page.locator(`a[href^="${item.list}/"]`).first();
        await expect(firstDetailLink).toBeVisible();
        await firstDetailLink.click({ force: true });
        await expect(page).toHaveURL(item.detailPattern);
        await expect(page.locator('main.content')).toBeVisible();
    }
});
